package io.k8s.api.storage.v1beta1

import dev.hnaderi.k8s._

/** CSIStorageCapacity stores the result of one CSI GetCapacity call. For a given StorageClass, this describes the available capacity in a particular topology segment.  This can be used when considering where to instantiate new PersistentVolumes.

For example this can express things like: - StorageClass "standard" has "1234 GiB" available in "topology.kubernetes.io/zone=us-east1" - StorageClass "localssd" has "10 GiB" available in "kubernetes.io/hostname=knode-abc123"

The following three cases all imply that no capacity is available for a certain combination: - no object exists with suitable topology and storage class name - such an object exists, but the capacity is unset - such an object exists, but the capacity is zero

The producer of these objects can decide which approach is more suitable.

They are consumed by the kube-scheduler when a CSI driver opts into capacity-aware scheduling with CSIDriverSpec.StorageCapacity. The scheduler compares the MaximumVolumeSize against the requested size of pending volumes to filter out unsuitable nodes. If MaximumVolumeSize is unset, it falls back to a comparison against the less precise Capacity. If that is also unset, the scheduler assumes that capacity is insufficient and tries some other node. */
final case class CSIStorageCapacity(
  storageClassName : String,
  maximumVolumeSize : Option[io.k8s.apimachinery.pkg.api.resource.Quantity] = None,
  nodeTopology : Option[io.k8s.apimachinery.pkg.apis.meta.v1.LabelSelector] = None,
  metadata : Option[io.k8s.apimachinery.pkg.apis.meta.v1.ObjectMeta] = None,
  capacity : Option[io.k8s.apimachinery.pkg.api.resource.Quantity] = None
) extends KObject {
  protected val _resourceKind = ResourceKind("storage.k8s.io", "CSIStorageCapacity", "v1beta1")

  def withStorageClassName(value: String) : CSIStorageCapacity = copy(storageClassName = value)
  def withMaximumVolumeSize(value: io.k8s.apimachinery.pkg.api.resource.Quantity) : CSIStorageCapacity = copy(maximumVolumeSize = Some(value))
  def withNodeTopology(value: io.k8s.apimachinery.pkg.apis.meta.v1.LabelSelector) : CSIStorageCapacity = copy(nodeTopology = Some(value))
  def withMetadata(value: io.k8s.apimachinery.pkg.apis.meta.v1.ObjectMeta) : CSIStorageCapacity = copy(metadata = Some(value))
  def withCapacity(value: io.k8s.apimachinery.pkg.api.resource.Quantity) : CSIStorageCapacity = copy(capacity = Some(value))
}

