package io.k8s.api.flowcontrol.v1

import dev.hnaderi.k8s.utils._

/** LimitedPriorityLevelConfiguration specifies how to handle requests that are subject to limits. It addresses two issues:
  - How are requests for this priority level limited?
  - What should be done with requests that exceed the limit? */
final case class LimitedPriorityLevelConfiguration(
  borrowingLimitPercent : Option[Int] = None,
  lendablePercent : Option[Int] = None,
  limitResponse : Option[io.k8s.api.flowcontrol.v1.LimitResponse] = None,
  nominalConcurrencyShares : Option[Int] = None
) {

  /** Returns a new data with borrowingLimitPercent set to new value */
  def withBorrowingLimitPercent(value: Int) : LimitedPriorityLevelConfiguration = copy(borrowingLimitPercent = Some(value))
  /** if borrowingLimitPercent has a value, transforms to the result of function*/
  def mapBorrowingLimitPercent(f: Int => Int) : LimitedPriorityLevelConfiguration = copy(borrowingLimitPercent = borrowingLimitPercent.map(f))

  /** Returns a new data with lendablePercent set to new value */
  def withLendablePercent(value: Int) : LimitedPriorityLevelConfiguration = copy(lendablePercent = Some(value))
  /** if lendablePercent has a value, transforms to the result of function*/
  def mapLendablePercent(f: Int => Int) : LimitedPriorityLevelConfiguration = copy(lendablePercent = lendablePercent.map(f))

  /** Returns a new data with limitResponse set to new value */
  def withLimitResponse(value: io.k8s.api.flowcontrol.v1.LimitResponse) : LimitedPriorityLevelConfiguration = copy(limitResponse = Some(value))
  /** if limitResponse has a value, transforms to the result of function*/
  def mapLimitResponse(f: io.k8s.api.flowcontrol.v1.LimitResponse => io.k8s.api.flowcontrol.v1.LimitResponse) : LimitedPriorityLevelConfiguration = copy(limitResponse = limitResponse.map(f))

  /** Returns a new data with nominalConcurrencyShares set to new value */
  def withNominalConcurrencyShares(value: Int) : LimitedPriorityLevelConfiguration = copy(nominalConcurrencyShares = Some(value))
  /** if nominalConcurrencyShares has a value, transforms to the result of function*/
  def mapNominalConcurrencyShares(f: Int => Int) : LimitedPriorityLevelConfiguration = copy(nominalConcurrencyShares = nominalConcurrencyShares.map(f))
}

object LimitedPriorityLevelConfiguration {

    implicit val encoder : Encoder[io.k8s.api.flowcontrol.v1.LimitedPriorityLevelConfiguration] = new Encoder[io.k8s.api.flowcontrol.v1.LimitedPriorityLevelConfiguration] {
        def apply[T : Builder](o: io.k8s.api.flowcontrol.v1.LimitedPriorityLevelConfiguration) : T = {
          val obj = ObjectWriter[T]()
          obj
            .write("borrowingLimitPercent", o.borrowingLimitPercent)
            .write("lendablePercent", o.lendablePercent)
            .write("limitResponse", o.limitResponse)
            .write("nominalConcurrencyShares", o.nominalConcurrencyShares)
            .build
        }
    }

    implicit val decoder: Decoder[LimitedPriorityLevelConfiguration] = new Decoder[LimitedPriorityLevelConfiguration] {
      def apply[T : Reader](t: T): Either[String, LimitedPriorityLevelConfiguration] = for {
          obj <- ObjectReader(t)
          borrowingLimitPercent <- obj.readOpt[Int]("borrowingLimitPercent")
          lendablePercent <- obj.readOpt[Int]("lendablePercent")
          limitResponse <- obj.readOpt[io.k8s.api.flowcontrol.v1.LimitResponse]("limitResponse")
          nominalConcurrencyShares <- obj.readOpt[Int]("nominalConcurrencyShares")
      } yield LimitedPriorityLevelConfiguration (
          borrowingLimitPercent = borrowingLimitPercent,
          lendablePercent = lendablePercent,
          limitResponse = limitResponse,
          nominalConcurrencyShares = nominalConcurrencyShares
        )
    }
}

