package io.k8s.api.core.v1

import dev.hnaderi.k8s.utils._

/** ResourceRequirements describes the compute resource requirements. */
final case class ResourceRequirements(
  claims : Option[Seq[io.k8s.api.core.v1.ResourceClaim]] = None,
  limits : Option[Map[String, io.k8s.apimachinery.pkg.api.resource.Quantity]] = None,
  requests : Option[Map[String, io.k8s.apimachinery.pkg.api.resource.Quantity]] = None
) {

  /** Returns a new data with claims set to new value */
  def withClaims(value: Seq[io.k8s.api.core.v1.ResourceClaim]) : ResourceRequirements = copy(claims = Some(value))
  /** Appends new values to claims */
  def addClaims(newValues: io.k8s.api.core.v1.ResourceClaim*) : ResourceRequirements = copy(claims = Some(claims.fold(newValues)(_ ++ newValues)))
  /** if claims has a value, transforms to the result of function*/
  def mapClaims(f: Seq[io.k8s.api.core.v1.ResourceClaim] => Seq[io.k8s.api.core.v1.ResourceClaim]) : ResourceRequirements = copy(claims = claims.map(f))

  /** Returns a new data with limits set to new value */
  def withLimits(value: Map[String, io.k8s.apimachinery.pkg.api.resource.Quantity]) : ResourceRequirements = copy(limits = Some(value))
  /** Adds new values to limits */
  def addLimits(newValues: (String, io.k8s.apimachinery.pkg.api.resource.Quantity)*) : ResourceRequirements = copy(limits = Some(limits.fold(newValues.toMap)(_ ++ newValues)))
  /** if limits has a value, transforms to the result of function*/
  def mapLimits(f: Map[String, io.k8s.apimachinery.pkg.api.resource.Quantity] => Map[String, io.k8s.apimachinery.pkg.api.resource.Quantity]) : ResourceRequirements = copy(limits = limits.map(f))

  /** Returns a new data with requests set to new value */
  def withRequests(value: Map[String, io.k8s.apimachinery.pkg.api.resource.Quantity]) : ResourceRequirements = copy(requests = Some(value))
  /** Adds new values to requests */
  def addRequests(newValues: (String, io.k8s.apimachinery.pkg.api.resource.Quantity)*) : ResourceRequirements = copy(requests = Some(requests.fold(newValues.toMap)(_ ++ newValues)))
  /** if requests has a value, transforms to the result of function*/
  def mapRequests(f: Map[String, io.k8s.apimachinery.pkg.api.resource.Quantity] => Map[String, io.k8s.apimachinery.pkg.api.resource.Quantity]) : ResourceRequirements = copy(requests = requests.map(f))
}

object ResourceRequirements {

    implicit val encoder : Encoder[io.k8s.api.core.v1.ResourceRequirements] = new Encoder[io.k8s.api.core.v1.ResourceRequirements] {
        def apply[T : Builder](o: io.k8s.api.core.v1.ResourceRequirements) : T = {
          val obj = ObjectWriter[T]()
          obj
            .write("claims", o.claims)
            .write("limits", o.limits)
            .write("requests", o.requests)
            .build
        }
    }

    implicit val decoder: Decoder[ResourceRequirements] = new Decoder[ResourceRequirements] {
      def apply[T : Reader](t: T): Either[String, ResourceRequirements] = for {
          obj <- ObjectReader(t)
          claims <- obj.readOpt[Seq[io.k8s.api.core.v1.ResourceClaim]]("claims")
          limits <- obj.readOpt[Map[String, io.k8s.apimachinery.pkg.api.resource.Quantity]]("limits")
          requests <- obj.readOpt[Map[String, io.k8s.apimachinery.pkg.api.resource.Quantity]]("requests")
      } yield ResourceRequirements (
          claims = claims,
          limits = limits,
          requests = requests
        )
    }
}

