package io.k8s.api.core.v1

import dev.hnaderi.k8s.utils._

/** Represents a Fibre Channel volume. Fibre Channel volumes can only be mounted as read/write once. Fibre Channel volumes support ownership management and SELinux relabeling. */
final case class FCVolumeSource(
  targetWWNs : Option[Seq[String]] = None,
  readOnly : Option[Boolean] = None,
  fsType : Option[String] = None,
  wwids : Option[Seq[String]] = None,
  lun : Option[Int] = None
) {

  /** Returns a new data with targetWWNs set to new value */
  def withTargetWWNs(value: Seq[String]) : FCVolumeSource = copy(targetWWNs = Some(value))
  /** Appends new values to targetWWNs */
  def addTargetWWNs(newValues: String*) : FCVolumeSource = copy(targetWWNs = Some(targetWWNs.fold(newValues)(_ ++ newValues)))
  /** if targetWWNs has a value, transforms to the result of function*/
  def mapTargetWWNs(f: Seq[String] => Seq[String]) : FCVolumeSource = copy(targetWWNs = targetWWNs.map(f))

  /** Returns a new data with readOnly set to new value */
  def withReadOnly(value: Boolean) : FCVolumeSource = copy(readOnly = Some(value))
  /** if readOnly has a value, transforms to the result of function*/
  def mapReadOnly(f: Boolean => Boolean) : FCVolumeSource = copy(readOnly = readOnly.map(f))

  /** Returns a new data with fsType set to new value */
  def withFsType(value: String) : FCVolumeSource = copy(fsType = Some(value))
  /** if fsType has a value, transforms to the result of function*/
  def mapFsType(f: String => String) : FCVolumeSource = copy(fsType = fsType.map(f))

  /** Returns a new data with wwids set to new value */
  def withWwids(value: Seq[String]) : FCVolumeSource = copy(wwids = Some(value))
  /** Appends new values to wwids */
  def addWwids(newValues: String*) : FCVolumeSource = copy(wwids = Some(wwids.fold(newValues)(_ ++ newValues)))
  /** if wwids has a value, transforms to the result of function*/
  def mapWwids(f: Seq[String] => Seq[String]) : FCVolumeSource = copy(wwids = wwids.map(f))

  /** Returns a new data with lun set to new value */
  def withLun(value: Int) : FCVolumeSource = copy(lun = Some(value))
  /** if lun has a value, transforms to the result of function*/
  def mapLun(f: Int => Int) : FCVolumeSource = copy(lun = lun.map(f))
}

object FCVolumeSource {

    implicit val encoder : Encoder[io.k8s.api.core.v1.FCVolumeSource] = new Encoder[io.k8s.api.core.v1.FCVolumeSource] {
        def apply[T : Builder](o: io.k8s.api.core.v1.FCVolumeSource) : T = {
          val obj = ObjectWriter[T]()
          obj
            .write("targetWWNs", o.targetWWNs)
            .write("readOnly", o.readOnly)
            .write("fsType", o.fsType)
            .write("wwids", o.wwids)
            .write("lun", o.lun)
            .build
        }
    }

    implicit val decoder: Decoder[FCVolumeSource] = new Decoder[FCVolumeSource] {
      def apply[T : Reader](t: T): Either[String, FCVolumeSource] = for {
          obj <- ObjectReader(t)
          targetWWNs <- obj.readOpt[Seq[String]]("targetWWNs")
          readOnly <- obj.readOpt[Boolean]("readOnly")
          fsType <- obj.readOpt[String]("fsType")
          wwids <- obj.readOpt[Seq[String]]("wwids")
          lun <- obj.readOpt[Int]("lun")
      } yield FCVolumeSource (
          targetWWNs = targetWWNs,
          readOnly = readOnly,
          fsType = fsType,
          wwids = wwids,
          lun = lun
        )
    }
}

