package io.k8s.apimachinery.pkg.apis.meta.v1

import dev.hnaderi.k8s.utils._

/** OwnerReference contains enough information to let you identify an owning object. An owning object must be in the same namespace as the dependent, or be cluster-scoped, so there is no namespace field. */
final case class OwnerReference(
  name : String,
  uid : String,
  blockOwnerDeletion : Option[Boolean] = None,
  controller : Option[Boolean] = None
) {

  /** Returns a new data with name set to new value */
  def withName(value: String) : OwnerReference = copy(name = value)
  /** transforms name to result of function */
  def mapName(f: String => String) : OwnerReference = copy(name = f(name))

  /** Returns a new data with uid set to new value */
  def withUid(value: String) : OwnerReference = copy(uid = value)
  /** transforms uid to result of function */
  def mapUid(f: String => String) : OwnerReference = copy(uid = f(uid))

  /** Returns a new data with blockOwnerDeletion set to new value */
  def withBlockOwnerDeletion(value: Boolean) : OwnerReference = copy(blockOwnerDeletion = Some(value))
  /** if blockOwnerDeletion has a value, transforms to the result of function*/
  def mapBlockOwnerDeletion(f: Boolean => Boolean) : OwnerReference = copy(blockOwnerDeletion = blockOwnerDeletion.map(f))

  /** Returns a new data with controller set to new value */
  def withController(value: Boolean) : OwnerReference = copy(controller = Some(value))
  /** if controller has a value, transforms to the result of function*/
  def mapController(f: Boolean => Boolean) : OwnerReference = copy(controller = controller.map(f))
}

object OwnerReference {

    implicit def encoder[T](implicit builder : Builder[T]) : Encoder[io.k8s.apimachinery.pkg.apis.meta.v1.OwnerReference, T] = new Encoder[io.k8s.apimachinery.pkg.apis.meta.v1.OwnerReference, T] {
        def apply(o: io.k8s.apimachinery.pkg.apis.meta.v1.OwnerReference) : T = {
          val obj = ObjectWriter[T]()
          obj
            .write("name", o.name)
            .write("uid", o.uid)
            .write("blockOwnerDeletion", o.blockOwnerDeletion)
            .write("controller", o.controller)
            .build
        }
    }

    implicit def decoderOf[T : Reader] : Decoder[T, OwnerReference] = new Decoder[T, OwnerReference] {
      def apply(t: T): Either[String, OwnerReference] = for {
          obj <- ObjectReader(t)
          name <- obj.read[String]("name")
          uid <- obj.read[String]("uid")
          blockOwnerDeletion <- obj.readOpt[Boolean]("blockOwnerDeletion")
          controller <- obj.readOpt[Boolean]("controller")
      } yield OwnerReference (
          name = name,
          uid = uid,
          blockOwnerDeletion = blockOwnerDeletion,
          controller = controller
        )
    }
}

