package io.k8s.api.networking.v1

import dev.hnaderi.k8s.utils._

/** NetworkPolicySpec provides the specification of a NetworkPolicy */
final case class NetworkPolicySpec(
  podSelector : io.k8s.apimachinery.pkg.apis.meta.v1.LabelSelector,
  egress : Option[Seq[io.k8s.api.networking.v1.NetworkPolicyEgressRule]] = None,
  ingress : Option[Seq[io.k8s.api.networking.v1.NetworkPolicyIngressRule]] = None,
  policyTypes : Option[Seq[String]] = None
) {

  /** Returns a new data with podSelector set to new value */
  def withPodSelector(value: io.k8s.apimachinery.pkg.apis.meta.v1.LabelSelector) : NetworkPolicySpec = copy(podSelector = value)
  /** transforms podSelector to result of function */
  def mapPodSelector(f: io.k8s.apimachinery.pkg.apis.meta.v1.LabelSelector => io.k8s.apimachinery.pkg.apis.meta.v1.LabelSelector) : NetworkPolicySpec = copy(podSelector = f(podSelector))

  /** Returns a new data with egress set to new value */
  def withEgress(value: Seq[io.k8s.api.networking.v1.NetworkPolicyEgressRule]) : NetworkPolicySpec = copy(egress = Some(value))
  /** Appends new values to egress */
  def addEgress(newValues: io.k8s.api.networking.v1.NetworkPolicyEgressRule*) : NetworkPolicySpec = copy(egress = Some(egress.fold(newValues)(_ ++ newValues)))
  /** if egress has a value, transforms to the result of function*/
  def mapEgress(f: Seq[io.k8s.api.networking.v1.NetworkPolicyEgressRule] => Seq[io.k8s.api.networking.v1.NetworkPolicyEgressRule]) : NetworkPolicySpec = copy(egress = egress.map(f))

  /** Returns a new data with ingress set to new value */
  def withIngress(value: Seq[io.k8s.api.networking.v1.NetworkPolicyIngressRule]) : NetworkPolicySpec = copy(ingress = Some(value))
  /** Appends new values to ingress */
  def addIngress(newValues: io.k8s.api.networking.v1.NetworkPolicyIngressRule*) : NetworkPolicySpec = copy(ingress = Some(ingress.fold(newValues)(_ ++ newValues)))
  /** if ingress has a value, transforms to the result of function*/
  def mapIngress(f: Seq[io.k8s.api.networking.v1.NetworkPolicyIngressRule] => Seq[io.k8s.api.networking.v1.NetworkPolicyIngressRule]) : NetworkPolicySpec = copy(ingress = ingress.map(f))

  /** Returns a new data with policyTypes set to new value */
  def withPolicyTypes(value: Seq[String]) : NetworkPolicySpec = copy(policyTypes = Some(value))
  /** Appends new values to policyTypes */
  def addPolicyTypes(newValues: String*) : NetworkPolicySpec = copy(policyTypes = Some(policyTypes.fold(newValues)(_ ++ newValues)))
  /** if policyTypes has a value, transforms to the result of function*/
  def mapPolicyTypes(f: Seq[String] => Seq[String]) : NetworkPolicySpec = copy(policyTypes = policyTypes.map(f))
}

object NetworkPolicySpec {

    implicit def encoder[T](implicit builder : Builder[T]) : Encoder[io.k8s.api.networking.v1.NetworkPolicySpec, T] = new Encoder[io.k8s.api.networking.v1.NetworkPolicySpec, T] {
        def apply(o: io.k8s.api.networking.v1.NetworkPolicySpec) : T = {
          val obj = ObjectWriter[T]()
          obj
            .write("podSelector", o.podSelector)
            .write("egress", o.egress)
            .write("ingress", o.ingress)
            .write("policyTypes", o.policyTypes)
            .build
        }
    }

    implicit def decoderOf[T : Reader] : Decoder[T, NetworkPolicySpec] = new Decoder[T, NetworkPolicySpec] {
      def apply(t: T): Either[String, NetworkPolicySpec] = for {
          obj <- ObjectReader(t)
          podSelector <- obj.read[io.k8s.apimachinery.pkg.apis.meta.v1.LabelSelector]("podSelector")
          egress <- obj.readOpt[Seq[io.k8s.api.networking.v1.NetworkPolicyEgressRule]]("egress")
          ingress <- obj.readOpt[Seq[io.k8s.api.networking.v1.NetworkPolicyIngressRule]]("ingress")
          policyTypes <- obj.readOpt[Seq[String]]("policyTypes")
      } yield NetworkPolicySpec (
          podSelector = podSelector,
          egress = egress,
          ingress = ingress,
          policyTypes = policyTypes
        )
    }
}

