package io.k8s.api.core.v1

import dev.hnaderi.k8s.utils._

/** Defines a set of pods (namely those matching the labelSelector relative to the given namespace(s)) that this pod should be co-located (affinity) or not co-located (anti-affinity) with, where co-located is defined as running on a node whose value of the label with key <topologyKey> matches that of any node on which a pod of the set of pods is running */
final case class PodAffinityTerm(
  topologyKey : String,
  labelSelector : Option[io.k8s.apimachinery.pkg.apis.meta.v1.LabelSelector] = None,
  namespaceSelector : Option[io.k8s.apimachinery.pkg.apis.meta.v1.LabelSelector] = None,
  namespaces : Option[Seq[String]] = None
) {

  /** Returns a new data with topologyKey set to new value */
  def withTopologyKey(value: String) : PodAffinityTerm = copy(topologyKey = value)
  /** transforms topologyKey to result of function */
  def mapTopologyKey(f: String => String) : PodAffinityTerm = copy(topologyKey = f(topologyKey))

  /** Returns a new data with labelSelector set to new value */
  def withLabelSelector(value: io.k8s.apimachinery.pkg.apis.meta.v1.LabelSelector) : PodAffinityTerm = copy(labelSelector = Some(value))
  /** if labelSelector has a value, transforms to the result of function*/
  def mapLabelSelector(f: io.k8s.apimachinery.pkg.apis.meta.v1.LabelSelector => io.k8s.apimachinery.pkg.apis.meta.v1.LabelSelector) : PodAffinityTerm = copy(labelSelector = labelSelector.map(f))

  /** Returns a new data with namespaceSelector set to new value */
  def withNamespaceSelector(value: io.k8s.apimachinery.pkg.apis.meta.v1.LabelSelector) : PodAffinityTerm = copy(namespaceSelector = Some(value))
  /** if namespaceSelector has a value, transforms to the result of function*/
  def mapNamespaceSelector(f: io.k8s.apimachinery.pkg.apis.meta.v1.LabelSelector => io.k8s.apimachinery.pkg.apis.meta.v1.LabelSelector) : PodAffinityTerm = copy(namespaceSelector = namespaceSelector.map(f))

  /** Returns a new data with namespaces set to new value */
  def withNamespaces(value: Seq[String]) : PodAffinityTerm = copy(namespaces = Some(value))
  /** Appends new values to namespaces */
  def addNamespaces(newValues: String*) : PodAffinityTerm = copy(namespaces = Some(namespaces.fold(newValues)(_ ++ newValues)))
  /** if namespaces has a value, transforms to the result of function*/
  def mapNamespaces(f: Seq[String] => Seq[String]) : PodAffinityTerm = copy(namespaces = namespaces.map(f))
}

object PodAffinityTerm {

    implicit def encoder[T](implicit builder : Builder[T]) : Encoder[io.k8s.api.core.v1.PodAffinityTerm, T] = new Encoder[io.k8s.api.core.v1.PodAffinityTerm, T] {
        def apply(o: io.k8s.api.core.v1.PodAffinityTerm) : T = {
          val obj = ObjectWriter[T]()
          obj
            .write("topologyKey", o.topologyKey)
            .write("labelSelector", o.labelSelector)
            .write("namespaceSelector", o.namespaceSelector)
            .write("namespaces", o.namespaces)
            .build
        }
    }

    implicit def decoderOf[T : Reader] : Decoder[T, PodAffinityTerm] = new Decoder[T, PodAffinityTerm] {
      def apply(t: T): Either[String, PodAffinityTerm] = for {
          obj <- ObjectReader(t)
          topologyKey <- obj.read[String]("topologyKey")
          labelSelector <- obj.readOpt[io.k8s.apimachinery.pkg.apis.meta.v1.LabelSelector]("labelSelector")
          namespaceSelector <- obj.readOpt[io.k8s.apimachinery.pkg.apis.meta.v1.LabelSelector]("namespaceSelector")
          namespaces <- obj.readOpt[Seq[String]]("namespaces")
      } yield PodAffinityTerm (
          topologyKey = topologyKey,
          labelSelector = labelSelector,
          namespaceSelector = namespaceSelector,
          namespaces = namespaces
        )
    }
}

