package io.k8s.api.core.v1

import dev.hnaderi.k8s.utils._

/** Maps a string key to a path within a volume. */
final case class KeyToPath(
  key : String,
  path : String,
  mode : Option[Int] = None
) {

  /** Returns a new data with key set to new value */
  def withKey(value: String) : KeyToPath = copy(key = value)
  /** transforms key to result of function */
  def mapKey(f: String => String) : KeyToPath = copy(key = f(key))

  /** Returns a new data with path set to new value */
  def withPath(value: String) : KeyToPath = copy(path = value)
  /** transforms path to result of function */
  def mapPath(f: String => String) : KeyToPath = copy(path = f(path))

  /** Returns a new data with mode set to new value */
  def withMode(value: Int) : KeyToPath = copy(mode = Some(value))
  /** if mode has a value, transforms to the result of function*/
  def mapMode(f: Int => Int) : KeyToPath = copy(mode = mode.map(f))
}

object KeyToPath {

    implicit def encoder[T](implicit builder : Builder[T]) : Encoder[io.k8s.api.core.v1.KeyToPath, T] = new Encoder[io.k8s.api.core.v1.KeyToPath, T] {
        def apply(o: io.k8s.api.core.v1.KeyToPath) : T = {
          val obj = ObjectWriter[T]()
          obj
            .write("key", o.key)
            .write("path", o.path)
            .write("mode", o.mode)
            .build
        }
    }

    implicit def decoderOf[T : Reader] : Decoder[T, KeyToPath] = new Decoder[T, KeyToPath] {
      def apply(t: T): Either[String, KeyToPath] = for {
          obj <- ObjectReader(t)
          key <- obj.read[String]("key")
          path <- obj.read[String]("path")
          mode <- obj.readOpt[Int]("mode")
      } yield KeyToPath (
          key = key,
          path = path,
          mode = mode
        )
    }
}

