package io.k8s.api.core.v1

import dev.hnaderi.k8s.utils._

/** Represents a Glusterfs mount that lasts the lifetime of a pod. Glusterfs volumes do not support ownership management or SELinux relabeling. */
final case class GlusterfsVolumeSource(
  endpoints : String,
  path : String,
  readOnly : Option[Boolean] = None
) {

  /** Returns a new data with endpoints set to new value */
  def withEndpoints(value: String) : GlusterfsVolumeSource = copy(endpoints = value)
  /** transforms endpoints to result of function */
  def mapEndpoints(f: String => String) : GlusterfsVolumeSource = copy(endpoints = f(endpoints))

  /** Returns a new data with path set to new value */
  def withPath(value: String) : GlusterfsVolumeSource = copy(path = value)
  /** transforms path to result of function */
  def mapPath(f: String => String) : GlusterfsVolumeSource = copy(path = f(path))

  /** Returns a new data with readOnly set to new value */
  def withReadOnly(value: Boolean) : GlusterfsVolumeSource = copy(readOnly = Some(value))
  /** if readOnly has a value, transforms to the result of function*/
  def mapReadOnly(f: Boolean => Boolean) : GlusterfsVolumeSource = copy(readOnly = readOnly.map(f))
}

object GlusterfsVolumeSource {

    implicit def encoder[T](implicit builder : Builder[T]) : Encoder[io.k8s.api.core.v1.GlusterfsVolumeSource, T] = new Encoder[io.k8s.api.core.v1.GlusterfsVolumeSource, T] {
        def apply(o: io.k8s.api.core.v1.GlusterfsVolumeSource) : T = {
          val obj = ObjectWriter[T]()
          obj
            .write("endpoints", o.endpoints)
            .write("path", o.path)
            .write("readOnly", o.readOnly)
            .build
        }
    }

    implicit def decoderOf[T : Reader] : Decoder[T, GlusterfsVolumeSource] = new Decoder[T, GlusterfsVolumeSource] {
      def apply(t: T): Either[String, GlusterfsVolumeSource] = for {
          obj <- ObjectReader(t)
          endpoints <- obj.read[String]("endpoints")
          path <- obj.read[String]("path")
          readOnly <- obj.readOpt[Boolean]("readOnly")
      } yield GlusterfsVolumeSource (
          endpoints = endpoints,
          path = path,
          readOnly = readOnly
        )
    }
}

