package io.k8s.api.core.v1

import dev.hnaderi.k8s.utils._

/** EnvVar represents an environment variable present in a Container. */
final case class EnvVar(
  name : String,
  value : Option[String] = None,
  valueFrom : Option[io.k8s.api.core.v1.EnvVarSource] = None
) {

  /** Returns a new data with name set to new value */
  def withName(value: String) : EnvVar = copy(name = value)
  /** transforms name to result of function */
  def mapName(f: String => String) : EnvVar = copy(name = f(name))

  /** Returns a new data with value set to new value */
  def withValue(value: String) : EnvVar = copy(value = Some(value))
  /** if value has a value, transforms to the result of function*/
  def mapValue(f: String => String) : EnvVar = copy(value = value.map(f))

  /** Returns a new data with valueFrom set to new value */
  def withValueFrom(value: io.k8s.api.core.v1.EnvVarSource) : EnvVar = copy(valueFrom = Some(value))
  /** if valueFrom has a value, transforms to the result of function*/
  def mapValueFrom(f: io.k8s.api.core.v1.EnvVarSource => io.k8s.api.core.v1.EnvVarSource) : EnvVar = copy(valueFrom = valueFrom.map(f))
}

object EnvVar {

    implicit def encoder[T](implicit builder : Builder[T]) : Encoder[io.k8s.api.core.v1.EnvVar, T] = new Encoder[io.k8s.api.core.v1.EnvVar, T] {
        def apply(o: io.k8s.api.core.v1.EnvVar) : T = {
          val obj = ObjectWriter[T]()
          obj
            .write("name", o.name)
            .write("value", o.value)
            .write("valueFrom", o.valueFrom)
            .build
        }
    }

    implicit def decoderOf[T : Reader] : Decoder[T, EnvVar] = new Decoder[T, EnvVar] {
      def apply(t: T): Either[String, EnvVar] = for {
          obj <- ObjectReader(t)
          name <- obj.read[String]("name")
          value <- obj.readOpt[String]("value")
          valueFrom <- obj.readOpt[io.k8s.api.core.v1.EnvVarSource]("valueFrom")
      } yield EnvVar (
          name = name,
          value = value,
          valueFrom = valueFrom
        )
    }
}

