package io.k8s.api.core.v1

import dev.hnaderi.k8s.utils._

/** Selects a key from a ConfigMap. */
final case class ConfigMapKeySelector(
  key : String,
  name : Option[String] = None,
  optional : Option[Boolean] = None
) {

  /** Returns a new data with key set to new value */
  def withKey(value: String) : ConfigMapKeySelector = copy(key = value)
  /** transforms key to result of function */
  def mapKey(f: String => String) : ConfigMapKeySelector = copy(key = f(key))

  /** Returns a new data with name set to new value */
  def withName(value: String) : ConfigMapKeySelector = copy(name = Some(value))
  /** if name has a value, transforms to the result of function*/
  def mapName(f: String => String) : ConfigMapKeySelector = copy(name = name.map(f))

  /** Returns a new data with optional set to new value */
  def withOptional(value: Boolean) : ConfigMapKeySelector = copy(optional = Some(value))
  /** if optional has a value, transforms to the result of function*/
  def mapOptional(f: Boolean => Boolean) : ConfigMapKeySelector = copy(optional = optional.map(f))
}

object ConfigMapKeySelector {

    implicit def encoder[T](implicit builder : Builder[T]) : Encoder[io.k8s.api.core.v1.ConfigMapKeySelector, T] = new Encoder[io.k8s.api.core.v1.ConfigMapKeySelector, T] {
        def apply(o: io.k8s.api.core.v1.ConfigMapKeySelector) : T = {
          val obj = ObjectWriter[T]()
          obj
            .write("key", o.key)
            .write("name", o.name)
            .write("optional", o.optional)
            .build
        }
    }

    implicit def decoderOf[T : Reader] : Decoder[T, ConfigMapKeySelector] = new Decoder[T, ConfigMapKeySelector] {
      def apply(t: T): Either[String, ConfigMapKeySelector] = for {
          obj <- ObjectReader(t)
          key <- obj.read[String]("key")
          name <- obj.readOpt[String]("name")
          optional <- obj.readOpt[Boolean]("optional")
      } yield ConfigMapKeySelector (
          key = key,
          name = name,
          optional = optional
        )
    }
}

