package io.k8s.api.authorization.v1


/** ResourceRule is the list of actions the subject is allowed to perform on resources. The list ordering isn't significant, may contain duplicates, and possibly be incomplete. */
final case class ResourceRule(
  verbs : Seq[String],
  apiGroups : Option[Seq[String]] = None,
  resourceNames : Option[Seq[String]] = None,
  resources : Option[Seq[String]] = None
) {

  /** Returns a new data with verbs set to new value */
  def withVerbs(value: Seq[String]) : ResourceRule = copy(verbs = value)
  /** Appends new values to verbs */
  def addVerbs(newValues: String*) : ResourceRule = copy(verbs = verbs ++ newValues)
  /** transforms verbs to result of function */
  def mapVerbs(f: Seq[String] => Seq[String]) : ResourceRule = copy(verbs = f(verbs))

  /** Returns a new data with apiGroups set to new value */
  def withApiGroups(value: Seq[String]) : ResourceRule = copy(apiGroups = Some(value))
  /** Appends new values to apiGroups */
  def addApiGroups(newValues: String*) : ResourceRule = copy(apiGroups = Some(apiGroups.fold(newValues)(_ ++ newValues)))
  /** if apiGroups has a value, transforms to the result of function*/
  def mapApiGroups(f: Seq[String] => Seq[String]) : ResourceRule = copy(apiGroups = apiGroups.map(f))

  /** Returns a new data with resourceNames set to new value */
  def withResourceNames(value: Seq[String]) : ResourceRule = copy(resourceNames = Some(value))
  /** Appends new values to resourceNames */
  def addResourceNames(newValues: String*) : ResourceRule = copy(resourceNames = Some(resourceNames.fold(newValues)(_ ++ newValues)))
  /** if resourceNames has a value, transforms to the result of function*/
  def mapResourceNames(f: Seq[String] => Seq[String]) : ResourceRule = copy(resourceNames = resourceNames.map(f))

  /** Returns a new data with resources set to new value */
  def withResources(value: Seq[String]) : ResourceRule = copy(resources = Some(value))
  /** Appends new values to resources */
  def addResources(newValues: String*) : ResourceRule = copy(resources = Some(resources.fold(newValues)(_ ++ newValues)))
  /** if resources has a value, transforms to the result of function*/
  def mapResources(f: Seq[String] => Seq[String]) : ResourceRule = copy(resources = resources.map(f))
}

