package io.k8s.api.admissionregistration.v1alpha1

import dev.hnaderi.k8s.utils._

/** Validation specifies the CEL expression which is used to apply the validation. */
final case class Validation(
  expression : String,
  message : Option[String] = None,
  messageExpression : Option[String] = None,
  reason : Option[String] = None
) {

  /** Returns a new data with expression set to new value */
  def withExpression(value: String) : Validation = copy(expression = value)
  /** transforms expression to result of function */
  def mapExpression(f: String => String) : Validation = copy(expression = f(expression))

  /** Returns a new data with message set to new value */
  def withMessage(value: String) : Validation = copy(message = Some(value))
  /** if message has a value, transforms to the result of function*/
  def mapMessage(f: String => String) : Validation = copy(message = message.map(f))

  /** Returns a new data with messageExpression set to new value */
  def withMessageExpression(value: String) : Validation = copy(messageExpression = Some(value))
  /** if messageExpression has a value, transforms to the result of function*/
  def mapMessageExpression(f: String => String) : Validation = copy(messageExpression = messageExpression.map(f))

  /** Returns a new data with reason set to new value */
  def withReason(value: String) : Validation = copy(reason = Some(value))
  /** if reason has a value, transforms to the result of function*/
  def mapReason(f: String => String) : Validation = copy(reason = reason.map(f))
}

object Validation {

    implicit val encoder : Encoder[io.k8s.api.admissionregistration.v1alpha1.Validation] = new Encoder[io.k8s.api.admissionregistration.v1alpha1.Validation] {
        def apply[T : Builder](o: io.k8s.api.admissionregistration.v1alpha1.Validation) : T = {
          val obj = ObjectWriter[T]()
          obj
            .write("expression", o.expression)
            .write("message", o.message)
            .write("messageExpression", o.messageExpression)
            .write("reason", o.reason)
            .build
        }
    }

    implicit val decoder: Decoder[Validation] = new Decoder[Validation] {
      def apply[T : Reader](t: T): Either[String, Validation] = for {
          obj <- ObjectReader(t)
          expression <- obj.read[String]("expression")
          message <- obj.readOpt[String]("message")
          messageExpression <- obj.readOpt[String]("messageExpression")
          reason <- obj.readOpt[String]("reason")
      } yield Validation (
          expression = expression,
          message = message,
          messageExpression = messageExpression,
          reason = reason
        )
    }
}

