package io.k8s.api.core.v1

import dev.hnaderi.k8s.utils._

/** A scoped-resource selector requirement is a selector that contains values, a scope name, and an operator that relates the scope name and values. */
final case class ScopedResourceSelectorRequirement(
  operator : String,
  scopeName : String,
  values : Option[Seq[String]] = None
) {

  /** Returns a new data with operator set to new value */
  def withOperator(value: String) : ScopedResourceSelectorRequirement = copy(operator = value)
  /** transforms operator to result of function */
  def mapOperator(f: String => String) : ScopedResourceSelectorRequirement = copy(operator = f(operator))

  /** Returns a new data with scopeName set to new value */
  def withScopeName(value: String) : ScopedResourceSelectorRequirement = copy(scopeName = value)
  /** transforms scopeName to result of function */
  def mapScopeName(f: String => String) : ScopedResourceSelectorRequirement = copy(scopeName = f(scopeName))

  /** Returns a new data with values set to new value */
  def withValues(value: Seq[String]) : ScopedResourceSelectorRequirement = copy(values = Some(value))
  /** Appends new values to values */
  def addValues(newValues: String*) : ScopedResourceSelectorRequirement = copy(values = Some(values.fold(newValues)(_ ++ newValues)))
  /** if values has a value, transforms to the result of function*/
  def mapValues(f: Seq[String] => Seq[String]) : ScopedResourceSelectorRequirement = copy(values = values.map(f))
}

object ScopedResourceSelectorRequirement {

    implicit val encoder : Encoder[io.k8s.api.core.v1.ScopedResourceSelectorRequirement] = new Encoder[io.k8s.api.core.v1.ScopedResourceSelectorRequirement] {
        def apply[T : Builder](o: io.k8s.api.core.v1.ScopedResourceSelectorRequirement) : T = {
          val obj = ObjectWriter[T]()
          obj
            .write("operator", o.operator)
            .write("scopeName", o.scopeName)
            .write("values", o.values)
            .build
        }
    }

    implicit val decoder: Decoder[ScopedResourceSelectorRequirement] = new Decoder[ScopedResourceSelectorRequirement] {
      def apply[T : Reader](t: T): Either[String, ScopedResourceSelectorRequirement] = for {
          obj <- ObjectReader(t)
          operator <- obj.read[String]("operator")
          scopeName <- obj.read[String]("scopeName")
          values <- obj.readOpt[Seq[String]]("values")
      } yield ScopedResourceSelectorRequirement (
          operator = operator,
          scopeName = scopeName,
          values = values
        )
    }
}

