package io.k8s.apimachinery.pkg.apis.meta.v1

import dev.hnaderi.k8s.utils._

/** APIResource specifies the name of a resource and whether it is namespaced. */
final case class APIResource(
  name : String,
  verbs : Seq[String],
  namespaced : Boolean,
  singularName : String,
  version : Option[String] = None,
  categories : Option[Seq[String]] = None,
  shortNames : Option[Seq[String]] = None,
  storageVersionHash : Option[String] = None,
  group : Option[String] = None
) {

  /** Returns a new data with name set to new value */
  def withName(value: String) : APIResource = copy(name = value)
  /** transforms name to result of function */
  def mapName(f: String => String) : APIResource = copy(name = f(name))

  /** Returns a new data with verbs set to new value */
  def withVerbs(value: Seq[String]) : APIResource = copy(verbs = value)
  /** Appends new values to verbs */
  def addVerbs(newValues: String*) : APIResource = copy(verbs = verbs ++ newValues)
  /** transforms verbs to result of function */
  def mapVerbs(f: Seq[String] => Seq[String]) : APIResource = copy(verbs = f(verbs))

  /** Returns a new data with namespaced set to new value */
  def withNamespaced(value: Boolean) : APIResource = copy(namespaced = value)
  /** transforms namespaced to result of function */
  def mapNamespaced(f: Boolean => Boolean) : APIResource = copy(namespaced = f(namespaced))

  /** Returns a new data with singularName set to new value */
  def withSingularName(value: String) : APIResource = copy(singularName = value)
  /** transforms singularName to result of function */
  def mapSingularName(f: String => String) : APIResource = copy(singularName = f(singularName))

  /** Returns a new data with version set to new value */
  def withVersion(value: String) : APIResource = copy(version = Some(value))
  /** if version has a value, transforms to the result of function*/
  def mapVersion(f: String => String) : APIResource = copy(version = version.map(f))

  /** Returns a new data with categories set to new value */
  def withCategories(value: Seq[String]) : APIResource = copy(categories = Some(value))
  /** Appends new values to categories */
  def addCategories(newValues: String*) : APIResource = copy(categories = Some(categories.fold(newValues)(_ ++ newValues)))
  /** if categories has a value, transforms to the result of function*/
  def mapCategories(f: Seq[String] => Seq[String]) : APIResource = copy(categories = categories.map(f))

  /** Returns a new data with shortNames set to new value */
  def withShortNames(value: Seq[String]) : APIResource = copy(shortNames = Some(value))
  /** Appends new values to shortNames */
  def addShortNames(newValues: String*) : APIResource = copy(shortNames = Some(shortNames.fold(newValues)(_ ++ newValues)))
  /** if shortNames has a value, transforms to the result of function*/
  def mapShortNames(f: Seq[String] => Seq[String]) : APIResource = copy(shortNames = shortNames.map(f))

  /** Returns a new data with storageVersionHash set to new value */
  def withStorageVersionHash(value: String) : APIResource = copy(storageVersionHash = Some(value))
  /** if storageVersionHash has a value, transforms to the result of function*/
  def mapStorageVersionHash(f: String => String) : APIResource = copy(storageVersionHash = storageVersionHash.map(f))

  /** Returns a new data with group set to new value */
  def withGroup(value: String) : APIResource = copy(group = Some(value))
  /** if group has a value, transforms to the result of function*/
  def mapGroup(f: String => String) : APIResource = copy(group = group.map(f))
}

object APIResource {

    implicit val encoder : Encoder[io.k8s.apimachinery.pkg.apis.meta.v1.APIResource] = new Encoder[io.k8s.apimachinery.pkg.apis.meta.v1.APIResource] {
        def apply[T : Builder](o: io.k8s.apimachinery.pkg.apis.meta.v1.APIResource) : T = {
          val obj = ObjectWriter[T]()
          obj
            .write("name", o.name)
            .write("verbs", o.verbs)
            .write("namespaced", o.namespaced)
            .write("singularName", o.singularName)
            .write("version", o.version)
            .write("categories", o.categories)
            .write("shortNames", o.shortNames)
            .write("storageVersionHash", o.storageVersionHash)
            .write("group", o.group)
            .build
        }
    }

    implicit val decoder: Decoder[APIResource] = new Decoder[APIResource] {
      def apply[T : Reader](t: T): Either[String, APIResource] = for {
          obj <- ObjectReader(t)
          name <- obj.read[String]("name")
          verbs <- obj.read[Seq[String]]("verbs")
          namespaced <- obj.read[Boolean]("namespaced")
          singularName <- obj.read[String]("singularName")
          version <- obj.readOpt[String]("version")
          categories <- obj.readOpt[Seq[String]]("categories")
          shortNames <- obj.readOpt[Seq[String]]("shortNames")
          storageVersionHash <- obj.readOpt[String]("storageVersionHash")
          group <- obj.readOpt[String]("group")
      } yield APIResource (
          name = name,
          verbs = verbs,
          namespaced = namespaced,
          singularName = singularName,
          version = version,
          categories = categories,
          shortNames = shortNames,
          storageVersionHash = storageVersionHash,
          group = group
        )
    }
}

