package io.k8s.api.node.v1

import dev.hnaderi.k8s.utils._

/** Scheduling specifies the scheduling constraints for nodes supporting a RuntimeClass. */
final case class Scheduling(
  nodeSelector : Option[Map[String, String]] = None,
  tolerations : Option[Seq[io.k8s.api.core.v1.Toleration]] = None
) {

  /** Returns a new data with nodeSelector set to new value */
  def withNodeSelector(value: Map[String, String]) : Scheduling = copy(nodeSelector = Some(value))
  /** Adds new values to nodeSelector */
  def addNodeSelector(newValues: (String, String)*) : Scheduling = copy(nodeSelector = Some(nodeSelector.fold(newValues.toMap)(_ ++ newValues)))
  /** if nodeSelector has a value, transforms to the result of function*/
  def mapNodeSelector(f: Map[String, String] => Map[String, String]) : Scheduling = copy(nodeSelector = nodeSelector.map(f))

  /** Returns a new data with tolerations set to new value */
  def withTolerations(value: Seq[io.k8s.api.core.v1.Toleration]) : Scheduling = copy(tolerations = Some(value))
  /** Appends new values to tolerations */
  def addTolerations(newValues: io.k8s.api.core.v1.Toleration*) : Scheduling = copy(tolerations = Some(tolerations.fold(newValues)(_ ++ newValues)))
  /** if tolerations has a value, transforms to the result of function*/
  def mapTolerations(f: Seq[io.k8s.api.core.v1.Toleration] => Seq[io.k8s.api.core.v1.Toleration]) : Scheduling = copy(tolerations = tolerations.map(f))
}

object Scheduling {

    implicit val encoder : Encoder[io.k8s.api.node.v1.Scheduling] = new Encoder[io.k8s.api.node.v1.Scheduling] {
        def apply[T : Builder](o: io.k8s.api.node.v1.Scheduling) : T = {
          val obj = ObjectWriter[T]()
          obj
            .write("nodeSelector", o.nodeSelector)
            .write("tolerations", o.tolerations)
            .build
        }
    }

    implicit val decoder: Decoder[Scheduling] = new Decoder[Scheduling] {
      def apply[T : Reader](t: T): Either[String, Scheduling] = for {
          obj <- ObjectReader(t)
          nodeSelector <- obj.readOpt[Map[String, String]]("nodeSelector")
          tolerations <- obj.readOpt[Seq[io.k8s.api.core.v1.Toleration]]("tolerations")
      } yield Scheduling (
          nodeSelector = nodeSelector,
          tolerations = tolerations
        )
    }
}

