package io.k8s.api.storage.v1

import dev.hnaderi.k8s.utils._

/** TokenRequest contains parameters of a service account token. */
final case class TokenRequest(
  audience : String,
  expirationSeconds : Option[Int] = None
) {

  /** Returns a new data with audience set to new value */
  def withAudience(value: String) : TokenRequest = copy(audience = value)
  /** transforms audience to result of function */
  def mapAudience(f: String => String) : TokenRequest = copy(audience = f(audience))

  /** Returns a new data with expirationSeconds set to new value */
  def withExpirationSeconds(value: Int) : TokenRequest = copy(expirationSeconds = Some(value))
  /** if expirationSeconds has a value, transforms to the result of function*/
  def mapExpirationSeconds(f: Int => Int) : TokenRequest = copy(expirationSeconds = expirationSeconds.map(f))
}

object TokenRequest {

    implicit def encoder[T](implicit builder : Builder[T]) : Encoder[io.k8s.api.storage.v1.TokenRequest, T] = new Encoder[io.k8s.api.storage.v1.TokenRequest, T] {
        def apply(o: io.k8s.api.storage.v1.TokenRequest) : T = {
          val obj = ObjectWriter[T]()
          obj
            .write("audience", o.audience)
            .write("expirationSeconds", o.expirationSeconds)
            .build
        }
    }

    implicit def decoderOf[T : Reader] : Decoder[T, TokenRequest] = new Decoder[T, TokenRequest] {
      def apply(t: T): Either[String, TokenRequest] = for {
          obj <- ObjectReader(t)
          audience <- obj.read[String]("audience")
          expirationSeconds <- obj.readOpt[Int]("expirationSeconds")
      } yield TokenRequest (
          audience = audience,
          expirationSeconds = expirationSeconds
        )
    }
}

