package io.k8s.api.storage.v1

import dev.hnaderi.k8s.utils._

/** CSINodeDriver holds information about the specification of one CSI driver installed on a node */
final case class CSINodeDriver(
  name : String,
  nodeID : String,
  allocatable : Option[io.k8s.api.storage.v1.VolumeNodeResources] = None,
  topologyKeys : Option[Seq[String]] = None
) {

  /** Returns a new data with name set to new value */
  def withName(value: String) : CSINodeDriver = copy(name = value)
  /** transforms name to result of function */
  def mapName(f: String => String) : CSINodeDriver = copy(name = f(name))

  /** Returns a new data with nodeID set to new value */
  def withNodeID(value: String) : CSINodeDriver = copy(nodeID = value)
  /** transforms nodeID to result of function */
  def mapNodeID(f: String => String) : CSINodeDriver = copy(nodeID = f(nodeID))

  /** Returns a new data with allocatable set to new value */
  def withAllocatable(value: io.k8s.api.storage.v1.VolumeNodeResources) : CSINodeDriver = copy(allocatable = Some(value))
  /** if allocatable has a value, transforms to the result of function*/
  def mapAllocatable(f: io.k8s.api.storage.v1.VolumeNodeResources => io.k8s.api.storage.v1.VolumeNodeResources) : CSINodeDriver = copy(allocatable = allocatable.map(f))

  /** Returns a new data with topologyKeys set to new value */
  def withTopologyKeys(value: Seq[String]) : CSINodeDriver = copy(topologyKeys = Some(value))
  /** Appends new values to topologyKeys */
  def addTopologyKeys(newValues: String*) : CSINodeDriver = copy(topologyKeys = Some(topologyKeys.fold(newValues)(_ ++ newValues)))
  /** if topologyKeys has a value, transforms to the result of function*/
  def mapTopologyKeys(f: Seq[String] => Seq[String]) : CSINodeDriver = copy(topologyKeys = topologyKeys.map(f))
}

object CSINodeDriver {

    implicit def encoder[T](implicit builder : Builder[T]) : Encoder[io.k8s.api.storage.v1.CSINodeDriver, T] = new Encoder[io.k8s.api.storage.v1.CSINodeDriver, T] {
        def apply(o: io.k8s.api.storage.v1.CSINodeDriver) : T = {
          val obj = ObjectWriter[T]()
          obj
            .write("name", o.name)
            .write("nodeID", o.nodeID)
            .write("allocatable", o.allocatable)
            .write("topologyKeys", o.topologyKeys)
            .build
        }
    }

    implicit def decoderOf[T : Reader] : Decoder[T, CSINodeDriver] = new Decoder[T, CSINodeDriver] {
      def apply(t: T): Either[String, CSINodeDriver] = for {
          obj <- ObjectReader(t)
          name <- obj.read[String]("name")
          nodeID <- obj.read[String]("nodeID")
          allocatable <- obj.readOpt[io.k8s.api.storage.v1.VolumeNodeResources]("allocatable")
          topologyKeys <- obj.readOpt[Seq[String]]("topologyKeys")
      } yield CSINodeDriver (
          name = name,
          nodeID = nodeID,
          allocatable = allocatable,
          topologyKeys = topologyKeys
        )
    }
}

