package io.k8s.api.core.v1

import dev.hnaderi.k8s.utils._

/** TopologySpreadConstraint specifies how to spread matching pods among the given topology. */
final case class TopologySpreadConstraint(
  maxSkew : Int,
  topologyKey : String,
  whenUnsatisfiable : String,
  matchLabelKeys : Option[Seq[String]] = None,
  minDomains : Option[Int] = None,
  labelSelector : Option[io.k8s.apimachinery.pkg.apis.meta.v1.LabelSelector] = None,
  nodeAffinityPolicy : Option[String] = None,
  nodeTaintsPolicy : Option[String] = None
) {

  /** Returns a new data with maxSkew set to new value */
  def withMaxSkew(value: Int) : TopologySpreadConstraint = copy(maxSkew = value)
  /** transforms maxSkew to result of function */
  def mapMaxSkew(f: Int => Int) : TopologySpreadConstraint = copy(maxSkew = f(maxSkew))

  /** Returns a new data with topologyKey set to new value */
  def withTopologyKey(value: String) : TopologySpreadConstraint = copy(topologyKey = value)
  /** transforms topologyKey to result of function */
  def mapTopologyKey(f: String => String) : TopologySpreadConstraint = copy(topologyKey = f(topologyKey))

  /** Returns a new data with whenUnsatisfiable set to new value */
  def withWhenUnsatisfiable(value: String) : TopologySpreadConstraint = copy(whenUnsatisfiable = value)
  /** transforms whenUnsatisfiable to result of function */
  def mapWhenUnsatisfiable(f: String => String) : TopologySpreadConstraint = copy(whenUnsatisfiable = f(whenUnsatisfiable))

  /** Returns a new data with matchLabelKeys set to new value */
  def withMatchLabelKeys(value: Seq[String]) : TopologySpreadConstraint = copy(matchLabelKeys = Some(value))
  /** Appends new values to matchLabelKeys */
  def addMatchLabelKeys(newValues: String*) : TopologySpreadConstraint = copy(matchLabelKeys = Some(matchLabelKeys.fold(newValues)(_ ++ newValues)))
  /** if matchLabelKeys has a value, transforms to the result of function*/
  def mapMatchLabelKeys(f: Seq[String] => Seq[String]) : TopologySpreadConstraint = copy(matchLabelKeys = matchLabelKeys.map(f))

  /** Returns a new data with minDomains set to new value */
  def withMinDomains(value: Int) : TopologySpreadConstraint = copy(minDomains = Some(value))
  /** if minDomains has a value, transforms to the result of function*/
  def mapMinDomains(f: Int => Int) : TopologySpreadConstraint = copy(minDomains = minDomains.map(f))

  /** Returns a new data with labelSelector set to new value */
  def withLabelSelector(value: io.k8s.apimachinery.pkg.apis.meta.v1.LabelSelector) : TopologySpreadConstraint = copy(labelSelector = Some(value))
  /** if labelSelector has a value, transforms to the result of function*/
  def mapLabelSelector(f: io.k8s.apimachinery.pkg.apis.meta.v1.LabelSelector => io.k8s.apimachinery.pkg.apis.meta.v1.LabelSelector) : TopologySpreadConstraint = copy(labelSelector = labelSelector.map(f))

  /** Returns a new data with nodeAffinityPolicy set to new value */
  def withNodeAffinityPolicy(value: String) : TopologySpreadConstraint = copy(nodeAffinityPolicy = Some(value))
  /** if nodeAffinityPolicy has a value, transforms to the result of function*/
  def mapNodeAffinityPolicy(f: String => String) : TopologySpreadConstraint = copy(nodeAffinityPolicy = nodeAffinityPolicy.map(f))

  /** Returns a new data with nodeTaintsPolicy set to new value */
  def withNodeTaintsPolicy(value: String) : TopologySpreadConstraint = copy(nodeTaintsPolicy = Some(value))
  /** if nodeTaintsPolicy has a value, transforms to the result of function*/
  def mapNodeTaintsPolicy(f: String => String) : TopologySpreadConstraint = copy(nodeTaintsPolicy = nodeTaintsPolicy.map(f))
}

object TopologySpreadConstraint {

    implicit def encoder[T](implicit builder : Builder[T]) : Encoder[io.k8s.api.core.v1.TopologySpreadConstraint, T] = new Encoder[io.k8s.api.core.v1.TopologySpreadConstraint, T] {
        def apply(o: io.k8s.api.core.v1.TopologySpreadConstraint) : T = {
          val obj = ObjectWriter[T]()
          obj
            .write("maxSkew", o.maxSkew)
            .write("topologyKey", o.topologyKey)
            .write("whenUnsatisfiable", o.whenUnsatisfiable)
            .write("matchLabelKeys", o.matchLabelKeys)
            .write("minDomains", o.minDomains)
            .write("labelSelector", o.labelSelector)
            .write("nodeAffinityPolicy", o.nodeAffinityPolicy)
            .write("nodeTaintsPolicy", o.nodeTaintsPolicy)
            .build
        }
    }

    implicit def decoderOf[T : Reader] : Decoder[T, TopologySpreadConstraint] = new Decoder[T, TopologySpreadConstraint] {
      def apply(t: T): Either[String, TopologySpreadConstraint] = for {
          obj <- ObjectReader(t)
          maxSkew <- obj.read[Int]("maxSkew")
          topologyKey <- obj.read[String]("topologyKey")
          whenUnsatisfiable <- obj.read[String]("whenUnsatisfiable")
          matchLabelKeys <- obj.readOpt[Seq[String]]("matchLabelKeys")
          minDomains <- obj.readOpt[Int]("minDomains")
          labelSelector <- obj.readOpt[io.k8s.apimachinery.pkg.apis.meta.v1.LabelSelector]("labelSelector")
          nodeAffinityPolicy <- obj.readOpt[String]("nodeAffinityPolicy")
          nodeTaintsPolicy <- obj.readOpt[String]("nodeTaintsPolicy")
      } yield TopologySpreadConstraint (
          maxSkew = maxSkew,
          topologyKey = topologyKey,
          whenUnsatisfiable = whenUnsatisfiable,
          matchLabelKeys = matchLabelKeys,
          minDomains = minDomains,
          labelSelector = labelSelector,
          nodeAffinityPolicy = nodeAffinityPolicy,
          nodeTaintsPolicy = nodeTaintsPolicy
        )
    }
}

