package io.k8s.api.core.v1

import dev.hnaderi.k8s.utils._

/** Represents a Rados Block Device mount that lasts the lifetime of a pod. RBD volumes support ownership management and SELinux relabeling. */
final case class RBDVolumeSource(
  image : String,
  monitors : Seq[String],
  readOnly : Option[Boolean] = None,
  fsType : Option[String] = None,
  secretRef : Option[io.k8s.api.core.v1.LocalObjectReference] = None,
  pool : Option[String] = None,
  keyring : Option[String] = None,
  user : Option[String] = None
) {

  /** Returns a new data with image set to new value */
  def withImage(value: String) : RBDVolumeSource = copy(image = value)
  /** transforms image to result of function */
  def mapImage(f: String => String) : RBDVolumeSource = copy(image = f(image))

  /** Returns a new data with monitors set to new value */
  def withMonitors(value: Seq[String]) : RBDVolumeSource = copy(monitors = value)
  /** Appends new values to monitors */
  def addMonitors(newValues: String*) : RBDVolumeSource = copy(monitors = monitors ++ newValues)
  /** transforms monitors to result of function */
  def mapMonitors(f: Seq[String] => Seq[String]) : RBDVolumeSource = copy(monitors = f(monitors))

  /** Returns a new data with readOnly set to new value */
  def withReadOnly(value: Boolean) : RBDVolumeSource = copy(readOnly = Some(value))
  /** if readOnly has a value, transforms to the result of function*/
  def mapReadOnly(f: Boolean => Boolean) : RBDVolumeSource = copy(readOnly = readOnly.map(f))

  /** Returns a new data with fsType set to new value */
  def withFsType(value: String) : RBDVolumeSource = copy(fsType = Some(value))
  /** if fsType has a value, transforms to the result of function*/
  def mapFsType(f: String => String) : RBDVolumeSource = copy(fsType = fsType.map(f))

  /** Returns a new data with secretRef set to new value */
  def withSecretRef(value: io.k8s.api.core.v1.LocalObjectReference) : RBDVolumeSource = copy(secretRef = Some(value))
  /** if secretRef has a value, transforms to the result of function*/
  def mapSecretRef(f: io.k8s.api.core.v1.LocalObjectReference => io.k8s.api.core.v1.LocalObjectReference) : RBDVolumeSource = copy(secretRef = secretRef.map(f))

  /** Returns a new data with pool set to new value */
  def withPool(value: String) : RBDVolumeSource = copy(pool = Some(value))
  /** if pool has a value, transforms to the result of function*/
  def mapPool(f: String => String) : RBDVolumeSource = copy(pool = pool.map(f))

  /** Returns a new data with keyring set to new value */
  def withKeyring(value: String) : RBDVolumeSource = copy(keyring = Some(value))
  /** if keyring has a value, transforms to the result of function*/
  def mapKeyring(f: String => String) : RBDVolumeSource = copy(keyring = keyring.map(f))

  /** Returns a new data with user set to new value */
  def withUser(value: String) : RBDVolumeSource = copy(user = Some(value))
  /** if user has a value, transforms to the result of function*/
  def mapUser(f: String => String) : RBDVolumeSource = copy(user = user.map(f))
}

object RBDVolumeSource {

    implicit def encoder[T](implicit builder : Builder[T]) : Encoder[io.k8s.api.core.v1.RBDVolumeSource, T] = new Encoder[io.k8s.api.core.v1.RBDVolumeSource, T] {
        def apply(o: io.k8s.api.core.v1.RBDVolumeSource) : T = {
          val obj = ObjectWriter[T]()
          obj
            .write("image", o.image)
            .write("monitors", o.monitors)
            .write("readOnly", o.readOnly)
            .write("fsType", o.fsType)
            .write("secretRef", o.secretRef)
            .write("pool", o.pool)
            .write("keyring", o.keyring)
            .write("user", o.user)
            .build
        }
    }

    implicit def decoderOf[T : Reader] : Decoder[T, RBDVolumeSource] = new Decoder[T, RBDVolumeSource] {
      def apply(t: T): Either[String, RBDVolumeSource] = for {
          obj <- ObjectReader(t)
          image <- obj.read[String]("image")
          monitors <- obj.read[Seq[String]]("monitors")
          readOnly <- obj.readOpt[Boolean]("readOnly")
          fsType <- obj.readOpt[String]("fsType")
          secretRef <- obj.readOpt[io.k8s.api.core.v1.LocalObjectReference]("secretRef")
          pool <- obj.readOpt[String]("pool")
          keyring <- obj.readOpt[String]("keyring")
          user <- obj.readOpt[String]("user")
      } yield RBDVolumeSource (
          image = image,
          monitors = monitors,
          readOnly = readOnly,
          fsType = fsType,
          secretRef = secretRef,
          pool = pool,
          keyring = keyring,
          user = user
        )
    }
}

