package io.k8s.api.core.v1

import dev.hnaderi.k8s.utils._

/** PodDNSConfig defines the DNS parameters of a pod in addition to those generated from DNSPolicy. */
final case class PodDNSConfig(
  nameservers : Option[Seq[String]] = None,
  options : Option[Seq[io.k8s.api.core.v1.PodDNSConfigOption]] = None,
  searches : Option[Seq[String]] = None
) {

  /** Returns a new data with nameservers set to new value */
  def withNameservers(value: Seq[String]) : PodDNSConfig = copy(nameservers = Some(value))
  /** Appends new values to nameservers */
  def addNameservers(newValues: String*) : PodDNSConfig = copy(nameservers = Some(nameservers.fold(newValues)(_ ++ newValues)))
  /** if nameservers has a value, transforms to the result of function*/
  def mapNameservers(f: Seq[String] => Seq[String]) : PodDNSConfig = copy(nameservers = nameservers.map(f))

  /** Returns a new data with options set to new value */
  def withOptions(value: Seq[io.k8s.api.core.v1.PodDNSConfigOption]) : PodDNSConfig = copy(options = Some(value))
  /** Appends new values to options */
  def addOptions(newValues: io.k8s.api.core.v1.PodDNSConfigOption*) : PodDNSConfig = copy(options = Some(options.fold(newValues)(_ ++ newValues)))
  /** if options has a value, transforms to the result of function*/
  def mapOptions(f: Seq[io.k8s.api.core.v1.PodDNSConfigOption] => Seq[io.k8s.api.core.v1.PodDNSConfigOption]) : PodDNSConfig = copy(options = options.map(f))

  /** Returns a new data with searches set to new value */
  def withSearches(value: Seq[String]) : PodDNSConfig = copy(searches = Some(value))
  /** Appends new values to searches */
  def addSearches(newValues: String*) : PodDNSConfig = copy(searches = Some(searches.fold(newValues)(_ ++ newValues)))
  /** if searches has a value, transforms to the result of function*/
  def mapSearches(f: Seq[String] => Seq[String]) : PodDNSConfig = copy(searches = searches.map(f))
}

object PodDNSConfig {

    implicit def encoder[T](implicit builder : Builder[T]) : Encoder[io.k8s.api.core.v1.PodDNSConfig, T] = new Encoder[io.k8s.api.core.v1.PodDNSConfig, T] {
        def apply(o: io.k8s.api.core.v1.PodDNSConfig) : T = {
          val obj = ObjectWriter[T]()
          obj
            .write("nameservers", o.nameservers)
            .write("options", o.options)
            .write("searches", o.searches)
            .build
        }
    }

    implicit def decoderOf[T : Reader] : Decoder[T, PodDNSConfig] = new Decoder[T, PodDNSConfig] {
      def apply(t: T): Either[String, PodDNSConfig] = for {
          obj <- ObjectReader(t)
          nameservers <- obj.readOpt[Seq[String]]("nameservers")
          options <- obj.readOpt[Seq[io.k8s.api.core.v1.PodDNSConfigOption]]("options")
          searches <- obj.readOpt[Seq[String]]("searches")
      } yield PodDNSConfig (
          nameservers = nameservers,
          options = options,
          searches = searches
        )
    }
}

