package io.k8s.api.core.v1

import dev.hnaderi.k8s.utils._

/** A node selector requirement is a selector that contains values, a key, and an operator that relates the key and values. */
final case class NodeSelectorRequirement(
  key : String,
  operator : String,
  values : Option[Seq[String]] = None
) {

  /** Returns a new data with key set to new value */
  def withKey(value: String) : NodeSelectorRequirement = copy(key = value)
  /** transforms key to result of function */
  def mapKey(f: String => String) : NodeSelectorRequirement = copy(key = f(key))

  /** Returns a new data with operator set to new value */
  def withOperator(value: String) : NodeSelectorRequirement = copy(operator = value)
  /** transforms operator to result of function */
  def mapOperator(f: String => String) : NodeSelectorRequirement = copy(operator = f(operator))

  /** Returns a new data with values set to new value */
  def withValues(value: Seq[String]) : NodeSelectorRequirement = copy(values = Some(value))
  /** Appends new values to values */
  def addValues(newValues: String*) : NodeSelectorRequirement = copy(values = Some(values.fold(newValues)(_ ++ newValues)))
  /** if values has a value, transforms to the result of function*/
  def mapValues(f: Seq[String] => Seq[String]) : NodeSelectorRequirement = copy(values = values.map(f))
}

object NodeSelectorRequirement {

    implicit def encoder[T](implicit builder : Builder[T]) : Encoder[io.k8s.api.core.v1.NodeSelectorRequirement, T] = new Encoder[io.k8s.api.core.v1.NodeSelectorRequirement, T] {
        def apply(o: io.k8s.api.core.v1.NodeSelectorRequirement) : T = {
          val obj = ObjectWriter[T]()
          obj
            .write("key", o.key)
            .write("operator", o.operator)
            .write("values", o.values)
            .build
        }
    }

    implicit def decoderOf[T : Reader] : Decoder[T, NodeSelectorRequirement] = new Decoder[T, NodeSelectorRequirement] {
      def apply(t: T): Either[String, NodeSelectorRequirement] = for {
          obj <- ObjectReader(t)
          key <- obj.read[String]("key")
          operator <- obj.read[String]("operator")
          values <- obj.readOpt[Seq[String]]("values")
      } yield NodeSelectorRequirement (
          key = key,
          operator = operator,
          values = values
        )
    }
}

