package io.k8s.api.core.v1

import dev.hnaderi.k8s.utils._

/** A single application container that you want to run within a pod. */
final case class Container(
  name : String,
  readinessProbe : Option[io.k8s.api.core.v1.Probe] = None,
  livenessProbe : Option[io.k8s.api.core.v1.Probe] = None,
  lifecycle : Option[io.k8s.api.core.v1.Lifecycle] = None,
  image : Option[String] = None,
  stdinOnce : Option[Boolean] = None,
  ports : Option[Seq[io.k8s.api.core.v1.ContainerPort]] = None,
  startupProbe : Option[io.k8s.api.core.v1.Probe] = None,
  command : Option[Seq[String]] = None,
  terminationMessagePolicy : Option[String] = None,
  stdin : Option[Boolean] = None,
  terminationMessagePath : Option[String] = None,
  resources : Option[io.k8s.api.core.v1.ResourceRequirements] = None,
  envFrom : Option[Seq[io.k8s.api.core.v1.EnvFromSource]] = None,
  volumeDevices : Option[Seq[io.k8s.api.core.v1.VolumeDevice]] = None,
  args : Option[Seq[String]] = None,
  tty : Option[Boolean] = None,
  imagePullPolicy : Option[String] = None,
  workingDir : Option[String] = None,
  volumeMounts : Option[Seq[io.k8s.api.core.v1.VolumeMount]] = None,
  env : Option[Seq[io.k8s.api.core.v1.EnvVar]] = None,
  securityContext : Option[io.k8s.api.core.v1.SecurityContext] = None
) {

  /** Returns a new data with name set to new value */
  def withName(value: String) : Container = copy(name = value)
  /** transforms name to result of function */
  def mapName(f: String => String) : Container = copy(name = f(name))

  /** Returns a new data with readinessProbe set to new value */
  def withReadinessProbe(value: io.k8s.api.core.v1.Probe) : Container = copy(readinessProbe = Some(value))
  /** if readinessProbe has a value, transforms to the result of function*/
  def mapReadinessProbe(f: io.k8s.api.core.v1.Probe => io.k8s.api.core.v1.Probe) : Container = copy(readinessProbe = readinessProbe.map(f))

  /** Returns a new data with livenessProbe set to new value */
  def withLivenessProbe(value: io.k8s.api.core.v1.Probe) : Container = copy(livenessProbe = Some(value))
  /** if livenessProbe has a value, transforms to the result of function*/
  def mapLivenessProbe(f: io.k8s.api.core.v1.Probe => io.k8s.api.core.v1.Probe) : Container = copy(livenessProbe = livenessProbe.map(f))

  /** Returns a new data with lifecycle set to new value */
  def withLifecycle(value: io.k8s.api.core.v1.Lifecycle) : Container = copy(lifecycle = Some(value))
  /** if lifecycle has a value, transforms to the result of function*/
  def mapLifecycle(f: io.k8s.api.core.v1.Lifecycle => io.k8s.api.core.v1.Lifecycle) : Container = copy(lifecycle = lifecycle.map(f))

  /** Returns a new data with image set to new value */
  def withImage(value: String) : Container = copy(image = Some(value))
  /** if image has a value, transforms to the result of function*/
  def mapImage(f: String => String) : Container = copy(image = image.map(f))

  /** Returns a new data with stdinOnce set to new value */
  def withStdinOnce(value: Boolean) : Container = copy(stdinOnce = Some(value))
  /** if stdinOnce has a value, transforms to the result of function*/
  def mapStdinOnce(f: Boolean => Boolean) : Container = copy(stdinOnce = stdinOnce.map(f))

  /** Returns a new data with ports set to new value */
  def withPorts(value: Seq[io.k8s.api.core.v1.ContainerPort]) : Container = copy(ports = Some(value))
  /** Appends new values to ports */
  def addPorts(newValues: io.k8s.api.core.v1.ContainerPort*) : Container = copy(ports = Some(ports.fold(newValues)(_ ++ newValues)))
  /** if ports has a value, transforms to the result of function*/
  def mapPorts(f: Seq[io.k8s.api.core.v1.ContainerPort] => Seq[io.k8s.api.core.v1.ContainerPort]) : Container = copy(ports = ports.map(f))

  /** Returns a new data with startupProbe set to new value */
  def withStartupProbe(value: io.k8s.api.core.v1.Probe) : Container = copy(startupProbe = Some(value))
  /** if startupProbe has a value, transforms to the result of function*/
  def mapStartupProbe(f: io.k8s.api.core.v1.Probe => io.k8s.api.core.v1.Probe) : Container = copy(startupProbe = startupProbe.map(f))

  /** Returns a new data with command set to new value */
  def withCommand(value: Seq[String]) : Container = copy(command = Some(value))
  /** Appends new values to command */
  def addCommand(newValues: String*) : Container = copy(command = Some(command.fold(newValues)(_ ++ newValues)))
  /** if command has a value, transforms to the result of function*/
  def mapCommand(f: Seq[String] => Seq[String]) : Container = copy(command = command.map(f))

  /** Returns a new data with terminationMessagePolicy set to new value */
  def withTerminationMessagePolicy(value: String) : Container = copy(terminationMessagePolicy = Some(value))
  /** if terminationMessagePolicy has a value, transforms to the result of function*/
  def mapTerminationMessagePolicy(f: String => String) : Container = copy(terminationMessagePolicy = terminationMessagePolicy.map(f))

  /** Returns a new data with stdin set to new value */
  def withStdin(value: Boolean) : Container = copy(stdin = Some(value))
  /** if stdin has a value, transforms to the result of function*/
  def mapStdin(f: Boolean => Boolean) : Container = copy(stdin = stdin.map(f))

  /** Returns a new data with terminationMessagePath set to new value */
  def withTerminationMessagePath(value: String) : Container = copy(terminationMessagePath = Some(value))
  /** if terminationMessagePath has a value, transforms to the result of function*/
  def mapTerminationMessagePath(f: String => String) : Container = copy(terminationMessagePath = terminationMessagePath.map(f))

  /** Returns a new data with resources set to new value */
  def withResources(value: io.k8s.api.core.v1.ResourceRequirements) : Container = copy(resources = Some(value))
  /** if resources has a value, transforms to the result of function*/
  def mapResources(f: io.k8s.api.core.v1.ResourceRequirements => io.k8s.api.core.v1.ResourceRequirements) : Container = copy(resources = resources.map(f))

  /** Returns a new data with envFrom set to new value */
  def withEnvFrom(value: Seq[io.k8s.api.core.v1.EnvFromSource]) : Container = copy(envFrom = Some(value))
  /** Appends new values to envFrom */
  def addEnvFrom(newValues: io.k8s.api.core.v1.EnvFromSource*) : Container = copy(envFrom = Some(envFrom.fold(newValues)(_ ++ newValues)))
  /** if envFrom has a value, transforms to the result of function*/
  def mapEnvFrom(f: Seq[io.k8s.api.core.v1.EnvFromSource] => Seq[io.k8s.api.core.v1.EnvFromSource]) : Container = copy(envFrom = envFrom.map(f))

  /** Returns a new data with volumeDevices set to new value */
  def withVolumeDevices(value: Seq[io.k8s.api.core.v1.VolumeDevice]) : Container = copy(volumeDevices = Some(value))
  /** Appends new values to volumeDevices */
  def addVolumeDevices(newValues: io.k8s.api.core.v1.VolumeDevice*) : Container = copy(volumeDevices = Some(volumeDevices.fold(newValues)(_ ++ newValues)))
  /** if volumeDevices has a value, transforms to the result of function*/
  def mapVolumeDevices(f: Seq[io.k8s.api.core.v1.VolumeDevice] => Seq[io.k8s.api.core.v1.VolumeDevice]) : Container = copy(volumeDevices = volumeDevices.map(f))

  /** Returns a new data with args set to new value */
  def withArgs(value: Seq[String]) : Container = copy(args = Some(value))
  /** Appends new values to args */
  def addArgs(newValues: String*) : Container = copy(args = Some(args.fold(newValues)(_ ++ newValues)))
  /** if args has a value, transforms to the result of function*/
  def mapArgs(f: Seq[String] => Seq[String]) : Container = copy(args = args.map(f))

  /** Returns a new data with tty set to new value */
  def withTty(value: Boolean) : Container = copy(tty = Some(value))
  /** if tty has a value, transforms to the result of function*/
  def mapTty(f: Boolean => Boolean) : Container = copy(tty = tty.map(f))

  /** Returns a new data with imagePullPolicy set to new value */
  def withImagePullPolicy(value: String) : Container = copy(imagePullPolicy = Some(value))
  /** if imagePullPolicy has a value, transforms to the result of function*/
  def mapImagePullPolicy(f: String => String) : Container = copy(imagePullPolicy = imagePullPolicy.map(f))

  /** Returns a new data with workingDir set to new value */
  def withWorkingDir(value: String) : Container = copy(workingDir = Some(value))
  /** if workingDir has a value, transforms to the result of function*/
  def mapWorkingDir(f: String => String) : Container = copy(workingDir = workingDir.map(f))

  /** Returns a new data with volumeMounts set to new value */
  def withVolumeMounts(value: Seq[io.k8s.api.core.v1.VolumeMount]) : Container = copy(volumeMounts = Some(value))
  /** Appends new values to volumeMounts */
  def addVolumeMounts(newValues: io.k8s.api.core.v1.VolumeMount*) : Container = copy(volumeMounts = Some(volumeMounts.fold(newValues)(_ ++ newValues)))
  /** if volumeMounts has a value, transforms to the result of function*/
  def mapVolumeMounts(f: Seq[io.k8s.api.core.v1.VolumeMount] => Seq[io.k8s.api.core.v1.VolumeMount]) : Container = copy(volumeMounts = volumeMounts.map(f))

  /** Returns a new data with env set to new value */
  def withEnv(value: Seq[io.k8s.api.core.v1.EnvVar]) : Container = copy(env = Some(value))
  /** Appends new values to env */
  def addEnv(newValues: io.k8s.api.core.v1.EnvVar*) : Container = copy(env = Some(env.fold(newValues)(_ ++ newValues)))
  /** if env has a value, transforms to the result of function*/
  def mapEnv(f: Seq[io.k8s.api.core.v1.EnvVar] => Seq[io.k8s.api.core.v1.EnvVar]) : Container = copy(env = env.map(f))

  /** Returns a new data with securityContext set to new value */
  def withSecurityContext(value: io.k8s.api.core.v1.SecurityContext) : Container = copy(securityContext = Some(value))
  /** if securityContext has a value, transforms to the result of function*/
  def mapSecurityContext(f: io.k8s.api.core.v1.SecurityContext => io.k8s.api.core.v1.SecurityContext) : Container = copy(securityContext = securityContext.map(f))
}

object Container {

    implicit def encoder[T](implicit builder : Builder[T]) : Encoder[io.k8s.api.core.v1.Container, T] = new Encoder[io.k8s.api.core.v1.Container, T] {
        def apply(o: io.k8s.api.core.v1.Container) : T = {
          val obj = ObjectWriter[T]()
          obj
            .write("name", o.name)
            .write("readinessProbe", o.readinessProbe)
            .write("livenessProbe", o.livenessProbe)
            .write("lifecycle", o.lifecycle)
            .write("image", o.image)
            .write("stdinOnce", o.stdinOnce)
            .write("ports", o.ports)
            .write("startupProbe", o.startupProbe)
            .write("command", o.command)
            .write("terminationMessagePolicy", o.terminationMessagePolicy)
            .write("stdin", o.stdin)
            .write("terminationMessagePath", o.terminationMessagePath)
            .write("resources", o.resources)
            .write("envFrom", o.envFrom)
            .write("volumeDevices", o.volumeDevices)
            .write("args", o.args)
            .write("tty", o.tty)
            .write("imagePullPolicy", o.imagePullPolicy)
            .write("workingDir", o.workingDir)
            .write("volumeMounts", o.volumeMounts)
            .write("env", o.env)
            .write("securityContext", o.securityContext)
            .build
        }
    }

    implicit def decoderOf[T : Reader] : Decoder[T, Container] = new Decoder[T, Container] {
      def apply(t: T): Either[String, Container] = for {
          obj <- ObjectReader(t)
          name <- obj.read[String]("name")
          readinessProbe <- obj.readOpt[io.k8s.api.core.v1.Probe]("readinessProbe")
          livenessProbe <- obj.readOpt[io.k8s.api.core.v1.Probe]("livenessProbe")
          lifecycle <- obj.readOpt[io.k8s.api.core.v1.Lifecycle]("lifecycle")
          image <- obj.readOpt[String]("image")
          stdinOnce <- obj.readOpt[Boolean]("stdinOnce")
          ports <- obj.readOpt[Seq[io.k8s.api.core.v1.ContainerPort]]("ports")
          startupProbe <- obj.readOpt[io.k8s.api.core.v1.Probe]("startupProbe")
          command <- obj.readOpt[Seq[String]]("command")
          terminationMessagePolicy <- obj.readOpt[String]("terminationMessagePolicy")
          stdin <- obj.readOpt[Boolean]("stdin")
          terminationMessagePath <- obj.readOpt[String]("terminationMessagePath")
          resources <- obj.readOpt[io.k8s.api.core.v1.ResourceRequirements]("resources")
          envFrom <- obj.readOpt[Seq[io.k8s.api.core.v1.EnvFromSource]]("envFrom")
          volumeDevices <- obj.readOpt[Seq[io.k8s.api.core.v1.VolumeDevice]]("volumeDevices")
          args <- obj.readOpt[Seq[String]]("args")
          tty <- obj.readOpt[Boolean]("tty")
          imagePullPolicy <- obj.readOpt[String]("imagePullPolicy")
          workingDir <- obj.readOpt[String]("workingDir")
          volumeMounts <- obj.readOpt[Seq[io.k8s.api.core.v1.VolumeMount]]("volumeMounts")
          env <- obj.readOpt[Seq[io.k8s.api.core.v1.EnvVar]]("env")
          securityContext <- obj.readOpt[io.k8s.api.core.v1.SecurityContext]("securityContext")
      } yield Container (
          name = name,
          readinessProbe = readinessProbe,
          livenessProbe = livenessProbe,
          lifecycle = lifecycle,
          image = image,
          stdinOnce = stdinOnce,
          ports = ports,
          startupProbe = startupProbe,
          command = command,
          terminationMessagePolicy = terminationMessagePolicy,
          stdin = stdin,
          terminationMessagePath = terminationMessagePath,
          resources = resources,
          envFrom = envFrom,
          volumeDevices = volumeDevices,
          args = args,
          tty = tty,
          imagePullPolicy = imagePullPolicy,
          workingDir = workingDir,
          volumeMounts = volumeMounts,
          env = env,
          securityContext = securityContext
        )
    }
}

