package io.k8s.api.core.v1

import dev.hnaderi.k8s.utils._

/** Represents a cinder volume resource in Openstack. A Cinder volume must exist before mounting to a container. The volume must also be in the same region as the kubelet. Cinder volumes support ownership management and SELinux relabeling. */
final case class CinderPersistentVolumeSource(
  volumeID : String,
  fsType : Option[String] = None,
  readOnly : Option[Boolean] = None,
  secretRef : Option[io.k8s.api.core.v1.SecretReference] = None
) {

  /** Returns a new data with volumeID set to new value */
  def withVolumeID(value: String) : CinderPersistentVolumeSource = copy(volumeID = value)
  /** transforms volumeID to result of function */
  def mapVolumeID(f: String => String) : CinderPersistentVolumeSource = copy(volumeID = f(volumeID))

  /** Returns a new data with fsType set to new value */
  def withFsType(value: String) : CinderPersistentVolumeSource = copy(fsType = Some(value))
  /** if fsType has a value, transforms to the result of function*/
  def mapFsType(f: String => String) : CinderPersistentVolumeSource = copy(fsType = fsType.map(f))

  /** Returns a new data with readOnly set to new value */
  def withReadOnly(value: Boolean) : CinderPersistentVolumeSource = copy(readOnly = Some(value))
  /** if readOnly has a value, transforms to the result of function*/
  def mapReadOnly(f: Boolean => Boolean) : CinderPersistentVolumeSource = copy(readOnly = readOnly.map(f))

  /** Returns a new data with secretRef set to new value */
  def withSecretRef(value: io.k8s.api.core.v1.SecretReference) : CinderPersistentVolumeSource = copy(secretRef = Some(value))
  /** if secretRef has a value, transforms to the result of function*/
  def mapSecretRef(f: io.k8s.api.core.v1.SecretReference => io.k8s.api.core.v1.SecretReference) : CinderPersistentVolumeSource = copy(secretRef = secretRef.map(f))
}

object CinderPersistentVolumeSource {

    implicit def encoder[T](implicit builder : Builder[T]) : Encoder[io.k8s.api.core.v1.CinderPersistentVolumeSource, T] = new Encoder[io.k8s.api.core.v1.CinderPersistentVolumeSource, T] {
        def apply(o: io.k8s.api.core.v1.CinderPersistentVolumeSource) : T = {
          val obj = ObjectWriter[T]()
          obj
            .write("volumeID", o.volumeID)
            .write("fsType", o.fsType)
            .write("readOnly", o.readOnly)
            .write("secretRef", o.secretRef)
            .build
        }
    }

    implicit def decoderOf[T : Reader] : Decoder[T, CinderPersistentVolumeSource] = new Decoder[T, CinderPersistentVolumeSource] {
      def apply(t: T): Either[String, CinderPersistentVolumeSource] = for {
          obj <- ObjectReader(t)
          volumeID <- obj.read[String]("volumeID")
          fsType <- obj.readOpt[String]("fsType")
          readOnly <- obj.readOpt[Boolean]("readOnly")
          secretRef <- obj.readOpt[io.k8s.api.core.v1.SecretReference]("secretRef")
      } yield CinderPersistentVolumeSource (
          volumeID = volumeID,
          fsType = fsType,
          readOnly = readOnly,
          secretRef = secretRef
        )
    }
}

