package io.k8s.api.admissionregistration.v1

import dev.hnaderi.k8s.utils._

/** MatchResources decides whether to run the admission control policy on an object based on whether it meets the match criteria. The exclude rules take precedence over include rules (if a resource matches both, it is excluded) */
final case class MatchResources(
  resourceRules : Option[Seq[io.k8s.api.admissionregistration.v1.NamedRuleWithOperations]] = None,
  matchPolicy : Option[String] = None,
  namespaceSelector : Option[io.k8s.apimachinery.pkg.apis.meta.v1.LabelSelector] = None,
  objectSelector : Option[io.k8s.apimachinery.pkg.apis.meta.v1.LabelSelector] = None,
  excludeResourceRules : Option[Seq[io.k8s.api.admissionregistration.v1.NamedRuleWithOperations]] = None
) {

  /** Returns a new data with resourceRules set to new value */
  def withResourceRules(value: Seq[io.k8s.api.admissionregistration.v1.NamedRuleWithOperations]) : MatchResources = copy(resourceRules = Some(value))
  /** Appends new values to resourceRules */
  def addResourceRules(newValues: io.k8s.api.admissionregistration.v1.NamedRuleWithOperations*) : MatchResources = copy(resourceRules = Some(resourceRules.fold(newValues)(_ ++ newValues)))
  /** if resourceRules has a value, transforms to the result of function*/
  def mapResourceRules(f: Seq[io.k8s.api.admissionregistration.v1.NamedRuleWithOperations] => Seq[io.k8s.api.admissionregistration.v1.NamedRuleWithOperations]) : MatchResources = copy(resourceRules = resourceRules.map(f))

  /** Returns a new data with matchPolicy set to new value */
  def withMatchPolicy(value: String) : MatchResources = copy(matchPolicy = Some(value))
  /** if matchPolicy has a value, transforms to the result of function*/
  def mapMatchPolicy(f: String => String) : MatchResources = copy(matchPolicy = matchPolicy.map(f))

  /** Returns a new data with namespaceSelector set to new value */
  def withNamespaceSelector(value: io.k8s.apimachinery.pkg.apis.meta.v1.LabelSelector) : MatchResources = copy(namespaceSelector = Some(value))
  /** if namespaceSelector has a value, transforms to the result of function*/
  def mapNamespaceSelector(f: io.k8s.apimachinery.pkg.apis.meta.v1.LabelSelector => io.k8s.apimachinery.pkg.apis.meta.v1.LabelSelector) : MatchResources = copy(namespaceSelector = namespaceSelector.map(f))

  /** Returns a new data with objectSelector set to new value */
  def withObjectSelector(value: io.k8s.apimachinery.pkg.apis.meta.v1.LabelSelector) : MatchResources = copy(objectSelector = Some(value))
  /** if objectSelector has a value, transforms to the result of function*/
  def mapObjectSelector(f: io.k8s.apimachinery.pkg.apis.meta.v1.LabelSelector => io.k8s.apimachinery.pkg.apis.meta.v1.LabelSelector) : MatchResources = copy(objectSelector = objectSelector.map(f))

  /** Returns a new data with excludeResourceRules set to new value */
  def withExcludeResourceRules(value: Seq[io.k8s.api.admissionregistration.v1.NamedRuleWithOperations]) : MatchResources = copy(excludeResourceRules = Some(value))
  /** Appends new values to excludeResourceRules */
  def addExcludeResourceRules(newValues: io.k8s.api.admissionregistration.v1.NamedRuleWithOperations*) : MatchResources = copy(excludeResourceRules = Some(excludeResourceRules.fold(newValues)(_ ++ newValues)))
  /** if excludeResourceRules has a value, transforms to the result of function*/
  def mapExcludeResourceRules(f: Seq[io.k8s.api.admissionregistration.v1.NamedRuleWithOperations] => Seq[io.k8s.api.admissionregistration.v1.NamedRuleWithOperations]) : MatchResources = copy(excludeResourceRules = excludeResourceRules.map(f))
}

object MatchResources {

    implicit val encoder : Encoder[io.k8s.api.admissionregistration.v1.MatchResources] = new Encoder[io.k8s.api.admissionregistration.v1.MatchResources] {
        def apply[T : Builder](o: io.k8s.api.admissionregistration.v1.MatchResources) : T = {
          val obj = ObjectWriter[T]()
          obj
            .write("resourceRules", o.resourceRules)
            .write("matchPolicy", o.matchPolicy)
            .write("namespaceSelector", o.namespaceSelector)
            .write("objectSelector", o.objectSelector)
            .write("excludeResourceRules", o.excludeResourceRules)
            .build
        }
    }

    implicit val decoder: Decoder[MatchResources] = new Decoder[MatchResources] {
      def apply[T : Reader](t: T): Either[String, MatchResources] = for {
          obj <- ObjectReader(t)
          resourceRules <- obj.readOpt[Seq[io.k8s.api.admissionregistration.v1.NamedRuleWithOperations]]("resourceRules")
          matchPolicy <- obj.readOpt[String]("matchPolicy")
          namespaceSelector <- obj.readOpt[io.k8s.apimachinery.pkg.apis.meta.v1.LabelSelector]("namespaceSelector")
          objectSelector <- obj.readOpt[io.k8s.apimachinery.pkg.apis.meta.v1.LabelSelector]("objectSelector")
          excludeResourceRules <- obj.readOpt[Seq[io.k8s.api.admissionregistration.v1.NamedRuleWithOperations]]("excludeResourceRules")
      } yield MatchResources (
          resourceRules = resourceRules,
          matchPolicy = matchPolicy,
          namespaceSelector = namespaceSelector,
          objectSelector = objectSelector,
          excludeResourceRules = excludeResourceRules
        )
    }
}

