package io.k8s.api.authorization.v1

import dev.hnaderi.k8s.utils._

/** SubjectRulesReviewStatus contains the result of a rules check. This check can be incomplete depending on the set of authorizers the server is configured with and any errors experienced during evaluation. Because authorization rules are additive, if a rule appears in a list it's safe to assume the subject has that permission, even if that list is incomplete. */
final case class SubjectRulesReviewStatus(
  incomplete : Boolean,
  nonResourceRules : Seq[io.k8s.api.authorization.v1.NonResourceRule],
  resourceRules : Seq[io.k8s.api.authorization.v1.ResourceRule],
  evaluationError : Option[String] = None
) {

  /** Returns a new data with incomplete set to new value */
  def withIncomplete(value: Boolean) : SubjectRulesReviewStatus = copy(incomplete = value)
  /** transforms incomplete to result of function */
  def mapIncomplete(f: Boolean => Boolean) : SubjectRulesReviewStatus = copy(incomplete = f(incomplete))

  /** Returns a new data with nonResourceRules set to new value */
  def withNonResourceRules(value: Seq[io.k8s.api.authorization.v1.NonResourceRule]) : SubjectRulesReviewStatus = copy(nonResourceRules = value)
  /** Appends new values to nonResourceRules */
  def addNonResourceRules(newValues: io.k8s.api.authorization.v1.NonResourceRule*) : SubjectRulesReviewStatus = copy(nonResourceRules = nonResourceRules ++ newValues)
  /** transforms nonResourceRules to result of function */
  def mapNonResourceRules(f: Seq[io.k8s.api.authorization.v1.NonResourceRule] => Seq[io.k8s.api.authorization.v1.NonResourceRule]) : SubjectRulesReviewStatus = copy(nonResourceRules = f(nonResourceRules))

  /** Returns a new data with resourceRules set to new value */
  def withResourceRules(value: Seq[io.k8s.api.authorization.v1.ResourceRule]) : SubjectRulesReviewStatus = copy(resourceRules = value)
  /** Appends new values to resourceRules */
  def addResourceRules(newValues: io.k8s.api.authorization.v1.ResourceRule*) : SubjectRulesReviewStatus = copy(resourceRules = resourceRules ++ newValues)
  /** transforms resourceRules to result of function */
  def mapResourceRules(f: Seq[io.k8s.api.authorization.v1.ResourceRule] => Seq[io.k8s.api.authorization.v1.ResourceRule]) : SubjectRulesReviewStatus = copy(resourceRules = f(resourceRules))

  /** Returns a new data with evaluationError set to new value */
  def withEvaluationError(value: String) : SubjectRulesReviewStatus = copy(evaluationError = Some(value))
  /** if evaluationError has a value, transforms to the result of function*/
  def mapEvaluationError(f: String => String) : SubjectRulesReviewStatus = copy(evaluationError = evaluationError.map(f))
}

object SubjectRulesReviewStatus {

    implicit val encoder : Encoder[io.k8s.api.authorization.v1.SubjectRulesReviewStatus] = new Encoder[io.k8s.api.authorization.v1.SubjectRulesReviewStatus] {
        def apply[T : Builder](o: io.k8s.api.authorization.v1.SubjectRulesReviewStatus) : T = {
          val obj = ObjectWriter[T]()
          obj
            .write("incomplete", o.incomplete)
            .write("nonResourceRules", o.nonResourceRules)
            .write("resourceRules", o.resourceRules)
            .write("evaluationError", o.evaluationError)
            .build
        }
    }

    implicit val decoder: Decoder[SubjectRulesReviewStatus] = new Decoder[SubjectRulesReviewStatus] {
      def apply[T : Reader](t: T): Either[String, SubjectRulesReviewStatus] = for {
          obj <- ObjectReader(t)
          incomplete <- obj.read[Boolean]("incomplete")
          nonResourceRules <- obj.read[Seq[io.k8s.api.authorization.v1.NonResourceRule]]("nonResourceRules")
          resourceRules <- obj.read[Seq[io.k8s.api.authorization.v1.ResourceRule]]("resourceRules")
          evaluationError <- obj.readOpt[String]("evaluationError")
      } yield SubjectRulesReviewStatus (
          incomplete = incomplete,
          nonResourceRules = nonResourceRules,
          resourceRules = resourceRules,
          evaluationError = evaluationError
        )
    }
}

