package io.k8s.api.core.v1


/** Represents a Quobyte mount that lasts the lifetime of a pod. Quobyte volumes do not support ownership management or SELinux relabeling. */
final case class QuobyteVolumeSource(
  registry : String,
  volume : String,
  readOnly : Option[Boolean] = None,
  tenant : Option[String] = None,
  group : Option[String] = None,
  user : Option[String] = None
) {

  /** Returns a new data with registry set to new value */
  def withRegistry(value: String) : QuobyteVolumeSource = copy(registry = value)
  /** transforms registry to result of function */
  def mapRegistry(f: String => String) : QuobyteVolumeSource = copy(registry = f(registry))

  /** Returns a new data with volume set to new value */
  def withVolume(value: String) : QuobyteVolumeSource = copy(volume = value)
  /** transforms volume to result of function */
  def mapVolume(f: String => String) : QuobyteVolumeSource = copy(volume = f(volume))

  /** Returns a new data with readOnly set to new value */
  def withReadOnly(value: Boolean) : QuobyteVolumeSource = copy(readOnly = Some(value))
  /** if readOnly has a value, transforms to the result of function*/
  def mapReadOnly(f: Boolean => Boolean) : QuobyteVolumeSource = copy(readOnly = readOnly.map(f))

  /** Returns a new data with tenant set to new value */
  def withTenant(value: String) : QuobyteVolumeSource = copy(tenant = Some(value))
  /** if tenant has a value, transforms to the result of function*/
  def mapTenant(f: String => String) : QuobyteVolumeSource = copy(tenant = tenant.map(f))

  /** Returns a new data with group set to new value */
  def withGroup(value: String) : QuobyteVolumeSource = copy(group = Some(value))
  /** if group has a value, transforms to the result of function*/
  def mapGroup(f: String => String) : QuobyteVolumeSource = copy(group = group.map(f))

  /** Returns a new data with user set to new value */
  def withUser(value: String) : QuobyteVolumeSource = copy(user = Some(value))
  /** if user has a value, transforms to the result of function*/
  def mapUser(f: String => String) : QuobyteVolumeSource = copy(user = user.map(f))
}

