package io.k8s.api.core.v1


/** Adapts a ConfigMap into a volume.

The contents of the target ConfigMap's Data field will be presented in a volume as files using the keys in the Data field as the file names, unless the items element is populated with specific mappings of keys to paths. ConfigMap volumes support ownership management and SELinux relabeling. */
final case class ConfigMapVolumeSource(
  defaultMode : Option[Int] = None,
  items : Option[Seq[io.k8s.api.core.v1.KeyToPath]] = None,
  name : Option[String] = None,
  optional : Option[Boolean] = None
) {

  /** Returns a new data with defaultMode set to new value */
  def withDefaultMode(value: Int) : ConfigMapVolumeSource = copy(defaultMode = Some(value))
  /** if defaultMode has a value, transforms to the result of function*/
  def mapDefaultMode(f: Int => Int) : ConfigMapVolumeSource = copy(defaultMode = defaultMode.map(f))

  /** Returns a new data with items set to new value */
  def withItems(value: Seq[io.k8s.api.core.v1.KeyToPath]) : ConfigMapVolumeSource = copy(items = Some(value))
  /** Appends new values to items */
  def addItems(newValues: io.k8s.api.core.v1.KeyToPath*) : ConfigMapVolumeSource = copy(items = Some(items.fold(newValues)(_ ++ newValues)))
  /** if items has a value, transforms to the result of function*/
  def mapItems(f: Seq[io.k8s.api.core.v1.KeyToPath] => Seq[io.k8s.api.core.v1.KeyToPath]) : ConfigMapVolumeSource = copy(items = items.map(f))

  /** Returns a new data with name set to new value */
  def withName(value: String) : ConfigMapVolumeSource = copy(name = Some(value))
  /** if name has a value, transforms to the result of function*/
  def mapName(f: String => String) : ConfigMapVolumeSource = copy(name = name.map(f))

  /** Returns a new data with optional set to new value */
  def withOptional(value: Boolean) : ConfigMapVolumeSource = copy(optional = Some(value))
  /** if optional has a value, transforms to the result of function*/
  def mapOptional(f: Boolean => Boolean) : ConfigMapVolumeSource = copy(optional = optional.map(f))
}

