package io.k8s.api.apps.v1

import dev.hnaderi.k8s.utils._

/** ReplicaSetStatus represents the current status of a ReplicaSet. */
final case class ReplicaSetStatus(
  replicas : Int,
  conditions : Option[Seq[io.k8s.api.apps.v1.ReplicaSetCondition]] = None,
  availableReplicas : Option[Int] = None,
  observedGeneration : Option[Long] = None,
  readyReplicas : Option[Int] = None,
  fullyLabeledReplicas : Option[Int] = None
) {

  /** Returns a new data with replicas set to new value */
  def withReplicas(value: Int) : ReplicaSetStatus = copy(replicas = value)
  /** transforms replicas to result of function */
  def mapReplicas(f: Int => Int) : ReplicaSetStatus = copy(replicas = f(replicas))

  /** Returns a new data with conditions set to new value */
  def withConditions(value: Seq[io.k8s.api.apps.v1.ReplicaSetCondition]) : ReplicaSetStatus = copy(conditions = Some(value))
  /** Appends new values to conditions */
  def addConditions(newValues: io.k8s.api.apps.v1.ReplicaSetCondition*) : ReplicaSetStatus = copy(conditions = Some(conditions.fold(newValues)(_ ++ newValues)))
  /** if conditions has a value, transforms to the result of function*/
  def mapConditions(f: Seq[io.k8s.api.apps.v1.ReplicaSetCondition] => Seq[io.k8s.api.apps.v1.ReplicaSetCondition]) : ReplicaSetStatus = copy(conditions = conditions.map(f))

  /** Returns a new data with availableReplicas set to new value */
  def withAvailableReplicas(value: Int) : ReplicaSetStatus = copy(availableReplicas = Some(value))
  /** if availableReplicas has a value, transforms to the result of function*/
  def mapAvailableReplicas(f: Int => Int) : ReplicaSetStatus = copy(availableReplicas = availableReplicas.map(f))

  /** Returns a new data with observedGeneration set to new value */
  def withObservedGeneration(value: Long) : ReplicaSetStatus = copy(observedGeneration = Some(value))
  /** if observedGeneration has a value, transforms to the result of function*/
  def mapObservedGeneration(f: Long => Long) : ReplicaSetStatus = copy(observedGeneration = observedGeneration.map(f))

  /** Returns a new data with readyReplicas set to new value */
  def withReadyReplicas(value: Int) : ReplicaSetStatus = copy(readyReplicas = Some(value))
  /** if readyReplicas has a value, transforms to the result of function*/
  def mapReadyReplicas(f: Int => Int) : ReplicaSetStatus = copy(readyReplicas = readyReplicas.map(f))

  /** Returns a new data with fullyLabeledReplicas set to new value */
  def withFullyLabeledReplicas(value: Int) : ReplicaSetStatus = copy(fullyLabeledReplicas = Some(value))
  /** if fullyLabeledReplicas has a value, transforms to the result of function*/
  def mapFullyLabeledReplicas(f: Int => Int) : ReplicaSetStatus = copy(fullyLabeledReplicas = fullyLabeledReplicas.map(f))
}

object ReplicaSetStatus {

    implicit val encoder : Encoder[io.k8s.api.apps.v1.ReplicaSetStatus] = new Encoder[io.k8s.api.apps.v1.ReplicaSetStatus] {
        def apply[T : Builder](o: io.k8s.api.apps.v1.ReplicaSetStatus) : T = {
          val obj = ObjectWriter[T]()
          obj
            .write("replicas", o.replicas)
            .write("conditions", o.conditions)
            .write("availableReplicas", o.availableReplicas)
            .write("observedGeneration", o.observedGeneration)
            .write("readyReplicas", o.readyReplicas)
            .write("fullyLabeledReplicas", o.fullyLabeledReplicas)
            .build
        }
    }

    implicit val decoder: Decoder[ReplicaSetStatus] = new Decoder[ReplicaSetStatus] {
      def apply[T : Reader](t: T): Either[String, ReplicaSetStatus] = for {
          obj <- ObjectReader(t)
          replicas <- obj.read[Int]("replicas")
          conditions <- obj.readOpt[Seq[io.k8s.api.apps.v1.ReplicaSetCondition]]("conditions")
          availableReplicas <- obj.readOpt[Int]("availableReplicas")
          observedGeneration <- obj.readOpt[Long]("observedGeneration")
          readyReplicas <- obj.readOpt[Int]("readyReplicas")
          fullyLabeledReplicas <- obj.readOpt[Int]("fullyLabeledReplicas")
      } yield ReplicaSetStatus (
          replicas = replicas,
          conditions = conditions,
          availableReplicas = availableReplicas,
          observedGeneration = observedGeneration,
          readyReplicas = readyReplicas,
          fullyLabeledReplicas = fullyLabeledReplicas
        )
    }
}

