package io.k8s.api.core.v1

import dev.hnaderi.k8s.utils._

/** Defines a set of pods (namely those matching the labelSelector relative to the given namespace(s)) that this pod should be co-located (affinity) or not co-located (anti-affinity) with, where co-located is defined as running on a node whose value of the label with key <topologyKey> matches that of any node on which a pod of the set of pods is running */
final case class PodAffinityTerm(
  topologyKey : String,
  matchLabelKeys : Option[Seq[String]] = None,
  labelSelector : Option[io.k8s.apimachinery.pkg.apis.meta.v1.LabelSelector] = None,
  namespaceSelector : Option[io.k8s.apimachinery.pkg.apis.meta.v1.LabelSelector] = None,
  mismatchLabelKeys : Option[Seq[String]] = None,
  namespaces : Option[Seq[String]] = None
) {

  /** Returns a new data with topologyKey set to new value */
  def withTopologyKey(value: String) : PodAffinityTerm = copy(topologyKey = value)
  /** transforms topologyKey to result of function */
  def mapTopologyKey(f: String => String) : PodAffinityTerm = copy(topologyKey = f(topologyKey))

  /** Returns a new data with matchLabelKeys set to new value */
  def withMatchLabelKeys(value: Seq[String]) : PodAffinityTerm = copy(matchLabelKeys = Some(value))
  /** Appends new values to matchLabelKeys */
  def addMatchLabelKeys(newValues: String*) : PodAffinityTerm = copy(matchLabelKeys = Some(matchLabelKeys.fold(newValues)(_ ++ newValues)))
  /** if matchLabelKeys has a value, transforms to the result of function*/
  def mapMatchLabelKeys(f: Seq[String] => Seq[String]) : PodAffinityTerm = copy(matchLabelKeys = matchLabelKeys.map(f))

  /** Returns a new data with labelSelector set to new value */
  def withLabelSelector(value: io.k8s.apimachinery.pkg.apis.meta.v1.LabelSelector) : PodAffinityTerm = copy(labelSelector = Some(value))
  /** if labelSelector has a value, transforms to the result of function*/
  def mapLabelSelector(f: io.k8s.apimachinery.pkg.apis.meta.v1.LabelSelector => io.k8s.apimachinery.pkg.apis.meta.v1.LabelSelector) : PodAffinityTerm = copy(labelSelector = labelSelector.map(f))

  /** Returns a new data with namespaceSelector set to new value */
  def withNamespaceSelector(value: io.k8s.apimachinery.pkg.apis.meta.v1.LabelSelector) : PodAffinityTerm = copy(namespaceSelector = Some(value))
  /** if namespaceSelector has a value, transforms to the result of function*/
  def mapNamespaceSelector(f: io.k8s.apimachinery.pkg.apis.meta.v1.LabelSelector => io.k8s.apimachinery.pkg.apis.meta.v1.LabelSelector) : PodAffinityTerm = copy(namespaceSelector = namespaceSelector.map(f))

  /** Returns a new data with mismatchLabelKeys set to new value */
  def withMismatchLabelKeys(value: Seq[String]) : PodAffinityTerm = copy(mismatchLabelKeys = Some(value))
  /** Appends new values to mismatchLabelKeys */
  def addMismatchLabelKeys(newValues: String*) : PodAffinityTerm = copy(mismatchLabelKeys = Some(mismatchLabelKeys.fold(newValues)(_ ++ newValues)))
  /** if mismatchLabelKeys has a value, transforms to the result of function*/
  def mapMismatchLabelKeys(f: Seq[String] => Seq[String]) : PodAffinityTerm = copy(mismatchLabelKeys = mismatchLabelKeys.map(f))

  /** Returns a new data with namespaces set to new value */
  def withNamespaces(value: Seq[String]) : PodAffinityTerm = copy(namespaces = Some(value))
  /** Appends new values to namespaces */
  def addNamespaces(newValues: String*) : PodAffinityTerm = copy(namespaces = Some(namespaces.fold(newValues)(_ ++ newValues)))
  /** if namespaces has a value, transforms to the result of function*/
  def mapNamespaces(f: Seq[String] => Seq[String]) : PodAffinityTerm = copy(namespaces = namespaces.map(f))
}

object PodAffinityTerm {

    implicit val encoder : Encoder[io.k8s.api.core.v1.PodAffinityTerm] = new Encoder[io.k8s.api.core.v1.PodAffinityTerm] {
        def apply[T : Builder](o: io.k8s.api.core.v1.PodAffinityTerm) : T = {
          val obj = ObjectWriter[T]()
          obj
            .write("topologyKey", o.topologyKey)
            .write("matchLabelKeys", o.matchLabelKeys)
            .write("labelSelector", o.labelSelector)
            .write("namespaceSelector", o.namespaceSelector)
            .write("mismatchLabelKeys", o.mismatchLabelKeys)
            .write("namespaces", o.namespaces)
            .build
        }
    }

    implicit val decoder: Decoder[PodAffinityTerm] = new Decoder[PodAffinityTerm] {
      def apply[T : Reader](t: T): Either[String, PodAffinityTerm] = for {
          obj <- ObjectReader(t)
          topologyKey <- obj.read[String]("topologyKey")
          matchLabelKeys <- obj.readOpt[Seq[String]]("matchLabelKeys")
          labelSelector <- obj.readOpt[io.k8s.apimachinery.pkg.apis.meta.v1.LabelSelector]("labelSelector")
          namespaceSelector <- obj.readOpt[io.k8s.apimachinery.pkg.apis.meta.v1.LabelSelector]("namespaceSelector")
          mismatchLabelKeys <- obj.readOpt[Seq[String]]("mismatchLabelKeys")
          namespaces <- obj.readOpt[Seq[String]]("namespaces")
      } yield PodAffinityTerm (
          topologyKey = topologyKey,
          matchLabelKeys = matchLabelKeys,
          labelSelector = labelSelector,
          namespaceSelector = namespaceSelector,
          mismatchLabelKeys = mismatchLabelKeys,
          namespaces = namespaces
        )
    }
}

