package io.k8s.api.storage.v1

import dev.hnaderi.k8s._

/** StorageClass describes the parameters for a class of storage for which PersistentVolumes can be dynamically provisioned.

StorageClasses are non-namespaced; the name of the storage class according to etcd is in ObjectMeta.Name. */
final case class StorageClass(
  provisioner : String,
  mountOptions : Option[Seq[String]] = None,
  reclaimPolicy : Option[String] = None,
  volumeBindingMode : Option[String] = None,
  metadata : Option[io.k8s.apimachinery.pkg.apis.meta.v1.ObjectMeta] = None,
  allowVolumeExpansion : Option[Boolean] = None,
  allowedTopologies : Option[Seq[io.k8s.api.core.v1.TopologySelectorTerm]] = None,
  parameters : Option[Map[String, String]] = None
) extends KObject {
  protected val _resourceKind = ResourceKind("storage.k8s.io", "StorageClass", "v1")


  /** Returns a new data with provisioner set to new value */
  def withProvisioner(value: String) : StorageClass = copy(provisioner = value)
  /** transforms provisioner to result of function */
  def mapProvisioner(f: String => String) : StorageClass = copy(provisioner = f(provisioner))

  /** Returns a new data with mountOptions set to new value */
  def withMountOptions(value: Seq[String]) : StorageClass = copy(mountOptions = Some(value))
  /** Appends new values to mountOptions */
  def addMountOptions(newValues: String*) : StorageClass = copy(mountOptions = Some(mountOptions.fold(newValues)(_ ++ newValues)))
  /** if mountOptions has a value, transforms to the result of function*/
  def mapMountOptions(f: Seq[String] => Seq[String]) : StorageClass = copy(mountOptions = mountOptions.map(f))

  /** Returns a new data with reclaimPolicy set to new value */
  def withReclaimPolicy(value: String) : StorageClass = copy(reclaimPolicy = Some(value))
  /** if reclaimPolicy has a value, transforms to the result of function*/
  def mapReclaimPolicy(f: String => String) : StorageClass = copy(reclaimPolicy = reclaimPolicy.map(f))

  /** Returns a new data with volumeBindingMode set to new value */
  def withVolumeBindingMode(value: String) : StorageClass = copy(volumeBindingMode = Some(value))
  /** if volumeBindingMode has a value, transforms to the result of function*/
  def mapVolumeBindingMode(f: String => String) : StorageClass = copy(volumeBindingMode = volumeBindingMode.map(f))

  /** Returns a new data with metadata set to new value */
  def withMetadata(value: io.k8s.apimachinery.pkg.apis.meta.v1.ObjectMeta) : StorageClass = copy(metadata = Some(value))
  /** if metadata has a value, transforms to the result of function*/
  def mapMetadata(f: io.k8s.apimachinery.pkg.apis.meta.v1.ObjectMeta => io.k8s.apimachinery.pkg.apis.meta.v1.ObjectMeta) : StorageClass = copy(metadata = metadata.map(f))

  /** Returns a new data with allowVolumeExpansion set to new value */
  def withAllowVolumeExpansion(value: Boolean) : StorageClass = copy(allowVolumeExpansion = Some(value))
  /** if allowVolumeExpansion has a value, transforms to the result of function*/
  def mapAllowVolumeExpansion(f: Boolean => Boolean) : StorageClass = copy(allowVolumeExpansion = allowVolumeExpansion.map(f))

  /** Returns a new data with allowedTopologies set to new value */
  def withAllowedTopologies(value: Seq[io.k8s.api.core.v1.TopologySelectorTerm]) : StorageClass = copy(allowedTopologies = Some(value))
  /** Appends new values to allowedTopologies */
  def addAllowedTopologies(newValues: io.k8s.api.core.v1.TopologySelectorTerm*) : StorageClass = copy(allowedTopologies = Some(allowedTopologies.fold(newValues)(_ ++ newValues)))
  /** if allowedTopologies has a value, transforms to the result of function*/
  def mapAllowedTopologies(f: Seq[io.k8s.api.core.v1.TopologySelectorTerm] => Seq[io.k8s.api.core.v1.TopologySelectorTerm]) : StorageClass = copy(allowedTopologies = allowedTopologies.map(f))

  /** Returns a new data with parameters set to new value */
  def withParameters(value: Map[String, String]) : StorageClass = copy(parameters = Some(value))
  /** Adds new values to parameters */
  def addParameters(newValues: (String, String)*) : StorageClass = copy(parameters = Some(parameters.fold(newValues.toMap)(_ ++ newValues)))
  /** if parameters has a value, transforms to the result of function*/
  def mapParameters(f: Map[String, String] => Map[String, String]) : StorageClass = copy(parameters = parameters.map(f))
}

