package io.k8s.api.core.v1


/** Represents a Persistent Disk resource in AWS.

An AWS EBS disk must exist before mounting to a container. The disk must also be in the same AWS zone as the kubelet. An AWS EBS disk can only be mounted as read/write once. AWS EBS volumes support ownership management and SELinux relabeling. */
final case class AWSElasticBlockStoreVolumeSource(
  volumeID : String,
  fsType : Option[String] = None,
  partition : Option[Int] = None,
  readOnly : Option[Boolean] = None
) {

  /** Returns a new data with volumeID set to new value */
  def withVolumeID(value: String) : AWSElasticBlockStoreVolumeSource = copy(volumeID = value)
  /** transforms volumeID to result of function */
  def mapVolumeID(f: String => String) : AWSElasticBlockStoreVolumeSource = copy(volumeID = f(volumeID))

  /** Returns a new data with fsType set to new value */
  def withFsType(value: String) : AWSElasticBlockStoreVolumeSource = copy(fsType = Some(value))
  /** if fsType has a value, transforms to the result of function*/
  def mapFsType(f: String => String) : AWSElasticBlockStoreVolumeSource = copy(fsType = fsType.map(f))

  /** Returns a new data with partition set to new value */
  def withPartition(value: Int) : AWSElasticBlockStoreVolumeSource = copy(partition = Some(value))
  /** if partition has a value, transforms to the result of function*/
  def mapPartition(f: Int => Int) : AWSElasticBlockStoreVolumeSource = copy(partition = partition.map(f))

  /** Returns a new data with readOnly set to new value */
  def withReadOnly(value: Boolean) : AWSElasticBlockStoreVolumeSource = copy(readOnly = Some(value))
  /** if readOnly has a value, transforms to the result of function*/
  def mapReadOnly(f: Boolean => Boolean) : AWSElasticBlockStoreVolumeSource = copy(readOnly = readOnly.map(f))
}

