package io.k8s.api.core.v1


/** EndpointSubset is a group of addresses with a common set of ports. The expanded set of endpoints is the Cartesian product of Addresses x Ports. For example, given:

	{
	  Addresses: [{"ip": "10.10.1.1"}, {"ip": "10.10.2.2"}],
	  Ports:     [{"name": "a", "port": 8675}, {"name": "b", "port": 309}]
	}

The resulting set of endpoints can be viewed as:

	a: [ 10.10.1.1:8675, 10.10.2.2:8675 ],
	b: [ 10.10.1.1:309, 10.10.2.2:309 ] */
final case class EndpointSubset(
  addresses : Option[Seq[io.k8s.api.core.v1.EndpointAddress]] = None,
  notReadyAddresses : Option[Seq[io.k8s.api.core.v1.EndpointAddress]] = None,
  ports : Option[Seq[io.k8s.api.core.v1.EndpointPort]] = None
) {
  def withAddresses(value: Seq[io.k8s.api.core.v1.EndpointAddress]) : EndpointSubset = copy(addresses = Some(value))
  def addAddresses(newValues: io.k8s.api.core.v1.EndpointAddress*) : EndpointSubset = copy(addresses = Some(addresses.fold(newValues)(_ ++ newValues)))

  def withNotReadyAddresses(value: Seq[io.k8s.api.core.v1.EndpointAddress]) : EndpointSubset = copy(notReadyAddresses = Some(value))
  def addNotReadyAddresses(newValues: io.k8s.api.core.v1.EndpointAddress*) : EndpointSubset = copy(notReadyAddresses = Some(notReadyAddresses.fold(newValues)(_ ++ newValues)))

  def withPorts(value: Seq[io.k8s.api.core.v1.EndpointPort]) : EndpointSubset = copy(ports = Some(value))
  def addPorts(newValues: io.k8s.api.core.v1.EndpointPort*) : EndpointSubset = copy(ports = Some(ports.fold(newValues)(_ ++ newValues)))

}

