package io.k8s.api.autoscaling.v2

import dev.hnaderi.k8s.utils._

/** CrossVersionObjectReference contains enough information to let you identify the referred resource. */
final case class CrossVersionObjectReference(
  kind : String,
  name : String,
  apiVersion : Option[String] = None
) {

  /** Returns a new data with kind set to new value */
  def withKind(value: String) : CrossVersionObjectReference = copy(kind = value)
  /** transforms kind to result of function */
  def mapKind(f: String => String) : CrossVersionObjectReference = copy(kind = f(kind))

  /** Returns a new data with name set to new value */
  def withName(value: String) : CrossVersionObjectReference = copy(name = value)
  /** transforms name to result of function */
  def mapName(f: String => String) : CrossVersionObjectReference = copy(name = f(name))

  /** Returns a new data with apiVersion set to new value */
  def withApiVersion(value: String) : CrossVersionObjectReference = copy(apiVersion = Some(value))
  /** if apiVersion has a value, transforms to the result of function*/
  def mapApiVersion(f: String => String) : CrossVersionObjectReference = copy(apiVersion = apiVersion.map(f))
}

object CrossVersionObjectReference {

    implicit val encoder : Encoder[io.k8s.api.autoscaling.v2.CrossVersionObjectReference] = new Encoder[io.k8s.api.autoscaling.v2.CrossVersionObjectReference] {
        def apply[T : Builder](o: io.k8s.api.autoscaling.v2.CrossVersionObjectReference) : T = {
          val obj = ObjectWriter[T]()
          obj
            .write("kind", o.kind)
            .write("name", o.name)
            .write("apiVersion", o.apiVersion)
            .build
        }
    }

    implicit val decoder: Decoder[CrossVersionObjectReference] = new Decoder[CrossVersionObjectReference] {
      def apply[T : Reader](t: T): Either[String, CrossVersionObjectReference] = for {
          obj <- ObjectReader(t)
          kind <- obj.read[String]("kind")
          name <- obj.read[String]("name")
          apiVersion <- obj.readOpt[String]("apiVersion")
      } yield CrossVersionObjectReference (
          kind = kind,
          name = name,
          apiVersion = apiVersion
        )
    }
}

