package io.k8s.apiextensions_apiserver.pkg.apis.apiextensions.v1

import dev.hnaderi.k8s.utils._

/** CustomResourceDefinitionNames indicates the names to serve this CustomResourceDefinition */
final case class CustomResourceDefinitionNames(
  plural : String,
  singular : Option[String] = None,
  listKind : Option[String] = None,
  categories : Option[Seq[String]] = None,
  shortNames : Option[Seq[String]] = None
) {

  /** Returns a new data with plural set to new value */
  def withPlural(value: String) : CustomResourceDefinitionNames = copy(plural = value)
  /** transforms plural to result of function */
  def mapPlural(f: String => String) : CustomResourceDefinitionNames = copy(plural = f(plural))

  /** Returns a new data with singular set to new value */
  def withSingular(value: String) : CustomResourceDefinitionNames = copy(singular = Some(value))
  /** if singular has a value, transforms to the result of function*/
  def mapSingular(f: String => String) : CustomResourceDefinitionNames = copy(singular = singular.map(f))

  /** Returns a new data with listKind set to new value */
  def withListKind(value: String) : CustomResourceDefinitionNames = copy(listKind = Some(value))
  /** if listKind has a value, transforms to the result of function*/
  def mapListKind(f: String => String) : CustomResourceDefinitionNames = copy(listKind = listKind.map(f))

  /** Returns a new data with categories set to new value */
  def withCategories(value: Seq[String]) : CustomResourceDefinitionNames = copy(categories = Some(value))
  /** Appends new values to categories */
  def addCategories(newValues: String*) : CustomResourceDefinitionNames = copy(categories = Some(categories.fold(newValues)(_ ++ newValues)))
  /** if categories has a value, transforms to the result of function*/
  def mapCategories(f: Seq[String] => Seq[String]) : CustomResourceDefinitionNames = copy(categories = categories.map(f))

  /** Returns a new data with shortNames set to new value */
  def withShortNames(value: Seq[String]) : CustomResourceDefinitionNames = copy(shortNames = Some(value))
  /** Appends new values to shortNames */
  def addShortNames(newValues: String*) : CustomResourceDefinitionNames = copy(shortNames = Some(shortNames.fold(newValues)(_ ++ newValues)))
  /** if shortNames has a value, transforms to the result of function*/
  def mapShortNames(f: Seq[String] => Seq[String]) : CustomResourceDefinitionNames = copy(shortNames = shortNames.map(f))
}

object CustomResourceDefinitionNames {

    implicit val encoder : Encoder[io.k8s.apiextensions_apiserver.pkg.apis.apiextensions.v1.CustomResourceDefinitionNames] = new Encoder[io.k8s.apiextensions_apiserver.pkg.apis.apiextensions.v1.CustomResourceDefinitionNames] {
        def apply[T : Builder](o: io.k8s.apiextensions_apiserver.pkg.apis.apiextensions.v1.CustomResourceDefinitionNames) : T = {
          val obj = ObjectWriter[T]()
          obj
            .write("plural", o.plural)
            .write("singular", o.singular)
            .write("listKind", o.listKind)
            .write("categories", o.categories)
            .write("shortNames", o.shortNames)
            .build
        }
    }

    implicit val decoder: Decoder[CustomResourceDefinitionNames] = new Decoder[CustomResourceDefinitionNames] {
      def apply[T : Reader](t: T): Either[String, CustomResourceDefinitionNames] = for {
          obj <- ObjectReader(t)
          plural <- obj.read[String]("plural")
          singular <- obj.readOpt[String]("singular")
          listKind <- obj.readOpt[String]("listKind")
          categories <- obj.readOpt[Seq[String]]("categories")
          shortNames <- obj.readOpt[Seq[String]]("shortNames")
      } yield CustomResourceDefinitionNames (
          plural = plural,
          singular = singular,
          listKind = listKind,
          categories = categories,
          shortNames = shortNames
        )
    }
}

