package io.k8s.api.rbac.v1

import dev.hnaderi.k8s.utils._

/** Subject contains a reference to the object or user identities a role binding applies to.  This can either hold a direct API object reference, or a value for non-objects such as user and group names. */
final case class Subject(
  name : String,
  apiGroup : Option[String] = None,
  namespace : Option[String] = None
) {

  /** Returns a new data with name set to new value */
  def withName(value: String) : Subject = copy(name = value)
  /** transforms name to result of function */
  def mapName(f: String => String) : Subject = copy(name = f(name))

  /** Returns a new data with apiGroup set to new value */
  def withApiGroup(value: String) : Subject = copy(apiGroup = Some(value))
  /** if apiGroup has a value, transforms to the result of function*/
  def mapApiGroup(f: String => String) : Subject = copy(apiGroup = apiGroup.map(f))

  /** Returns a new data with namespace set to new value */
  def withNamespace(value: String) : Subject = copy(namespace = Some(value))
  /** if namespace has a value, transforms to the result of function*/
  def mapNamespace(f: String => String) : Subject = copy(namespace = namespace.map(f))
}

object Subject {

    implicit def encoder[T](implicit builder : Builder[T]) : Encoder[io.k8s.api.rbac.v1.Subject, T] = new Encoder[io.k8s.api.rbac.v1.Subject, T] {
        def apply(o: io.k8s.api.rbac.v1.Subject) : T = {
          val obj = ObjectWriter[T]()
          obj
            .write("name", o.name)
            .write("apiGroup", o.apiGroup)
            .write("namespace", o.namespace)
            .build
        }
    }

    implicit def decoderOf[T : Reader] : Decoder[T, Subject] = new Decoder[T, Subject] {
      def apply(t: T): Either[String, Subject] = for {
          obj <- ObjectReader(t)
          name <- obj.read[String]("name")
          apiGroup <- obj.readOpt[String]("apiGroup")
          namespace <- obj.readOpt[String]("namespace")
      } yield Subject (
          name = name,
          apiGroup = apiGroup,
          namespace = namespace
        )
    }
}

