package io.k8s.api.core.v1

import dev.hnaderi.k8s.utils._

/** TypedLocalObjectReference contains enough information to let you locate the typed referenced object inside the same namespace. */
final case class TypedLocalObjectReference(
  name : String,
  apiGroup : Option[String] = None
) {

  /** Returns a new data with name set to new value */
  def withName(value: String) : TypedLocalObjectReference = copy(name = value)
  /** transforms name to result of function */
  def mapName(f: String => String) : TypedLocalObjectReference = copy(name = f(name))

  /** Returns a new data with apiGroup set to new value */
  def withApiGroup(value: String) : TypedLocalObjectReference = copy(apiGroup = Some(value))
  /** if apiGroup has a value, transforms to the result of function*/
  def mapApiGroup(f: String => String) : TypedLocalObjectReference = copy(apiGroup = apiGroup.map(f))
}

object TypedLocalObjectReference {

    implicit def encoder[T](implicit builder : Builder[T]) : Encoder[io.k8s.api.core.v1.TypedLocalObjectReference, T] = new Encoder[io.k8s.api.core.v1.TypedLocalObjectReference, T] {
        def apply(o: io.k8s.api.core.v1.TypedLocalObjectReference) : T = {
          val obj = ObjectWriter[T]()
          obj
            .write("name", o.name)
            .write("apiGroup", o.apiGroup)
            .build
        }
    }

    implicit def decoderOf[T : Reader] : Decoder[T, TypedLocalObjectReference] = new Decoder[T, TypedLocalObjectReference] {
      def apply(t: T): Either[String, TypedLocalObjectReference] = for {
          obj <- ObjectReader(t)
          name <- obj.read[String]("name")
          apiGroup <- obj.readOpt[String]("apiGroup")
      } yield TypedLocalObjectReference (
          name = name,
          apiGroup = apiGroup
        )
    }
}

