package io.k8s.api.core.v1

import dev.hnaderi.k8s.utils._

/** Represents a Glusterfs mount that lasts the lifetime of a pod. Glusterfs volumes do not support ownership management or SELinux relabeling. */
final case class GlusterfsPersistentVolumeSource(
  endpoints : String,
  path : String,
  endpointsNamespace : Option[String] = None,
  readOnly : Option[Boolean] = None
) {

  /** Returns a new data with endpoints set to new value */
  def withEndpoints(value: String) : GlusterfsPersistentVolumeSource = copy(endpoints = value)
  /** transforms endpoints to result of function */
  def mapEndpoints(f: String => String) : GlusterfsPersistentVolumeSource = copy(endpoints = f(endpoints))

  /** Returns a new data with path set to new value */
  def withPath(value: String) : GlusterfsPersistentVolumeSource = copy(path = value)
  /** transforms path to result of function */
  def mapPath(f: String => String) : GlusterfsPersistentVolumeSource = copy(path = f(path))

  /** Returns a new data with endpointsNamespace set to new value */
  def withEndpointsNamespace(value: String) : GlusterfsPersistentVolumeSource = copy(endpointsNamespace = Some(value))
  /** if endpointsNamespace has a value, transforms to the result of function*/
  def mapEndpointsNamespace(f: String => String) : GlusterfsPersistentVolumeSource = copy(endpointsNamespace = endpointsNamespace.map(f))

  /** Returns a new data with readOnly set to new value */
  def withReadOnly(value: Boolean) : GlusterfsPersistentVolumeSource = copy(readOnly = Some(value))
  /** if readOnly has a value, transforms to the result of function*/
  def mapReadOnly(f: Boolean => Boolean) : GlusterfsPersistentVolumeSource = copy(readOnly = readOnly.map(f))
}

object GlusterfsPersistentVolumeSource {

    implicit def encoder[T](implicit builder : Builder[T]) : Encoder[io.k8s.api.core.v1.GlusterfsPersistentVolumeSource, T] = new Encoder[io.k8s.api.core.v1.GlusterfsPersistentVolumeSource, T] {
        def apply(o: io.k8s.api.core.v1.GlusterfsPersistentVolumeSource) : T = {
          val obj = ObjectWriter[T]()
          obj
            .write("endpoints", o.endpoints)
            .write("path", o.path)
            .write("endpointsNamespace", o.endpointsNamespace)
            .write("readOnly", o.readOnly)
            .build
        }
    }

    implicit def decoderOf[T : Reader] : Decoder[T, GlusterfsPersistentVolumeSource] = new Decoder[T, GlusterfsPersistentVolumeSource] {
      def apply(t: T): Either[String, GlusterfsPersistentVolumeSource] = for {
          obj <- ObjectReader(t)
          endpoints <- obj.read[String]("endpoints")
          path <- obj.read[String]("path")
          endpointsNamespace <- obj.readOpt[String]("endpointsNamespace")
          readOnly <- obj.readOpt[Boolean]("readOnly")
      } yield GlusterfsPersistentVolumeSource (
          endpoints = endpoints,
          path = path,
          endpointsNamespace = endpointsNamespace,
          readOnly = readOnly
        )
    }
}

