package io.k8s.api.core.v1

import dev.hnaderi.k8s.utils._

/** Represents a Persistent Disk resource in AWS.

An AWS EBS disk must exist before mounting to a container. The disk must also be in the same AWS zone as the kubelet. An AWS EBS disk can only be mounted as read/write once. AWS EBS volumes support ownership management and SELinux relabeling. */
final case class AWSElasticBlockStoreVolumeSource(
  volumeID : String,
  fsType : Option[String] = None,
  partition : Option[Int] = None,
  readOnly : Option[Boolean] = None
) {

  /** Returns a new data with volumeID set to new value */
  def withVolumeID(value: String) : AWSElasticBlockStoreVolumeSource = copy(volumeID = value)
  /** transforms volumeID to result of function */
  def mapVolumeID(f: String => String) : AWSElasticBlockStoreVolumeSource = copy(volumeID = f(volumeID))

  /** Returns a new data with fsType set to new value */
  def withFsType(value: String) : AWSElasticBlockStoreVolumeSource = copy(fsType = Some(value))
  /** if fsType has a value, transforms to the result of function*/
  def mapFsType(f: String => String) : AWSElasticBlockStoreVolumeSource = copy(fsType = fsType.map(f))

  /** Returns a new data with partition set to new value */
  def withPartition(value: Int) : AWSElasticBlockStoreVolumeSource = copy(partition = Some(value))
  /** if partition has a value, transforms to the result of function*/
  def mapPartition(f: Int => Int) : AWSElasticBlockStoreVolumeSource = copy(partition = partition.map(f))

  /** Returns a new data with readOnly set to new value */
  def withReadOnly(value: Boolean) : AWSElasticBlockStoreVolumeSource = copy(readOnly = Some(value))
  /** if readOnly has a value, transforms to the result of function*/
  def mapReadOnly(f: Boolean => Boolean) : AWSElasticBlockStoreVolumeSource = copy(readOnly = readOnly.map(f))
}

object AWSElasticBlockStoreVolumeSource {

    implicit def encoder[T](implicit builder : Builder[T]) : Encoder[io.k8s.api.core.v1.AWSElasticBlockStoreVolumeSource, T] = new Encoder[io.k8s.api.core.v1.AWSElasticBlockStoreVolumeSource, T] {
        def apply(o: io.k8s.api.core.v1.AWSElasticBlockStoreVolumeSource) : T = {
          val obj = ObjectWriter[T]()
          obj
            .write("volumeID", o.volumeID)
            .write("fsType", o.fsType)
            .write("partition", o.partition)
            .write("readOnly", o.readOnly)
            .build
        }
    }

    implicit def decoderOf[T : Reader] : Decoder[T, AWSElasticBlockStoreVolumeSource] = new Decoder[T, AWSElasticBlockStoreVolumeSource] {
      def apply(t: T): Either[String, AWSElasticBlockStoreVolumeSource] = for {
          obj <- ObjectReader(t)
          volumeID <- obj.read[String]("volumeID")
          fsType <- obj.readOpt[String]("fsType")
          partition <- obj.readOpt[Int]("partition")
          readOnly <- obj.readOpt[Boolean]("readOnly")
      } yield AWSElasticBlockStoreVolumeSource (
          volumeID = volumeID,
          fsType = fsType,
          partition = partition,
          readOnly = readOnly
        )
    }
}

