package io.k8s.api.core.v1

import dev.hnaderi.k8s.utils._

/** HostAlias holds the mapping between IP and hostnames that will be injected as an entry in the pod's hosts file. */
final case class HostAlias(
  ip : String,
  hostnames : Option[Seq[String]] = None
) {

  /** Returns a new data with ip set to new value */
  def withIp(value: String) : HostAlias = copy(ip = value)
  /** transforms ip to result of function */
  def mapIp(f: String => String) : HostAlias = copy(ip = f(ip))

  /** Returns a new data with hostnames set to new value */
  def withHostnames(value: Seq[String]) : HostAlias = copy(hostnames = Some(value))
  /** Appends new values to hostnames */
  def addHostnames(newValues: String*) : HostAlias = copy(hostnames = Some(hostnames.fold(newValues)(_ ++ newValues)))
  /** if hostnames has a value, transforms to the result of function*/
  def mapHostnames(f: Seq[String] => Seq[String]) : HostAlias = copy(hostnames = hostnames.map(f))
}

object HostAlias {

    implicit val encoder : Encoder[io.k8s.api.core.v1.HostAlias] = new Encoder[io.k8s.api.core.v1.HostAlias] {
        def apply[T : Builder](o: io.k8s.api.core.v1.HostAlias) : T = {
          val obj = ObjectWriter[T]()
          obj
            .write("ip", o.ip)
            .write("hostnames", o.hostnames)
            .build
        }
    }

    implicit val decoder: Decoder[HostAlias] = new Decoder[HostAlias] {
      def apply[T : Reader](t: T): Either[String, HostAlias] = for {
          obj <- ObjectReader(t)
          ip <- obj.read[String]("ip")
          hostnames <- obj.readOpt[Seq[String]]("hostnames")
      } yield HostAlias (
          ip = ip,
          hostnames = hostnames
        )
    }
}

