package io.k8s.api.core.v1

import dev.hnaderi.k8s.utils._

/** ObjectFieldSelector selects an APIVersioned field of an object. */
final case class ObjectFieldSelector(
  fieldPath : String,
  apiVersion : Option[String] = None
) {

  /** Returns a new data with fieldPath set to new value */
  def withFieldPath(value: String) : ObjectFieldSelector = copy(fieldPath = value)
  /** transforms fieldPath to result of function */
  def mapFieldPath(f: String => String) : ObjectFieldSelector = copy(fieldPath = f(fieldPath))

  /** Returns a new data with apiVersion set to new value */
  def withApiVersion(value: String) : ObjectFieldSelector = copy(apiVersion = Some(value))
  /** if apiVersion has a value, transforms to the result of function*/
  def mapApiVersion(f: String => String) : ObjectFieldSelector = copy(apiVersion = apiVersion.map(f))
}

object ObjectFieldSelector {

    implicit val encoder : Encoder[io.k8s.api.core.v1.ObjectFieldSelector] = new Encoder[io.k8s.api.core.v1.ObjectFieldSelector] {
        def apply[T : Builder](o: io.k8s.api.core.v1.ObjectFieldSelector) : T = {
          val obj = ObjectWriter[T]()
          obj
            .write("fieldPath", o.fieldPath)
            .write("apiVersion", o.apiVersion)
            .build
        }
    }

    implicit val decoder: Decoder[ObjectFieldSelector] = new Decoder[ObjectFieldSelector] {
      def apply[T : Reader](t: T): Either[String, ObjectFieldSelector] = for {
          obj <- ObjectReader(t)
          fieldPath <- obj.read[String]("fieldPath")
          apiVersion <- obj.readOpt[String]("apiVersion")
      } yield ObjectFieldSelector (
          fieldPath = fieldPath,
          apiVersion = apiVersion
        )
    }
}

