package io.k8s.api.core.v1

import dev.hnaderi.k8s.utils._

/** Adapts a Secret into a volume.

The contents of the target Secret's Data field will be presented in a volume as files using the keys in the Data field as the file names. Secret volumes support ownership management and SELinux relabeling. */
final case class SecretVolumeSource(
  defaultMode : Option[Int] = None,
  items : Option[Seq[io.k8s.api.core.v1.KeyToPath]] = None,
  optional : Option[Boolean] = None,
  secretName : Option[String] = None
) {

  /** Returns a new data with defaultMode set to new value */
  def withDefaultMode(value: Int) : SecretVolumeSource = copy(defaultMode = Some(value))
  /** if defaultMode has a value, transforms to the result of function*/
  def mapDefaultMode(f: Int => Int) : SecretVolumeSource = copy(defaultMode = defaultMode.map(f))

  /** Returns a new data with items set to new value */
  def withItems(value: Seq[io.k8s.api.core.v1.KeyToPath]) : SecretVolumeSource = copy(items = Some(value))
  /** Appends new values to items */
  def addItems(newValues: io.k8s.api.core.v1.KeyToPath*) : SecretVolumeSource = copy(items = Some(items.fold(newValues)(_ ++ newValues)))
  /** if items has a value, transforms to the result of function*/
  def mapItems(f: Seq[io.k8s.api.core.v1.KeyToPath] => Seq[io.k8s.api.core.v1.KeyToPath]) : SecretVolumeSource = copy(items = items.map(f))

  /** Returns a new data with optional set to new value */
  def withOptional(value: Boolean) : SecretVolumeSource = copy(optional = Some(value))
  /** if optional has a value, transforms to the result of function*/
  def mapOptional(f: Boolean => Boolean) : SecretVolumeSource = copy(optional = optional.map(f))

  /** Returns a new data with secretName set to new value */
  def withSecretName(value: String) : SecretVolumeSource = copy(secretName = Some(value))
  /** if secretName has a value, transforms to the result of function*/
  def mapSecretName(f: String => String) : SecretVolumeSource = copy(secretName = secretName.map(f))
}

object SecretVolumeSource {

    implicit val encoder : Encoder[io.k8s.api.core.v1.SecretVolumeSource] = new Encoder[io.k8s.api.core.v1.SecretVolumeSource] {
        def apply[T : Builder](o: io.k8s.api.core.v1.SecretVolumeSource) : T = {
          val obj = ObjectWriter[T]()
          obj
            .write("defaultMode", o.defaultMode)
            .write("items", o.items)
            .write("optional", o.optional)
            .write("secretName", o.secretName)
            .build
        }
    }

    implicit val decoder: Decoder[SecretVolumeSource] = new Decoder[SecretVolumeSource] {
      def apply[T : Reader](t: T): Either[String, SecretVolumeSource] = for {
          obj <- ObjectReader(t)
          defaultMode <- obj.readOpt[Int]("defaultMode")
          items <- obj.readOpt[Seq[io.k8s.api.core.v1.KeyToPath]]("items")
          optional <- obj.readOpt[Boolean]("optional")
          secretName <- obj.readOpt[String]("secretName")
      } yield SecretVolumeSource (
          defaultMode = defaultMode,
          items = items,
          optional = optional,
          secretName = secretName
        )
    }
}

