package dev.harrel.jsonschema.providers;

import dev.harrel.jsonschema.JsonNode;
import dev.harrel.jsonschema.JsonNodeFactory;
import org.codehaus.jettison.json.JSONArray;
import org.codehaus.jettison.json.JSONException;
import org.codehaus.jettison.json.JSONObject;
import org.codehaus.jettison.json.JSONTokener;

import java.util.*;

public final class JettisonNode extends SimpleJsonNode {

    private JettisonNode(Object node, String jsonPointer) {
        super(Objects.requireNonNull(node), jsonPointer);
    }

    public JettisonNode(Object node) {
        this(node, "");
    }

    @Override
    public boolean asBoolean() {
        return (Boolean) node;
    }

    @Override
    public String asString() {
        return Objects.toString(isNull() ? null : node);
    }

    @Override
    public List<JsonNode> asArray() {
        List<JsonNode> elements = new ArrayList<>();
        JSONArray arrayNode = (JSONArray) node;
        for (int i = 0; i < arrayNode.length(); ++i) {
            elements.add(new JettisonNode(arrayNode.opt(i), jsonPointer + "/" + elements.size()));
        }
        return elements;
    }

    @Override
    @SuppressWarnings("unchecked")
    public Map<String, JsonNode> asObject() {
        JSONObject jsonObject = (JSONObject) node;
        Map<String, JsonNode> map = MapUtil.newHashMap(jsonObject.length());
        for (Object object : jsonObject.toMap().entrySet()) {
            Map.Entry<Object, Object> entry = (Map.Entry<Object, Object>) object;
            map.put(entry.getKey().toString(), new JettisonNode(entry.getValue(), jsonPointer + "/" + JsonNode.encodeJsonPointer(entry.getKey().toString())));
        }
        return map;
    }

    @Override
    boolean isNull(Object node) {
        return JSONObject.NULL.equals(node) || JSONObject.EXPLICIT_NULL.equals(node);
    }

    @Override
    boolean isArray(Object node) {
        return node instanceof JSONArray;
    }

    @Override
    boolean isObject(Object node) {
        return node instanceof JSONObject;
    }

    public static final class Factory implements JsonNodeFactory {
        @Override
        public JsonNode wrap(Object node) {
            if (node instanceof JettisonNode) {
                return (JettisonNode) node;
            } else {
                return new JettisonNode(node);
            }
        }

        @Override
        public JsonNode create(String rawJson) {
            try {
                return new JettisonNode(new BigDecimalTokener(rawJson).nextValue());
            } catch (JSONException e) {
                throw new IllegalArgumentException(e);
            }
        }
    }

    static final class BigDecimalTokener extends JSONTokener {
        BigDecimalTokener(String s) {
            super(s);
            this.useBigDecimal = true;
        }
    }
}
