/*
 * Copyright 2019 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      https://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package dev.gradleplugins.fixtures.sources;

import lombok.Value;
import lombok.val;
import org.apache.commons.io.FileUtils;

import javax.tools.JavaFileObject;
import java.io.File;
import java.io.IOException;
import java.io.UncheckedIOException;
import java.net.URL;
import java.nio.charset.Charset;
import java.util.Arrays;

@Value
public class SourceFile {
    String path;
    String name;
    String content;

    public File writeToDirectory(File base) {
        return writeToDirectory(base, name);
    }

    private File writeToDirectory(File base, String name) {
        val file = new File(base, String.join(File.separator, path, name));
        writeToFile(file);
        return file;
    }

    public void writeToFile(File file) {
        try {
            FileUtils.touch(file);
            FileUtils.write(file, content, Charset.defaultCharset());
        } catch (IOException ex) {
            throw new UncheckedIOException(String.format("Unable to create source file at '%s'.", file.getAbsolutePath()), ex);
        }
    }

    public String withPath(String basePath) {
        return String.join("/", basePath, path, name);
    }

    @Override
    public String toString() {
        return "SourceFile{" +
                "path='" + path + '\'' +
                ", name='" + name + '\'' +
                ", content='" + firstContentLine(content) + '\'' +
                '}';
    }

    private static String firstContentLine(String content) {
        String[] tokens = content.split("\n", -1);
        return Arrays.stream(tokens).map(String::trim).filter(line -> !line.isEmpty()).findFirst().map(it -> it + "...").orElse("");
    }

    public SourceKind getKind() {
        return SourceKind.valueOf(name);
    }
//
//    /**
//     * Returns a {@link SourceFile} for the class path with the given {@code resourceName}.
//     *
//     * @param resourceName The name of the resource to load.
//     * @return a {@link SourceFile} instance, never null.
//     */
//    public static SourceFile forResource(String resourceName) {
//
//    }
//
//    /**
//     * Returns a {@link SourceFile} for the resource at the given {@link URL}.
//     * The {@linkplain SourceFile#getKind() kind} is inferred via the {@link Kind#extension}.
//     */
//    public static SourceFile forResource(URL resourceUrl) {
//        if ("jar".equals(resourceUrl.getProtocol())) {
//            return new SourceFile('')
//        } else {
//            return new ResourceSourceJavaFileObject(resourceUrl);
//        }
//    }
}
