/*
 * Copyright 2020 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.gradle.smoketests

import org.gradle.integtests.fixtures.executer.GradleContextualExecuter
import org.gradle.testkit.runner.BuildResult
import org.gradle.util.internal.VersionNumber

import static org.gradle.testkit.runner.TaskOutcome.FROM_CACHE
import static org.gradle.testkit.runner.TaskOutcome.NO_SOURCE
import static org.gradle.testkit.runner.TaskOutcome.SKIPPED
import static org.gradle.testkit.runner.TaskOutcome.SUCCESS
import static org.gradle.testkit.runner.TaskOutcome.UP_TO_DATE

class AndroidSantaTrackerCachingSmokeTest extends AbstractAndroidSantaTrackerSmokeTest {

    def "can cache Santa Tracker Android application (agp=#agpVersion)"() {

        given:
        AGP_VERSIONS.assumeCurrentJavaVersionIsSupportedBy(agpVersion)

        and:
        def originalDir = temporaryFolder.createDir("original")
        def relocatedDir = temporaryFolder.createDir("relocated")

        and:
        setupCopyOfSantaTracker(originalDir)
        setupCopyOfSantaTracker(relocatedDir)

        when: 'clean build'
        SantaTrackerConfigurationCacheWorkaround.beforeBuild(originalDir, homeDir)
        buildLocationMaybeExpectingWorkerExecutorAndConventionDeprecation(originalDir, agpVersion)

        then:
        assertConfigurationCacheStateStored()

        when: 'up-to-date build, reusing configuration cache when enabled'
        SantaTrackerConfigurationCacheWorkaround.beforeBuild(originalDir, homeDir)
        buildUpToDateLocation(originalDir, agpVersion)

        then:
        // TODO - this is here because AGP >=7.4 and <8.1.0 reads build/generated/source/kapt/debug at configuration time
        if (agpVersion.startsWith('7.3') || VersionNumber.parse(agpVersion) >= VersionNumber.parse('8.1.0')) {
            assertConfigurationCacheStateLoaded()
        } else {
            assertConfigurationCacheStateStored()
        }

        when: 'up-to-date build, reusing configuration cache when enabled'
        SantaTrackerConfigurationCacheWorkaround.beforeBuild(originalDir, homeDir)
        buildLocation(originalDir, agpVersion)

        then:
        assertConfigurationCacheStateLoaded()

        when: 'clean cached build'
        SantaTrackerConfigurationCacheWorkaround.beforeBuild(relocatedDir, homeDir)
        BuildResult relocatedResult = buildLocationMaybeExpectingWorkerExecutorAndConventionDeprecation(relocatedDir, agpVersion)

        then:
        assertConfigurationCacheStateStored()

        and:
        def expectedResults = agpVersion.startsWith('7.3')
            ? AndroidPluginExpectations7.EXPECTED_RESULTS_7_3
            : agpVersion.startsWith('7.4')
            ? AndroidPluginExpectations7.EXPECTED_RESULTS_7_4
            : agpVersion.startsWith('8.0')
            ? AndroidPluginExpectations8.EXPECTED_RESULTS_8_0
            : AndroidPluginExpectations8.EXPECTED_RESULTS_8_1
        verify(relocatedResult, expectedResults)

        when: 'clean cached build, reusing configuration cache when enabled'
        cleanLocation(relocatedDir, agpVersion)
        SantaTrackerConfigurationCacheWorkaround.beforeBuild(relocatedDir, homeDir)
        if (GradleContextualExecuter.notConfigCache) {
            buildLocationMaybeExpectingWorkerExecutorAndConventionDeprecation(relocatedDir, agpVersion)
        } else {
            buildLocationMaybeExpectingWorkerExecutorDeprecation(relocatedDir, agpVersion)
        }

        then:
        assertConfigurationCacheStateLoaded()

        where:
        agpVersion << TESTED_AGP_VERSIONS
    }
}

class AndroidPluginExpectations8 {

    static final EXPECTED_RESULTS_8_1 = [
        ':cityquiz:assembleDebug': SUCCESS,
        ':cityquiz:checkDebugAarMetadata': SUCCESS,
        ':cityquiz:checkDebugDuplicateClasses': SUCCESS,
        ':cityquiz:checkKotlinGradlePluginConfigurationErrors': SUCCESS,
        ':cityquiz:compileDebugJavaWithJavac': FROM_CACHE,
        ':cityquiz:compileDebugKotlin': FROM_CACHE,
        ':cityquiz:compileDebugShaders': NO_SOURCE,
        ':cityquiz:compressDebugAssets': FROM_CACHE,
        ':cityquiz:createDebugApkListingFileRedirect': SUCCESS,
        ':cityquiz:desugarDebugFileDependencies': FROM_CACHE,
        ':cityquiz:dexBuilderDebug': FROM_CACHE,
        ':cityquiz:extractDeepLinksDebug': FROM_CACHE,
        ':cityquiz:featureDebugWriter': SUCCESS,
        ':cityquiz:featureGlobalSyntheticsDebugMerge': SUCCESS,
        ':cityquiz:generateDebugAssets': UP_TO_DATE,
        ':cityquiz:generateDebugBuildConfig': FROM_CACHE,
        ':cityquiz:generateDebugFeatureTransitiveDeps': SUCCESS,
        ':cityquiz:generateDebugResValues': FROM_CACHE,
        ':cityquiz:generateDebugResources': UP_TO_DATE,
        ':cityquiz:javaPreCompileDebug': FROM_CACHE,
        ':cityquiz:mapDebugSourceSetPaths': SUCCESS,
        ':cityquiz:mergeDebugAssets': FROM_CACHE,
        ':cityquiz:mergeDebugGlobalSynthetics': FROM_CACHE,
        ':cityquiz:mergeDebugJavaResource': SUCCESS,
        ':cityquiz:mergeDebugJniLibFolders': FROM_CACHE,
        ':cityquiz:mergeDebugNativeLibs': NO_SOURCE,
        ':cityquiz:mergeDebugResources': FROM_CACHE,
        ':cityquiz:mergeDebugShaders': FROM_CACHE,
        ':cityquiz:mergeExtDexDebug': FROM_CACHE,
        ':cityquiz:mergeLibDexDebug': FROM_CACHE,
        ':cityquiz:mergeProjectDexDebug': FROM_CACHE,
        ':cityquiz:packageDebug': SUCCESS,
        ':cityquiz:preBuild': UP_TO_DATE,
        ':cityquiz:preDebugBuild': UP_TO_DATE,
        ':cityquiz:processApplicationManifestDebugForBundle': SUCCESS,
        ':cityquiz:processDebugJavaRes': SUCCESS,
        ':cityquiz:processDebugMainManifest': FROM_CACHE,
        ':cityquiz:processDebugManifestForPackage': FROM_CACHE,
        ':cityquiz:processDebugResources': FROM_CACHE,
        ':cityquiz:processManifestDebugForFeature': FROM_CACHE,
        ':cityquiz:stripDebugDebugSymbols': NO_SOURCE,
        ':common:assembleDebug': SUCCESS,
        ':common:bundleDebugAar': SUCCESS,
        ':common:bundleLibCompileToJarDebug': SUCCESS,
        ':common:bundleLibRuntimeToJarDebug': SUCCESS,
        ':common:checkKotlinGradlePluginConfigurationErrors': SUCCESS,
        ':common:compileDebugJavaWithJavac': FROM_CACHE,
        ':common:compileDebugKotlin': FROM_CACHE,
        ':common:compileDebugLibraryResources': FROM_CACHE,
        ':common:compileDebugShaders': NO_SOURCE,
        ':common:copyDebugJniLibsProjectAndLocalJars': SUCCESS,
        ':common:copyDebugJniLibsProjectOnly': SUCCESS,
        ':common:createFullJarDebug': SUCCESS,
        ':common:dataBindingGenBaseClassesDebug': FROM_CACHE,
        ':common:dataBindingMergeDependencyArtifactsDebug': SUCCESS,
        ':common:extractDebugAnnotations': FROM_CACHE,
        ':common:extractDeepLinksDebug': FROM_CACHE,
        ':common:extractDeepLinksForAarDebug': FROM_CACHE,
        ':common:generateDebugAssets': UP_TO_DATE,
        ':common:generateDebugBuildConfig': FROM_CACHE,
        ':common:generateDebugRFile': FROM_CACHE,
        ':common:generateDebugResValues': FROM_CACHE,
        ':common:generateDebugResources': UP_TO_DATE,
        ':common:javaPreCompileDebug': FROM_CACHE,
        ':common:mergeDebugConsumerProguardFiles': SUCCESS,
        ':common:mergeDebugGeneratedProguardFiles': SUCCESS,
        ':common:mergeDebugJavaResource': SUCCESS,
        ':common:mergeDebugJniLibFolders': FROM_CACHE,
        ':common:mergeDebugNativeLibs': NO_SOURCE,
        ':common:mergeDebugShaders': FROM_CACHE,
        ':common:packageDebugAssets': FROM_CACHE,
        ':common:packageDebugResources': FROM_CACHE,
        ':common:parseDebugLocalResources': FROM_CACHE,
        ':common:preBuild': UP_TO_DATE,
        ':common:preDebugBuild': UP_TO_DATE,
        ':common:prepareDebugArtProfile': SUCCESS,
        ':common:prepareLintJarForPublish': SUCCESS,
        ':common:processDebugJavaRes': SUCCESS,
        ':common:processDebugManifest': FROM_CACHE,
        ':common:stripDebugDebugSymbols': NO_SOURCE,
        ':common:syncDebugLibJars': FROM_CACHE,
        ':common:writeDebugAarMetadata': SUCCESS,
        ':dasherdancer:assembleDebug': SUCCESS,
        ':dasherdancer:checkDebugAarMetadata': SUCCESS,
        ':dasherdancer:checkDebugDuplicateClasses': SUCCESS,
        ':dasherdancer:checkKotlinGradlePluginConfigurationErrors': SUCCESS,
        ':dasherdancer:compileDebugJavaWithJavac': NO_SOURCE,
        ':dasherdancer:compileDebugKotlin': FROM_CACHE,
        ':dasherdancer:compileDebugShaders': NO_SOURCE,
        ':dasherdancer:compressDebugAssets': FROM_CACHE,
        ':dasherdancer:createDebugApkListingFileRedirect': SUCCESS,
        ':dasherdancer:desugarDebugFileDependencies': FROM_CACHE,
        ':dasherdancer:dexBuilderDebug': FROM_CACHE,
        ':dasherdancer:extractDeepLinksDebug': FROM_CACHE,
        ':dasherdancer:featureDebugWriter': SUCCESS,
        ':dasherdancer:featureGlobalSyntheticsDebugMerge': SUCCESS,
        ':dasherdancer:generateDebugAssets': UP_TO_DATE,
        ':dasherdancer:generateDebugFeatureTransitiveDeps': SUCCESS,
        ':dasherdancer:generateDebugResValues': FROM_CACHE,
        ':dasherdancer:generateDebugResources': UP_TO_DATE,
        ':dasherdancer:javaPreCompileDebug': FROM_CACHE,
        ':dasherdancer:mapDebugSourceSetPaths': SUCCESS,
        ':dasherdancer:mergeDebugAssets': FROM_CACHE,
        ':dasherdancer:mergeDebugGlobalSynthetics': FROM_CACHE,
        ':dasherdancer:mergeDebugJavaResource': SUCCESS,
        ':dasherdancer:mergeDebugJniLibFolders': FROM_CACHE,
        ':dasherdancer:mergeDebugNativeLibs': NO_SOURCE,
        ':dasherdancer:mergeDebugResources': FROM_CACHE,
        ':dasherdancer:mergeDebugShaders': FROM_CACHE,
        ':dasherdancer:mergeExtDexDebug': FROM_CACHE,
        ':dasherdancer:mergeLibDexDebug': FROM_CACHE,
        ':dasherdancer:mergeProjectDexDebug': FROM_CACHE,
        ':dasherdancer:packageDebug': SUCCESS,
        ':dasherdancer:preBuild': UP_TO_DATE,
        ':dasherdancer:preDebugBuild': UP_TO_DATE,
        ':dasherdancer:processApplicationManifestDebugForBundle': SUCCESS,
        ':dasherdancer:processDebugJavaRes': SUCCESS,
        ':dasherdancer:processDebugMainManifest': FROM_CACHE,
        ':dasherdancer:processDebugManifestForPackage': FROM_CACHE,
        ':dasherdancer:processDebugResources': FROM_CACHE,
        ':dasherdancer:processManifestDebugForFeature': FROM_CACHE,
        ':dasherdancer:stripDebugDebugSymbols': NO_SOURCE,
        ':doodles-lib:assembleDebug': SUCCESS,
        ':doodles-lib:bundleDebugAar': SUCCESS,
        ':doodles-lib:bundleLibCompileToJarDebug': SUCCESS,
        ':doodles-lib:bundleLibRuntimeToJarDebug': SUCCESS,
        ':doodles-lib:compileDebugJavaWithJavac': FROM_CACHE,
        ':doodles-lib:compileDebugLibraryResources': FROM_CACHE,
        ':doodles-lib:compileDebugShaders': NO_SOURCE,
        ':doodles-lib:copyDebugJniLibsProjectAndLocalJars': SUCCESS,
        ':doodles-lib:copyDebugJniLibsProjectOnly': SUCCESS,
        ':doodles-lib:createFullJarDebug': SUCCESS,
        ':doodles-lib:extractDebugAnnotations': FROM_CACHE,
        ':doodles-lib:extractDeepLinksDebug': FROM_CACHE,
        ':doodles-lib:extractDeepLinksForAarDebug': FROM_CACHE,
        ':doodles-lib:generateDebugAssets': UP_TO_DATE,
        ':doodles-lib:generateDebugRFile': FROM_CACHE,
        ':doodles-lib:generateDebugResValues': FROM_CACHE,
        ':doodles-lib:generateDebugResources': UP_TO_DATE,
        ':doodles-lib:javaPreCompileDebug': FROM_CACHE,
        ':doodles-lib:mergeDebugConsumerProguardFiles': SUCCESS,
        ':doodles-lib:mergeDebugGeneratedProguardFiles': SUCCESS,
        ':doodles-lib:mergeDebugJavaResource': SUCCESS,
        ':doodles-lib:mergeDebugJniLibFolders': FROM_CACHE,
        ':doodles-lib:mergeDebugNativeLibs': NO_SOURCE,
        ':doodles-lib:mergeDebugShaders': FROM_CACHE,
        ':doodles-lib:packageDebugAssets': FROM_CACHE,
        ':doodles-lib:packageDebugResources': FROM_CACHE,
        ':doodles-lib:parseDebugLocalResources': FROM_CACHE,
        ':doodles-lib:preBuild': UP_TO_DATE,
        ':doodles-lib:preDebugBuild': UP_TO_DATE,
        ':doodles-lib:prepareDebugArtProfile': SUCCESS,
        ':doodles-lib:prepareLintJarForPublish': SUCCESS,
        ':doodles-lib:processDebugJavaRes': NO_SOURCE,
        ':doodles-lib:processDebugManifest': FROM_CACHE,
        ':doodles-lib:stripDebugDebugSymbols': NO_SOURCE,
        ':doodles-lib:syncDebugLibJars': FROM_CACHE,
        ':doodles-lib:writeDebugAarMetadata': SUCCESS,
        ':gumball:assembleDebug': SUCCESS,
        ':gumball:checkDebugAarMetadata': SUCCESS,
        ':gumball:checkDebugDuplicateClasses': SUCCESS,
        ':gumball:compileDebugJavaWithJavac': FROM_CACHE,
        ':gumball:compileDebugShaders': NO_SOURCE,
        ':gumball:compressDebugAssets': FROM_CACHE,
        ':gumball:createDebugApkListingFileRedirect': SUCCESS,
        ':gumball:desugarDebugFileDependencies': FROM_CACHE,
        ':gumball:dexBuilderDebug': FROM_CACHE,
        ':gumball:extractDeepLinksDebug': FROM_CACHE,
        ':gumball:featureDebugWriter': SUCCESS,
        ':gumball:featureGlobalSyntheticsDebugMerge': SUCCESS,
        ':gumball:generateDebugAssets': UP_TO_DATE,
        ':gumball:generateDebugFeatureTransitiveDeps': SUCCESS,
        ':gumball:generateDebugResValues': FROM_CACHE,
        ':gumball:generateDebugResources': UP_TO_DATE,
        ':gumball:javaPreCompileDebug': FROM_CACHE,
        ':gumball:mapDebugSourceSetPaths': SUCCESS,
        ':gumball:mergeDebugAssets': FROM_CACHE,
        ':gumball:mergeDebugGlobalSynthetics': FROM_CACHE,
        ':gumball:mergeDebugJavaResource': SUCCESS,
        ':gumball:mergeDebugJniLibFolders': FROM_CACHE,
        ':gumball:mergeDebugNativeLibs': NO_SOURCE,
        ':gumball:mergeDebugResources': FROM_CACHE,
        ':gumball:mergeDebugShaders': FROM_CACHE,
        ':gumball:mergeExtDexDebug': FROM_CACHE,
        ':gumball:mergeLibDexDebug': FROM_CACHE,
        ':gumball:mergeProjectDexDebug': FROM_CACHE,
        ':gumball:packageDebug': SUCCESS,
        ':gumball:preBuild': UP_TO_DATE,
        ':gumball:preDebugBuild': UP_TO_DATE,
        ':gumball:processApplicationManifestDebugForBundle': SUCCESS,
        ':gumball:processDebugJavaRes': NO_SOURCE,
        ':gumball:processDebugMainManifest': FROM_CACHE,
        ':gumball:processDebugManifestForPackage': FROM_CACHE,
        ':gumball:processDebugResources': FROM_CACHE,
        ':gumball:processManifestDebugForFeature': FROM_CACHE,
        ':gumball:stripDebugDebugSymbols': NO_SOURCE,
        ':jetpack:assembleDebug': SUCCESS,
        ':jetpack:checkDebugAarMetadata': SUCCESS,
        ':jetpack:checkDebugDuplicateClasses': SUCCESS,
        ':jetpack:checkKotlinGradlePluginConfigurationErrors': SUCCESS,
        ':jetpack:compileDebugJavaWithJavac': NO_SOURCE,
        ':jetpack:compileDebugKotlin': FROM_CACHE,
        ':jetpack:compileDebugShaders': NO_SOURCE,
        ':jetpack:compressDebugAssets': FROM_CACHE,
        ':jetpack:createDebugApkListingFileRedirect': SUCCESS,
        ':jetpack:desugarDebugFileDependencies': FROM_CACHE,
        ':jetpack:dexBuilderDebug': FROM_CACHE,
        ':jetpack:extractDeepLinksDebug': FROM_CACHE,
        ':jetpack:featureDebugWriter': SUCCESS,
        ':jetpack:featureGlobalSyntheticsDebugMerge': SUCCESS,
        ':jetpack:generateDebugAssets': UP_TO_DATE,
        ':jetpack:generateDebugFeatureTransitiveDeps': SUCCESS,
        ':jetpack:generateDebugResValues': FROM_CACHE,
        ':jetpack:generateDebugResources': UP_TO_DATE,
        ':jetpack:javaPreCompileDebug': FROM_CACHE,
        ':jetpack:mapDebugSourceSetPaths': SUCCESS,
        ':jetpack:mergeDebugAssets': FROM_CACHE,
        ':jetpack:mergeDebugGlobalSynthetics': FROM_CACHE,
        ':jetpack:mergeDebugJavaResource': SUCCESS,
        ':jetpack:mergeDebugJniLibFolders': FROM_CACHE,
        ':jetpack:mergeDebugNativeLibs': NO_SOURCE,
        ':jetpack:mergeDebugResources': FROM_CACHE,
        ':jetpack:mergeDebugShaders': FROM_CACHE,
        ':jetpack:mergeExtDexDebug': FROM_CACHE,
        ':jetpack:mergeLibDexDebug': FROM_CACHE,
        ':jetpack:mergeProjectDexDebug': FROM_CACHE,
        ':jetpack:packageDebug': SUCCESS,
        ':jetpack:preBuild': UP_TO_DATE,
        ':jetpack:preDebugBuild': UP_TO_DATE,
        ':jetpack:processApplicationManifestDebugForBundle': SUCCESS,
        ':jetpack:processDebugJavaRes': SUCCESS,
        ':jetpack:processDebugMainManifest': FROM_CACHE,
        ':jetpack:processDebugManifestForPackage': FROM_CACHE,
        ':jetpack:processDebugResources': FROM_CACHE,
        ':jetpack:processManifestDebugForFeature': FROM_CACHE,
        ':jetpack:stripDebugDebugSymbols': NO_SOURCE,
        ':memory:assembleDebug': SUCCESS,
        ':memory:checkDebugAarMetadata': SUCCESS,
        ':memory:checkDebugDuplicateClasses': SUCCESS,
        ':memory:compileDebugJavaWithJavac': FROM_CACHE,
        ':memory:compileDebugShaders': NO_SOURCE,
        ':memory:compressDebugAssets': FROM_CACHE,
        ':memory:createDebugApkListingFileRedirect': SUCCESS,
        ':memory:desugarDebugFileDependencies': FROM_CACHE,
        ':memory:dexBuilderDebug': FROM_CACHE,
        ':memory:extractDeepLinksDebug': FROM_CACHE,
        ':memory:featureDebugWriter': SUCCESS,
        ':memory:featureGlobalSyntheticsDebugMerge': SUCCESS,
        ':memory:generateDebugAssets': UP_TO_DATE,
        ':memory:generateDebugFeatureTransitiveDeps': SUCCESS,
        ':memory:generateDebugResValues': FROM_CACHE,
        ':memory:generateDebugResources': UP_TO_DATE,
        ':memory:javaPreCompileDebug': FROM_CACHE,
        ':memory:mapDebugSourceSetPaths': SUCCESS,
        ':memory:mergeDebugAssets': FROM_CACHE,
        ':memory:mergeDebugGlobalSynthetics': FROM_CACHE,
        ':memory:mergeDebugJavaResource': SUCCESS,
        ':memory:mergeDebugJniLibFolders': FROM_CACHE,
        ':memory:mergeDebugNativeLibs': NO_SOURCE,
        ':memory:mergeDebugResources': FROM_CACHE,
        ':memory:mergeDebugShaders': FROM_CACHE,
        ':memory:mergeExtDexDebug': FROM_CACHE,
        ':memory:mergeLibDexDebug': FROM_CACHE,
        ':memory:mergeProjectDexDebug': FROM_CACHE,
        ':memory:packageDebug': SUCCESS,
        ':memory:preBuild': UP_TO_DATE,
        ':memory:preDebugBuild': UP_TO_DATE,
        ':memory:processApplicationManifestDebugForBundle': SUCCESS,
        ':memory:processDebugJavaRes': NO_SOURCE,
        ':memory:processDebugMainManifest': FROM_CACHE,
        ':memory:processDebugManifestForPackage': FROM_CACHE,
        ':memory:processDebugResources': FROM_CACHE,
        ':memory:processManifestDebugForFeature': FROM_CACHE,
        ':memory:stripDebugDebugSymbols': NO_SOURCE,
        ':penguinswim:assembleDebug': SUCCESS,
        ':penguinswim:checkDebugAarMetadata': SUCCESS,
        ':penguinswim:checkDebugDuplicateClasses': SUCCESS,
        ':penguinswim:compileDebugJavaWithJavac': FROM_CACHE,
        ':penguinswim:compileDebugShaders': NO_SOURCE,
        ':penguinswim:compressDebugAssets': FROM_CACHE,
        ':penguinswim:createDebugApkListingFileRedirect': SUCCESS,
        ':penguinswim:desugarDebugFileDependencies': FROM_CACHE,
        ':penguinswim:dexBuilderDebug': FROM_CACHE,
        ':penguinswim:extractDeepLinksDebug': FROM_CACHE,
        ':penguinswim:featureDebugWriter': SUCCESS,
        ':penguinswim:featureGlobalSyntheticsDebugMerge': SUCCESS,
        ':penguinswim:generateDebugAssets': UP_TO_DATE,
        ':penguinswim:generateDebugFeatureTransitiveDeps': SUCCESS,
        ':penguinswim:generateDebugResValues': FROM_CACHE,
        ':penguinswim:generateDebugResources': UP_TO_DATE,
        ':penguinswim:javaPreCompileDebug': FROM_CACHE,
        ':penguinswim:mapDebugSourceSetPaths': SUCCESS,
        ':penguinswim:mergeDebugAssets': FROM_CACHE,
        ':penguinswim:mergeDebugGlobalSynthetics': FROM_CACHE,
        ':penguinswim:mergeDebugJavaResource': SUCCESS,
        ':penguinswim:mergeDebugJniLibFolders': FROM_CACHE,
        ':penguinswim:mergeDebugNativeLibs': NO_SOURCE,
        ':penguinswim:mergeDebugResources': FROM_CACHE,
        ':penguinswim:mergeDebugShaders': FROM_CACHE,
        ':penguinswim:mergeExtDexDebug': FROM_CACHE,
        ':penguinswim:mergeLibDexDebug': FROM_CACHE,
        ':penguinswim:mergeProjectDexDebug': FROM_CACHE,
        ':penguinswim:packageDebug': SUCCESS,
        ':penguinswim:preBuild': UP_TO_DATE,
        ':penguinswim:preDebugBuild': UP_TO_DATE,
        ':penguinswim:processApplicationManifestDebugForBundle': SUCCESS,
        ':penguinswim:processDebugJavaRes': NO_SOURCE,
        ':penguinswim:processDebugMainManifest': FROM_CACHE,
        ':penguinswim:processDebugManifestForPackage': FROM_CACHE,
        ':penguinswim:processDebugResources': FROM_CACHE,
        ':penguinswim:processManifestDebugForFeature': FROM_CACHE,
        ':penguinswim:stripDebugDebugSymbols': NO_SOURCE,
        ':playgames:assembleDebug': SUCCESS,
        ':playgames:bundleDebugAar': SUCCESS,
        ':playgames:bundleLibCompileToJarDebug': SUCCESS,
        ':playgames:bundleLibRuntimeToJarDebug': SUCCESS,
        ':playgames:compileDebugJavaWithJavac': FROM_CACHE,
        ':playgames:compileDebugLibraryResources': FROM_CACHE,
        ':playgames:compileDebugShaders': NO_SOURCE,
        ':playgames:copyDebugJniLibsProjectAndLocalJars': SUCCESS,
        ':playgames:copyDebugJniLibsProjectOnly': SUCCESS,
        ':playgames:createFullJarDebug': SUCCESS,
        ':playgames:extractDebugAnnotations': FROM_CACHE,
        ':playgames:extractDeepLinksDebug': FROM_CACHE,
        ':playgames:extractDeepLinksForAarDebug': FROM_CACHE,
        ':playgames:generateDebugAssets': UP_TO_DATE,
        ':playgames:generateDebugRFile': FROM_CACHE,
        ':playgames:generateDebugResValues': FROM_CACHE,
        ':playgames:generateDebugResources': UP_TO_DATE,
        ':playgames:javaPreCompileDebug': FROM_CACHE,
        ':playgames:mergeDebugConsumerProguardFiles': SUCCESS,
        ':playgames:mergeDebugGeneratedProguardFiles': SUCCESS,
        ':playgames:mergeDebugJavaResource': SUCCESS,
        ':playgames:mergeDebugJniLibFolders': FROM_CACHE,
        ':playgames:mergeDebugNativeLibs': NO_SOURCE,
        ':playgames:mergeDebugShaders': FROM_CACHE,
        ':playgames:packageDebugAssets': FROM_CACHE,
        ':playgames:packageDebugResources': FROM_CACHE,
        ':playgames:parseDebugLocalResources': FROM_CACHE,
        ':playgames:preBuild': UP_TO_DATE,
        ':playgames:preDebugBuild': UP_TO_DATE,
        ':playgames:prepareDebugArtProfile': SUCCESS,
        ':playgames:prepareLintJarForPublish': SUCCESS,
        ':playgames:processDebugJavaRes': NO_SOURCE,
        ':playgames:processDebugManifest': FROM_CACHE,
        ':playgames:stripDebugDebugSymbols': NO_SOURCE,
        ':playgames:syncDebugLibJars': FROM_CACHE,
        ':playgames:writeDebugAarMetadata': SUCCESS,
        ':presenttoss:assembleDebug': SUCCESS,
        ':presenttoss:checkDebugAarMetadata': SUCCESS,
        ':presenttoss:checkDebugDuplicateClasses': SUCCESS,
        ':presenttoss:compileDebugJavaWithJavac': FROM_CACHE,
        ':presenttoss:compileDebugShaders': NO_SOURCE,
        ':presenttoss:compressDebugAssets': FROM_CACHE,
        ':presenttoss:createDebugApkListingFileRedirect': SUCCESS,
        ':presenttoss:desugarDebugFileDependencies': FROM_CACHE,
        ':presenttoss:dexBuilderDebug': FROM_CACHE,
        ':presenttoss:extractDeepLinksDebug': FROM_CACHE,
        ':presenttoss:featureDebugWriter': SUCCESS,
        ':presenttoss:featureGlobalSyntheticsDebugMerge': SUCCESS,
        ':presenttoss:generateDebugAssets': UP_TO_DATE,
        ':presenttoss:generateDebugFeatureTransitiveDeps': SUCCESS,
        ':presenttoss:generateDebugResValues': FROM_CACHE,
        ':presenttoss:generateDebugResources': UP_TO_DATE,
        ':presenttoss:javaPreCompileDebug': FROM_CACHE,
        ':presenttoss:mapDebugSourceSetPaths': SUCCESS,
        ':presenttoss:mergeDebugAssets': FROM_CACHE,
        ':presenttoss:mergeDebugGlobalSynthetics': FROM_CACHE,
        ':presenttoss:mergeDebugJavaResource': SUCCESS,
        ':presenttoss:mergeDebugJniLibFolders': FROM_CACHE,
        ':presenttoss:mergeDebugNativeLibs': NO_SOURCE,
        ':presenttoss:mergeDebugResources': FROM_CACHE,
        ':presenttoss:mergeDebugShaders': FROM_CACHE,
        ':presenttoss:mergeExtDexDebug': FROM_CACHE,
        ':presenttoss:mergeLibDexDebug': FROM_CACHE,
        ':presenttoss:mergeProjectDexDebug': FROM_CACHE,
        ':presenttoss:packageDebug': SUCCESS,
        ':presenttoss:preBuild': UP_TO_DATE,
        ':presenttoss:preDebugBuild': UP_TO_DATE,
        ':presenttoss:processApplicationManifestDebugForBundle': SUCCESS,
        ':presenttoss:processDebugJavaRes': NO_SOURCE,
        ':presenttoss:processDebugMainManifest': FROM_CACHE,
        ':presenttoss:processDebugManifestForPackage': FROM_CACHE,
        ':presenttoss:processDebugResources': FROM_CACHE,
        ':presenttoss:processManifestDebugForFeature': FROM_CACHE,
        ':presenttoss:stripDebugDebugSymbols': NO_SOURCE,
        ':rocketsleigh:assembleDebug': SUCCESS,
        ':rocketsleigh:checkDebugAarMetadata': SUCCESS,
        ':rocketsleigh:checkDebugDuplicateClasses': SUCCESS,
        ':rocketsleigh:checkKotlinGradlePluginConfigurationErrors': SUCCESS,
        ':rocketsleigh:compileDebugJavaWithJavac': FROM_CACHE,
        ':rocketsleigh:compileDebugKotlin': FROM_CACHE,
        ':rocketsleigh:compileDebugShaders': NO_SOURCE,
        ':rocketsleigh:compressDebugAssets': FROM_CACHE,
        ':rocketsleigh:createDebugApkListingFileRedirect': SUCCESS,
        ':rocketsleigh:desugarDebugFileDependencies': FROM_CACHE,
        ':rocketsleigh:dexBuilderDebug': FROM_CACHE,
        ':rocketsleigh:extractDeepLinksDebug': FROM_CACHE,
        ':rocketsleigh:featureDebugWriter': SUCCESS,
        ':rocketsleigh:featureGlobalSyntheticsDebugMerge': SUCCESS,
        ':rocketsleigh:generateDebugAssets': UP_TO_DATE,
        ':rocketsleigh:generateDebugBuildConfig': FROM_CACHE,
        ':rocketsleigh:generateDebugFeatureTransitiveDeps': SUCCESS,
        ':rocketsleigh:generateDebugResValues': FROM_CACHE,
        ':rocketsleigh:generateDebugResources': UP_TO_DATE,
        ':rocketsleigh:javaPreCompileDebug': FROM_CACHE,
        ':rocketsleigh:mapDebugSourceSetPaths': SUCCESS,
        ':rocketsleigh:mergeDebugAssets': FROM_CACHE,
        ':rocketsleigh:mergeDebugGlobalSynthetics': FROM_CACHE,
        ':rocketsleigh:mergeDebugJavaResource': SUCCESS,
        ':rocketsleigh:mergeDebugJniLibFolders': FROM_CACHE,
        ':rocketsleigh:mergeDebugNativeLibs': NO_SOURCE,
        ':rocketsleigh:mergeDebugResources': FROM_CACHE,
        ':rocketsleigh:mergeDebugShaders': FROM_CACHE,
        ':rocketsleigh:mergeExtDexDebug': FROM_CACHE,
        ':rocketsleigh:mergeLibDexDebug': FROM_CACHE,
        ':rocketsleigh:mergeProjectDexDebug': FROM_CACHE,
        ':rocketsleigh:packageDebug': SUCCESS,
        ':rocketsleigh:preBuild': UP_TO_DATE,
        ':rocketsleigh:preDebugBuild': UP_TO_DATE,
        ':rocketsleigh:processApplicationManifestDebugForBundle': SUCCESS,
        ':rocketsleigh:processDebugJavaRes': SUCCESS,
        ':rocketsleigh:processDebugMainManifest': FROM_CACHE,
        ':rocketsleigh:processDebugManifestForPackage': FROM_CACHE,
        ':rocketsleigh:processDebugResources': FROM_CACHE,
        ':rocketsleigh:processManifestDebugForFeature': FROM_CACHE,
        ':rocketsleigh:stripDebugDebugSymbols': NO_SOURCE,
        ':santa-tracker:assembleDebug': SUCCESS,
        ':santa-tracker:bundleDebugClassesToCompileJar': SUCCESS,
        ':santa-tracker:bundleDebugClassesToRuntimeJar': SUCCESS,
        ':santa-tracker:checkDebugAarMetadata': SUCCESS,
        ':santa-tracker:checkDebugDuplicateClasses': SUCCESS,
        ':santa-tracker:checkDebugLibraries': SUCCESS,
        ':santa-tracker:checkKotlinGradlePluginConfigurationErrors': SUCCESS,
        ':santa-tracker:compileDebugJavaWithJavac': FROM_CACHE,
        ':santa-tracker:compileDebugKotlin': FROM_CACHE,
        ':santa-tracker:compileDebugShaders': NO_SOURCE,
        ':santa-tracker:compressDebugAssets': FROM_CACHE,
        ':santa-tracker:createDebugApkListingFileRedirect': SUCCESS,
        ':santa-tracker:createDebugCompatibleScreenManifests': SUCCESS,
        ':santa-tracker:dataBindingGenBaseClassesDebug': FROM_CACHE,
        ':santa-tracker:dataBindingMergeDependencyArtifactsDebug': SUCCESS,
        ':santa-tracker:desugarDebugFileDependencies': FROM_CACHE,
        ':santa-tracker:dexBuilderDebug': FROM_CACHE,
        ':santa-tracker:extractDeepLinksDebug': FROM_CACHE,
        ':santa-tracker:generateDebugAssets': UP_TO_DATE,
        ':santa-tracker:generateDebugBuildConfig': FROM_CACHE,
        ':santa-tracker:generateDebugFeatureMetadata': FROM_CACHE,
        ':santa-tracker:generateDebugFeatureTransitiveDeps': SUCCESS,
        ':santa-tracker:generateDebugResValues': FROM_CACHE,
        ':santa-tracker:generateDebugResources': UP_TO_DATE,
        ':santa-tracker:handleDebugMicroApk': FROM_CACHE,
        ':santa-tracker:javaPreCompileDebug': FROM_CACHE,
        ':santa-tracker:kaptDebugKotlin': FROM_CACHE,
        ':santa-tracker:kaptGenerateStubsDebugKotlin': FROM_CACHE,
        ':santa-tracker:mapDebugSourceSetPaths': SUCCESS,
        ':santa-tracker:mergeDebugAssets': FROM_CACHE,
        ':santa-tracker:mergeDebugGlobalSynthetics': FROM_CACHE,
        ':santa-tracker:mergeDebugJavaResource': SUCCESS,
        ':santa-tracker:mergeDebugJniLibFolders': FROM_CACHE,
        ':santa-tracker:mergeDebugNativeDebugMetadata': NO_SOURCE,
        ':santa-tracker:mergeDebugNativeLibs': NO_SOURCE,
        ':santa-tracker:mergeDebugResources': FROM_CACHE,
        ':santa-tracker:mergeDebugShaders': FROM_CACHE,
        ':santa-tracker:mergeExtDexDebug': FROM_CACHE,
        ':santa-tracker:mergeLibDexDebug': FROM_CACHE,
        ':santa-tracker:mergeProjectDexDebug': FROM_CACHE,
        ':santa-tracker:packageDebug': SUCCESS,
        ':santa-tracker:preBuild': UP_TO_DATE,
        ':santa-tracker:preDebugBuild': SUCCESS,
        ':santa-tracker:processDebugJavaRes': SUCCESS,
        ':santa-tracker:processDebugMainManifest': FROM_CACHE,
        ':santa-tracker:processDebugManifest': FROM_CACHE,
        ':santa-tracker:processDebugManifestForPackage': FROM_CACHE,
        ':santa-tracker:processDebugResources': FROM_CACHE,
        ':santa-tracker:signingConfigWriterDebug': SUCCESS,
        ':santa-tracker:stripDebugDebugSymbols': NO_SOURCE,
        ':santa-tracker:validateSigningDebug': SUCCESS,
        ':santa-tracker:writeDebugAppMetadata': SUCCESS,
        ':santa-tracker:writeDebugModuleMetadata': SUCCESS,
        ':santa-tracker:writeDebugSigningConfigVersions': SUCCESS,
        ':snowballrun:assembleDebug': SUCCESS,
        ':snowballrun:checkDebugAarMetadata': SUCCESS,
        ':snowballrun:checkDebugDuplicateClasses': SUCCESS,
        ':snowballrun:compileDebugJavaWithJavac': FROM_CACHE,
        ':snowballrun:compileDebugShaders': NO_SOURCE,
        ':snowballrun:compressDebugAssets': FROM_CACHE,
        ':snowballrun:createDebugApkListingFileRedirect': SUCCESS,
        ':snowballrun:desugarDebugFileDependencies': FROM_CACHE,
        ':snowballrun:dexBuilderDebug': FROM_CACHE,
        ':snowballrun:extractDeepLinksDebug': FROM_CACHE,
        ':snowballrun:featureDebugWriter': SUCCESS,
        ':snowballrun:featureGlobalSyntheticsDebugMerge': SUCCESS,
        ':snowballrun:generateDebugAssets': UP_TO_DATE,
        ':snowballrun:generateDebugFeatureTransitiveDeps': SUCCESS,
        ':snowballrun:generateDebugResValues': FROM_CACHE,
        ':snowballrun:generateDebugResources': UP_TO_DATE,
        ':snowballrun:javaPreCompileDebug': FROM_CACHE,
        ':snowballrun:mapDebugSourceSetPaths': SUCCESS,
        ':snowballrun:mergeDebugAssets': FROM_CACHE,
        ':snowballrun:mergeDebugGlobalSynthetics': FROM_CACHE,
        ':snowballrun:mergeDebugJavaResource': SUCCESS,
        ':snowballrun:mergeDebugJniLibFolders': FROM_CACHE,
        ':snowballrun:mergeDebugNativeLibs': NO_SOURCE,
        ':snowballrun:mergeDebugResources': FROM_CACHE,
        ':snowballrun:mergeDebugShaders': FROM_CACHE,
        ':snowballrun:mergeExtDexDebug': FROM_CACHE,
        ':snowballrun:mergeLibDexDebug': FROM_CACHE,
        ':snowballrun:mergeProjectDexDebug': FROM_CACHE,
        ':snowballrun:packageDebug': SUCCESS,
        ':snowballrun:preBuild': UP_TO_DATE,
        ':snowballrun:preDebugBuild': UP_TO_DATE,
        ':snowballrun:processApplicationManifestDebugForBundle': SUCCESS,
        ':snowballrun:processDebugJavaRes': NO_SOURCE,
        ':snowballrun:processDebugMainManifest': FROM_CACHE,
        ':snowballrun:processDebugManifestForPackage': FROM_CACHE,
        ':snowballrun:processDebugResources': FROM_CACHE,
        ':snowballrun:processManifestDebugForFeature': FROM_CACHE,
        ':snowballrun:stripDebugDebugSymbols': NO_SOURCE,
        ':tracker:assembleDebug': SUCCESS,
        ':tracker:bundleDebugAar': SUCCESS,
        ':tracker:bundleLibCompileToJarDebug': SUCCESS,
        ':tracker:bundleLibRuntimeToJarDebug': SUCCESS,
        ':tracker:checkKotlinGradlePluginConfigurationErrors': SUCCESS,
        ':tracker:compileDebugJavaWithJavac': SUCCESS,
        ':tracker:compileDebugKotlin': FROM_CACHE,
        ':tracker:compileDebugLibraryResources': FROM_CACHE,
        ':tracker:compileDebugShaders': NO_SOURCE,
        ':tracker:copyDebugJniLibsProjectAndLocalJars': SUCCESS,
        ':tracker:copyDebugJniLibsProjectOnly': SUCCESS,
        ':tracker:createFullJarDebug': SUCCESS,
        ':tracker:extractDebugAnnotations': FROM_CACHE,
        ':tracker:extractDeepLinksDebug': FROM_CACHE,
        ':tracker:extractDeepLinksForAarDebug': FROM_CACHE,
        ':tracker:generateDebugAssets': UP_TO_DATE,
        ':tracker:generateDebugRFile': FROM_CACHE,
        ':tracker:generateDebugResValues': FROM_CACHE,
        ':tracker:generateDebugResources': UP_TO_DATE,
        ':tracker:javaPreCompileDebug': FROM_CACHE,
        ':tracker:kaptDebugKotlin': SUCCESS,
        ':tracker:kaptGenerateStubsDebugKotlin': FROM_CACHE,
        ':tracker:mergeDebugConsumerProguardFiles': SUCCESS,
        ':tracker:mergeDebugGeneratedProguardFiles': SUCCESS,
        ':tracker:mergeDebugJavaResource': SUCCESS,
        ':tracker:mergeDebugJniLibFolders': FROM_CACHE,
        ':tracker:mergeDebugNativeLibs': NO_SOURCE,
        ':tracker:mergeDebugShaders': FROM_CACHE,
        ':tracker:packageDebugAssets': FROM_CACHE,
        ':tracker:packageDebugResources': FROM_CACHE,
        ':tracker:parseDebugLocalResources': FROM_CACHE,
        ':tracker:preBuild': UP_TO_DATE,
        ':tracker:preDebugBuild': UP_TO_DATE,
        ':tracker:prepareDebugArtProfile': SUCCESS,
        ':tracker:prepareLintJarForPublish': SUCCESS,
        ':tracker:processDebugJavaRes': SUCCESS,
        ':tracker:processDebugManifest': FROM_CACHE,
        ':tracker:stripDebugDebugSymbols': NO_SOURCE,
        ':tracker:syncDebugLibJars': FROM_CACHE,
        ':tracker:writeDebugAarMetadata': SUCCESS,
        ':wearable:assembleDebug': SUCCESS,
        ':wearable:checkDebugAarMetadata': SUCCESS,
        ':wearable:checkDebugDuplicateClasses': SUCCESS,
        ':wearable:checkKotlinGradlePluginConfigurationErrors': SUCCESS,
        ':wearable:compileDebugJavaWithJavac': FROM_CACHE,
        ':wearable:compileDebugKotlin': FROM_CACHE,
        ':wearable:compileDebugShaders': NO_SOURCE,
        ':wearable:compressDebugAssets': FROM_CACHE,
        ':wearable:createDebugApkListingFileRedirect': SUCCESS,
        ':wearable:createDebugCompatibleScreenManifests': SUCCESS,
        ':wearable:desugarDebugFileDependencies': FROM_CACHE,
        ':wearable:dexBuilderDebug': FROM_CACHE,
        ':wearable:extractDeepLinksDebug': FROM_CACHE,
        ':wearable:generateDebugAssets': UP_TO_DATE,
        ':wearable:generateDebugResValues': FROM_CACHE,
        ':wearable:generateDebugResources': UP_TO_DATE,
        ':wearable:javaPreCompileDebug': FROM_CACHE,
        ':wearable:kaptDebugKotlin': SKIPPED,
        ':wearable:kaptGenerateStubsDebugKotlin': SKIPPED,
        ':wearable:mapDebugSourceSetPaths': SUCCESS,
        ':wearable:mergeDebugAssets': FROM_CACHE,
        ':wearable:mergeDebugGlobalSynthetics': FROM_CACHE,
        ':wearable:mergeDebugJavaResource': SUCCESS,
        ':wearable:mergeDebugJniLibFolders': FROM_CACHE,
        ':wearable:mergeDebugNativeDebugMetadata': NO_SOURCE,
        ':wearable:mergeDebugNativeLibs': NO_SOURCE,
        ':wearable:mergeDebugResources': FROM_CACHE,
        ':wearable:mergeDebugShaders': FROM_CACHE,
        ':wearable:mergeExtDexDebug': FROM_CACHE,
        ':wearable:mergeLibDexDebug': FROM_CACHE,
        ':wearable:mergeProjectDexDebug': FROM_CACHE,
        ':wearable:packageDebug': SUCCESS,
        ':wearable:preBuild': UP_TO_DATE,
        ':wearable:preDebugBuild': UP_TO_DATE,
        ':wearable:processDebugJavaRes': SUCCESS,
        ':wearable:processDebugMainManifest': FROM_CACHE,
        ':wearable:processDebugManifest': FROM_CACHE,
        ':wearable:processDebugManifestForPackage': FROM_CACHE,
        ':wearable:processDebugResources': FROM_CACHE,
        ':wearable:stripDebugDebugSymbols': NO_SOURCE,
        ':wearable:validateSigningDebug': SUCCESS,
        ':wearable:writeDebugAppMetadata': SUCCESS,
        ':wearable:writeDebugSigningConfigVersions': SUCCESS,
    ]

    static final EXPECTED_RESULTS_8_0 = [
        ':cityquiz:assembleDebug': SUCCESS,
        ':cityquiz:checkDebugAarMetadata': SUCCESS,
        ':cityquiz:checkDebugDuplicateClasses': SUCCESS,
        ':cityquiz:checkKotlinGradlePluginConfigurationErrors': SUCCESS,
        ':cityquiz:compileDebugJavaWithJavac': FROM_CACHE,
        ':cityquiz:compileDebugKotlin': FROM_CACHE,
        ':cityquiz:compileDebugShaders': NO_SOURCE,
        ':cityquiz:compressDebugAssets': FROM_CACHE,
        ':cityquiz:createDebugApkListingFileRedirect': SUCCESS,
        ':cityquiz:createDebugCompatibleScreenManifests': SUCCESS,
        ':cityquiz:desugarDebugFileDependencies': FROM_CACHE,
        ':cityquiz:dexBuilderDebug': FROM_CACHE,
        ':cityquiz:extractDeepLinksDebug': FROM_CACHE,
        ':cityquiz:featureDebugWriter': SUCCESS,
        ':cityquiz:featureGlobalSyntheticsDebugMerge': SUCCESS,
        ':cityquiz:generateDebugAssets': UP_TO_DATE,
        ':cityquiz:generateDebugBuildConfig': FROM_CACHE,
        ':cityquiz:generateDebugFeatureTransitiveDeps': SUCCESS,
        ':cityquiz:generateDebugResValues': FROM_CACHE,
        ':cityquiz:generateDebugResources': UP_TO_DATE,
        ':cityquiz:javaPreCompileDebug': FROM_CACHE,
        ':cityquiz:mapDebugSourceSetPaths': SUCCESS,
        ':cityquiz:mergeDebugAssets': FROM_CACHE,
        ':cityquiz:mergeDebugGlobalSynthetics': FROM_CACHE,
        ':cityquiz:mergeDebugJavaResource': SUCCESS,
        ':cityquiz:mergeDebugJniLibFolders': FROM_CACHE,
        ':cityquiz:mergeDebugNativeLibs': NO_SOURCE,
        ':cityquiz:mergeDebugResources': FROM_CACHE,
        ':cityquiz:mergeDebugShaders': FROM_CACHE,
        ':cityquiz:mergeExtDexDebug': FROM_CACHE,
        ':cityquiz:mergeLibDexDebug': FROM_CACHE,
        ':cityquiz:mergeProjectDexDebug': FROM_CACHE,
        ':cityquiz:packageDebug': SUCCESS,
        ':cityquiz:preBuild': UP_TO_DATE,
        ':cityquiz:preDebugBuild': UP_TO_DATE,
        ':cityquiz:processApplicationManifestDebugForBundle': SUCCESS,
        ':cityquiz:processDebugJavaRes': NO_SOURCE,
        ':cityquiz:processDebugMainManifest': FROM_CACHE,
        ':cityquiz:processDebugManifest': FROM_CACHE,
        ':cityquiz:processDebugManifestForPackage': FROM_CACHE,
        ':cityquiz:processDebugResources': FROM_CACHE,
        ':cityquiz:processManifestDebugForFeature': FROM_CACHE,
        ':cityquiz:stripDebugDebugSymbols': NO_SOURCE,
        ':common:assembleDebug': SUCCESS,
        ':common:bundleDebugAar': SUCCESS,
        ':common:bundleLibCompileToJarDebug': SUCCESS,
        ':common:bundleLibResDebug': SUCCESS,
        ':common:bundleLibRuntimeToJarDebug': SUCCESS,
        ':common:checkKotlinGradlePluginConfigurationErrors': SUCCESS,
        ':common:compileDebugJavaWithJavac': FROM_CACHE,
        ':common:compileDebugKotlin': FROM_CACHE,
        ':common:compileDebugLibraryResources': FROM_CACHE,
        ':common:compileDebugShaders': NO_SOURCE,
        ':common:copyDebugJniLibsProjectAndLocalJars': SUCCESS,
        ':common:copyDebugJniLibsProjectOnly': SUCCESS,
        ':common:createFullJarDebug': FROM_CACHE,
        ':common:dataBindingGenBaseClassesDebug': FROM_CACHE,
        ':common:dataBindingMergeDependencyArtifactsDebug': SUCCESS,
        ':common:extractDebugAnnotations': FROM_CACHE,
        ':common:extractDeepLinksDebug': FROM_CACHE,
        ':common:extractDeepLinksForAarDebug': FROM_CACHE,
        ':common:generateDebugAssets': UP_TO_DATE,
        ':common:generateDebugBuildConfig': FROM_CACHE,
        ':common:generateDebugRFile': FROM_CACHE,
        ':common:generateDebugResValues': FROM_CACHE,
        ':common:generateDebugResources': UP_TO_DATE,
        ':common:javaPreCompileDebug': FROM_CACHE,
        ':common:mergeDebugConsumerProguardFiles': SUCCESS,
        ':common:mergeDebugGeneratedProguardFiles': SUCCESS,
        ':common:mergeDebugJavaResource': SUCCESS,
        ':common:mergeDebugJniLibFolders': FROM_CACHE,
        ':common:mergeDebugNativeLibs': NO_SOURCE,
        ':common:mergeDebugShaders': FROM_CACHE,
        ':common:packageDebugAssets': FROM_CACHE,
        ':common:packageDebugResources': FROM_CACHE,
        ':common:parseDebugLocalResources': FROM_CACHE,
        ':common:preBuild': UP_TO_DATE,
        ':common:preDebugBuild': UP_TO_DATE,
        ':common:prepareDebugArtProfile': SUCCESS,
        ':common:prepareLintJarForPublish': SUCCESS,
        ':common:processDebugJavaRes': NO_SOURCE,
        ':common:processDebugManifest': FROM_CACHE,
        ':common:stripDebugDebugSymbols': NO_SOURCE,
        ':common:syncDebugLibJars': FROM_CACHE,
        ':common:writeDebugAarMetadata': SUCCESS,
        ':dasherdancer:assembleDebug': SUCCESS,
        ':dasherdancer:checkDebugAarMetadata': SUCCESS,
        ':dasherdancer:checkDebugDuplicateClasses': SUCCESS,
        ':dasherdancer:checkKotlinGradlePluginConfigurationErrors': SUCCESS,
        ':dasherdancer:compileDebugJavaWithJavac': NO_SOURCE,
        ':dasherdancer:compileDebugKotlin': FROM_CACHE,
        ':dasherdancer:compileDebugShaders': NO_SOURCE,
        ':dasherdancer:compressDebugAssets': FROM_CACHE,
        ':dasherdancer:createDebugApkListingFileRedirect': SUCCESS,
        ':dasherdancer:createDebugCompatibleScreenManifests': SUCCESS,
        ':dasherdancer:desugarDebugFileDependencies': FROM_CACHE,
        ':dasherdancer:dexBuilderDebug': FROM_CACHE,
        ':dasherdancer:extractDeepLinksDebug': FROM_CACHE,
        ':dasherdancer:featureDebugWriter': SUCCESS,
        ':dasherdancer:featureGlobalSyntheticsDebugMerge': SUCCESS,
        ':dasherdancer:generateDebugAssets': UP_TO_DATE,
        ':dasherdancer:generateDebugFeatureTransitiveDeps': SUCCESS,
        ':dasherdancer:generateDebugResValues': FROM_CACHE,
        ':dasherdancer:generateDebugResources': UP_TO_DATE,
        ':dasherdancer:javaPreCompileDebug': FROM_CACHE,
        ':dasherdancer:mapDebugSourceSetPaths': SUCCESS,
        ':dasherdancer:mergeDebugAssets': FROM_CACHE,
        ':dasherdancer:mergeDebugGlobalSynthetics': FROM_CACHE,
        ':dasherdancer:mergeDebugJavaResource': SUCCESS,
        ':dasherdancer:mergeDebugJniLibFolders': FROM_CACHE,
        ':dasherdancer:mergeDebugNativeLibs': NO_SOURCE,
        ':dasherdancer:mergeDebugResources': FROM_CACHE,
        ':dasherdancer:mergeDebugShaders': FROM_CACHE,
        ':dasherdancer:mergeExtDexDebug': FROM_CACHE,
        ':dasherdancer:mergeLibDexDebug': FROM_CACHE,
        ':dasherdancer:mergeProjectDexDebug': FROM_CACHE,
        ':dasherdancer:packageDebug': SUCCESS,
        ':dasherdancer:preBuild': UP_TO_DATE,
        ':dasherdancer:preDebugBuild': UP_TO_DATE,
        ':dasherdancer:processApplicationManifestDebugForBundle': SUCCESS,
        ':dasherdancer:processDebugJavaRes': NO_SOURCE,
        ':dasherdancer:processDebugMainManifest': FROM_CACHE,
        ':dasherdancer:processDebugManifest': FROM_CACHE,
        ':dasherdancer:processDebugManifestForPackage': FROM_CACHE,
        ':dasherdancer:processDebugResources': FROM_CACHE,
        ':dasherdancer:processManifestDebugForFeature': FROM_CACHE,
        ':dasherdancer:stripDebugDebugSymbols': NO_SOURCE,
        ':doodles-lib:assembleDebug': SUCCESS,
        ':doodles-lib:bundleDebugAar': SUCCESS,
        ':doodles-lib:bundleLibCompileToJarDebug': SUCCESS,
        ':doodles-lib:bundleLibResDebug': NO_SOURCE,
        ':doodles-lib:bundleLibRuntimeToJarDebug': SUCCESS,
        ':doodles-lib:compileDebugJavaWithJavac': FROM_CACHE,
        ':doodles-lib:compileDebugLibraryResources': FROM_CACHE,
        ':doodles-lib:compileDebugShaders': NO_SOURCE,
        ':doodles-lib:copyDebugJniLibsProjectAndLocalJars': SUCCESS,
        ':doodles-lib:copyDebugJniLibsProjectOnly': SUCCESS,
        ':doodles-lib:createFullJarDebug': FROM_CACHE,
        ':doodles-lib:extractDebugAnnotations': FROM_CACHE,
        ':doodles-lib:extractDeepLinksDebug': FROM_CACHE,
        ':doodles-lib:extractDeepLinksForAarDebug': FROM_CACHE,
        ':doodles-lib:generateDebugAssets': UP_TO_DATE,
        ':doodles-lib:generateDebugRFile': FROM_CACHE,
        ':doodles-lib:generateDebugResValues': FROM_CACHE,
        ':doodles-lib:generateDebugResources': UP_TO_DATE,
        ':doodles-lib:javaPreCompileDebug': FROM_CACHE,
        ':doodles-lib:mergeDebugConsumerProguardFiles': SUCCESS,
        ':doodles-lib:mergeDebugGeneratedProguardFiles': SUCCESS,
        ':doodles-lib:mergeDebugJavaResource': SUCCESS,
        ':doodles-lib:mergeDebugJniLibFolders': FROM_CACHE,
        ':doodles-lib:mergeDebugNativeLibs': NO_SOURCE,
        ':doodles-lib:mergeDebugShaders': FROM_CACHE,
        ':doodles-lib:packageDebugAssets': FROM_CACHE,
        ':doodles-lib:packageDebugResources': FROM_CACHE,
        ':doodles-lib:parseDebugLocalResources': FROM_CACHE,
        ':doodles-lib:preBuild': UP_TO_DATE,
        ':doodles-lib:preDebugBuild': UP_TO_DATE,
        ':doodles-lib:prepareDebugArtProfile': SUCCESS,
        ':doodles-lib:prepareLintJarForPublish': SUCCESS,
        ':doodles-lib:processDebugJavaRes': NO_SOURCE,
        ':doodles-lib:processDebugManifest': FROM_CACHE,
        ':doodles-lib:stripDebugDebugSymbols': NO_SOURCE,
        ':doodles-lib:syncDebugLibJars': FROM_CACHE,
        ':doodles-lib:writeDebugAarMetadata': SUCCESS,
        ':gumball:assembleDebug': SUCCESS,
        ':gumball:checkDebugAarMetadata': SUCCESS,
        ':gumball:checkDebugDuplicateClasses': SUCCESS,
        ':gumball:compileDebugJavaWithJavac': FROM_CACHE,
        ':gumball:compileDebugShaders': NO_SOURCE,
        ':gumball:compressDebugAssets': FROM_CACHE,
        ':gumball:createDebugApkListingFileRedirect': SUCCESS,
        ':gumball:createDebugCompatibleScreenManifests': SUCCESS,
        ':gumball:desugarDebugFileDependencies': FROM_CACHE,
        ':gumball:dexBuilderDebug': FROM_CACHE,
        ':gumball:extractDeepLinksDebug': FROM_CACHE,
        ':gumball:featureDebugWriter': SUCCESS,
        ':gumball:featureGlobalSyntheticsDebugMerge': SUCCESS,
        ':gumball:generateDebugAssets': UP_TO_DATE,
        ':gumball:generateDebugFeatureTransitiveDeps': SUCCESS,
        ':gumball:generateDebugResValues': FROM_CACHE,
        ':gumball:generateDebugResources': UP_TO_DATE,
        ':gumball:javaPreCompileDebug': FROM_CACHE,
        ':gumball:mapDebugSourceSetPaths': SUCCESS,
        ':gumball:mergeDebugAssets': FROM_CACHE,
        ':gumball:mergeDebugGlobalSynthetics': FROM_CACHE,
        ':gumball:mergeDebugJavaResource': SUCCESS,
        ':gumball:mergeDebugJniLibFolders': FROM_CACHE,
        ':gumball:mergeDebugNativeLibs': NO_SOURCE,
        ':gumball:mergeDebugResources': FROM_CACHE,
        ':gumball:mergeDebugShaders': FROM_CACHE,
        ':gumball:mergeExtDexDebug': FROM_CACHE,
        ':gumball:mergeLibDexDebug': FROM_CACHE,
        ':gumball:mergeProjectDexDebug': FROM_CACHE,
        ':gumball:packageDebug': SUCCESS,
        ':gumball:preBuild': UP_TO_DATE,
        ':gumball:preDebugBuild': UP_TO_DATE,
        ':gumball:processApplicationManifestDebugForBundle': SUCCESS,
        ':gumball:processDebugJavaRes': NO_SOURCE,
        ':gumball:processDebugMainManifest': FROM_CACHE,
        ':gumball:processDebugManifest': FROM_CACHE,
        ':gumball:processDebugManifestForPackage': FROM_CACHE,
        ':gumball:processDebugResources': FROM_CACHE,
        ':gumball:processManifestDebugForFeature': FROM_CACHE,
        ':gumball:stripDebugDebugSymbols': NO_SOURCE,
        ':jetpack:assembleDebug': SUCCESS,
        ':jetpack:checkDebugAarMetadata': SUCCESS,
        ':jetpack:checkDebugDuplicateClasses': SUCCESS,
        ':jetpack:checkKotlinGradlePluginConfigurationErrors': SUCCESS,
        ':jetpack:compileDebugJavaWithJavac': NO_SOURCE,
        ':jetpack:compileDebugKotlin': FROM_CACHE,
        ':jetpack:compileDebugShaders': NO_SOURCE,
        ':jetpack:compressDebugAssets': FROM_CACHE,
        ':jetpack:createDebugApkListingFileRedirect': SUCCESS,
        ':jetpack:createDebugCompatibleScreenManifests': SUCCESS,
        ':jetpack:desugarDebugFileDependencies': FROM_CACHE,
        ':jetpack:dexBuilderDebug': FROM_CACHE,
        ':jetpack:extractDeepLinksDebug': FROM_CACHE,
        ':jetpack:featureDebugWriter': SUCCESS,
        ':jetpack:featureGlobalSyntheticsDebugMerge': SUCCESS,
        ':jetpack:generateDebugAssets': UP_TO_DATE,
        ':jetpack:generateDebugFeatureTransitiveDeps': SUCCESS,
        ':jetpack:generateDebugResValues': FROM_CACHE,
        ':jetpack:generateDebugResources': UP_TO_DATE,
        ':jetpack:javaPreCompileDebug': FROM_CACHE,
        ':jetpack:mapDebugSourceSetPaths': SUCCESS,
        ':jetpack:mergeDebugAssets': FROM_CACHE,
        ':jetpack:mergeDebugGlobalSynthetics': FROM_CACHE,
        ':jetpack:mergeDebugJavaResource': SUCCESS,
        ':jetpack:mergeDebugJniLibFolders': FROM_CACHE,
        ':jetpack:mergeDebugNativeLibs': NO_SOURCE,
        ':jetpack:mergeDebugResources': FROM_CACHE,
        ':jetpack:mergeDebugShaders': FROM_CACHE,
        ':jetpack:mergeExtDexDebug': FROM_CACHE,
        ':jetpack:mergeLibDexDebug': FROM_CACHE,
        ':jetpack:mergeProjectDexDebug': FROM_CACHE,
        ':jetpack:packageDebug': SUCCESS,
        ':jetpack:preBuild': UP_TO_DATE,
        ':jetpack:preDebugBuild': UP_TO_DATE,
        ':jetpack:processApplicationManifestDebugForBundle': SUCCESS,
        ':jetpack:processDebugJavaRes': NO_SOURCE,
        ':jetpack:processDebugMainManifest': FROM_CACHE,
        ':jetpack:processDebugManifest': FROM_CACHE,
        ':jetpack:processDebugManifestForPackage': FROM_CACHE,
        ':jetpack:processDebugResources': FROM_CACHE,
        ':jetpack:processManifestDebugForFeature': FROM_CACHE,
        ':jetpack:stripDebugDebugSymbols': NO_SOURCE,
        ':memory:assembleDebug': SUCCESS,
        ':memory:checkDebugAarMetadata': SUCCESS,
        ':memory:checkDebugDuplicateClasses': SUCCESS,
        ':memory:compileDebugJavaWithJavac': FROM_CACHE,
        ':memory:compileDebugShaders': NO_SOURCE,
        ':memory:compressDebugAssets': FROM_CACHE,
        ':memory:createDebugApkListingFileRedirect': SUCCESS,
        ':memory:createDebugCompatibleScreenManifests': SUCCESS,
        ':memory:desugarDebugFileDependencies': FROM_CACHE,
        ':memory:dexBuilderDebug': FROM_CACHE,
        ':memory:extractDeepLinksDebug': FROM_CACHE,
        ':memory:featureDebugWriter': SUCCESS,
        ':memory:featureGlobalSyntheticsDebugMerge': SUCCESS,
        ':memory:generateDebugAssets': UP_TO_DATE,
        ':memory:generateDebugFeatureTransitiveDeps': SUCCESS,
        ':memory:generateDebugResValues': FROM_CACHE,
        ':memory:generateDebugResources': UP_TO_DATE,
        ':memory:javaPreCompileDebug': FROM_CACHE,
        ':memory:mapDebugSourceSetPaths': SUCCESS,
        ':memory:mergeDebugAssets': FROM_CACHE,
        ':memory:mergeDebugGlobalSynthetics': FROM_CACHE,
        ':memory:mergeDebugJavaResource': SUCCESS,
        ':memory:mergeDebugJniLibFolders': FROM_CACHE,
        ':memory:mergeDebugNativeLibs': NO_SOURCE,
        ':memory:mergeDebugResources': FROM_CACHE,
        ':memory:mergeDebugShaders': FROM_CACHE,
        ':memory:mergeExtDexDebug': FROM_CACHE,
        ':memory:mergeLibDexDebug': FROM_CACHE,
        ':memory:mergeProjectDexDebug': FROM_CACHE,
        ':memory:packageDebug': SUCCESS,
        ':memory:preBuild': UP_TO_DATE,
        ':memory:preDebugBuild': UP_TO_DATE,
        ':memory:processApplicationManifestDebugForBundle': SUCCESS,
        ':memory:processDebugJavaRes': NO_SOURCE,
        ':memory:processDebugMainManifest': FROM_CACHE,
        ':memory:processDebugManifest': FROM_CACHE,
        ':memory:processDebugManifestForPackage': FROM_CACHE,
        ':memory:processDebugResources': FROM_CACHE,
        ':memory:processManifestDebugForFeature': FROM_CACHE,
        ':memory:stripDebugDebugSymbols': NO_SOURCE,
        ':penguinswim:assembleDebug': SUCCESS,
        ':penguinswim:checkDebugAarMetadata': SUCCESS,
        ':penguinswim:checkDebugDuplicateClasses': SUCCESS,
        ':penguinswim:compileDebugJavaWithJavac': FROM_CACHE,
        ':penguinswim:compileDebugShaders': NO_SOURCE,
        ':penguinswim:compressDebugAssets': FROM_CACHE,
        ':penguinswim:createDebugApkListingFileRedirect': SUCCESS,
        ':penguinswim:createDebugCompatibleScreenManifests': SUCCESS,
        ':penguinswim:desugarDebugFileDependencies': FROM_CACHE,
        ':penguinswim:dexBuilderDebug': FROM_CACHE,
        ':penguinswim:extractDeepLinksDebug': FROM_CACHE,
        ':penguinswim:featureDebugWriter': SUCCESS,
        ':penguinswim:featureGlobalSyntheticsDebugMerge': SUCCESS,
        ':penguinswim:generateDebugAssets': UP_TO_DATE,
        ':penguinswim:generateDebugFeatureTransitiveDeps': SUCCESS,
        ':penguinswim:generateDebugResValues': FROM_CACHE,
        ':penguinswim:generateDebugResources': UP_TO_DATE,
        ':penguinswim:javaPreCompileDebug': FROM_CACHE,
        ':penguinswim:mapDebugSourceSetPaths': SUCCESS,
        ':penguinswim:mergeDebugAssets': FROM_CACHE,
        ':penguinswim:mergeDebugGlobalSynthetics': FROM_CACHE,
        ':penguinswim:mergeDebugJavaResource': SUCCESS,
        ':penguinswim:mergeDebugJniLibFolders': FROM_CACHE,
        ':penguinswim:mergeDebugNativeLibs': NO_SOURCE,
        ':penguinswim:mergeDebugResources': FROM_CACHE,
        ':penguinswim:mergeDebugShaders': FROM_CACHE,
        ':penguinswim:mergeExtDexDebug': FROM_CACHE,
        ':penguinswim:mergeLibDexDebug': FROM_CACHE,
        ':penguinswim:mergeProjectDexDebug': FROM_CACHE,
        ':penguinswim:packageDebug': SUCCESS,
        ':penguinswim:preBuild': UP_TO_DATE,
        ':penguinswim:preDebugBuild': UP_TO_DATE,
        ':penguinswim:processApplicationManifestDebugForBundle': SUCCESS,
        ':penguinswim:processDebugJavaRes': NO_SOURCE,
        ':penguinswim:processDebugMainManifest': FROM_CACHE,
        ':penguinswim:processDebugManifest': FROM_CACHE,
        ':penguinswim:processDebugManifestForPackage': FROM_CACHE,
        ':penguinswim:processDebugResources': FROM_CACHE,
        ':penguinswim:processManifestDebugForFeature': FROM_CACHE,
        ':penguinswim:stripDebugDebugSymbols': NO_SOURCE,
        ':playgames:assembleDebug': SUCCESS,
        ':playgames:bundleDebugAar': SUCCESS,
        ':playgames:bundleLibCompileToJarDebug': SUCCESS,
        ':playgames:bundleLibResDebug': NO_SOURCE,
        ':playgames:bundleLibRuntimeToJarDebug': SUCCESS,
        ':playgames:compileDebugJavaWithJavac': FROM_CACHE,
        ':playgames:compileDebugLibraryResources': FROM_CACHE,
        ':playgames:compileDebugShaders': NO_SOURCE,
        ':playgames:copyDebugJniLibsProjectAndLocalJars': SUCCESS,
        ':playgames:copyDebugJniLibsProjectOnly': SUCCESS,
        ':playgames:createFullJarDebug': FROM_CACHE,
        ':playgames:extractDebugAnnotations': FROM_CACHE,
        ':playgames:extractDeepLinksDebug': FROM_CACHE,
        ':playgames:extractDeepLinksForAarDebug': FROM_CACHE,
        ':playgames:generateDebugAssets': UP_TO_DATE,
        ':playgames:generateDebugRFile': FROM_CACHE,
        ':playgames:generateDebugResValues': FROM_CACHE,
        ':playgames:generateDebugResources': UP_TO_DATE,
        ':playgames:javaPreCompileDebug': FROM_CACHE,
        ':playgames:mergeDebugConsumerProguardFiles': SUCCESS,
        ':playgames:mergeDebugGeneratedProguardFiles': SUCCESS,
        ':playgames:mergeDebugJavaResource': SUCCESS,
        ':playgames:mergeDebugJniLibFolders': FROM_CACHE,
        ':playgames:mergeDebugNativeLibs': NO_SOURCE,
        ':playgames:mergeDebugShaders': FROM_CACHE,
        ':playgames:packageDebugAssets': FROM_CACHE,
        ':playgames:packageDebugResources': FROM_CACHE,
        ':playgames:parseDebugLocalResources': FROM_CACHE,
        ':playgames:preBuild': UP_TO_DATE,
        ':playgames:preDebugBuild': UP_TO_DATE,
        ':playgames:prepareDebugArtProfile': SUCCESS,
        ':playgames:prepareLintJarForPublish': SUCCESS,
        ':playgames:processDebugJavaRes': NO_SOURCE,
        ':playgames:processDebugManifest': FROM_CACHE,
        ':playgames:stripDebugDebugSymbols': NO_SOURCE,
        ':playgames:syncDebugLibJars': FROM_CACHE,
        ':playgames:writeDebugAarMetadata': SUCCESS,
        ':presenttoss:assembleDebug': SUCCESS,
        ':presenttoss:checkDebugAarMetadata': SUCCESS,
        ':presenttoss:checkDebugDuplicateClasses': SUCCESS,
        ':presenttoss:compileDebugJavaWithJavac': FROM_CACHE,
        ':presenttoss:compileDebugShaders': NO_SOURCE,
        ':presenttoss:compressDebugAssets': FROM_CACHE,
        ':presenttoss:createDebugApkListingFileRedirect': SUCCESS,
        ':presenttoss:createDebugCompatibleScreenManifests': SUCCESS,
        ':presenttoss:desugarDebugFileDependencies': FROM_CACHE,
        ':presenttoss:dexBuilderDebug': FROM_CACHE,
        ':presenttoss:extractDeepLinksDebug': FROM_CACHE,
        ':presenttoss:featureDebugWriter': SUCCESS,
        ':presenttoss:featureGlobalSyntheticsDebugMerge': SUCCESS,
        ':presenttoss:generateDebugAssets': UP_TO_DATE,
        ':presenttoss:generateDebugFeatureTransitiveDeps': SUCCESS,
        ':presenttoss:generateDebugResValues': FROM_CACHE,
        ':presenttoss:generateDebugResources': UP_TO_DATE,
        ':presenttoss:javaPreCompileDebug': FROM_CACHE,
        ':presenttoss:mapDebugSourceSetPaths': SUCCESS,
        ':presenttoss:mergeDebugAssets': FROM_CACHE,
        ':presenttoss:mergeDebugGlobalSynthetics': FROM_CACHE,
        ':presenttoss:mergeDebugJavaResource': SUCCESS,
        ':presenttoss:mergeDebugJniLibFolders': FROM_CACHE,
        ':presenttoss:mergeDebugNativeLibs': NO_SOURCE,
        ':presenttoss:mergeDebugResources': FROM_CACHE,
        ':presenttoss:mergeDebugShaders': FROM_CACHE,
        ':presenttoss:mergeExtDexDebug': FROM_CACHE,
        ':presenttoss:mergeLibDexDebug': FROM_CACHE,
        ':presenttoss:mergeProjectDexDebug': FROM_CACHE,
        ':presenttoss:packageDebug': SUCCESS,
        ':presenttoss:preBuild': UP_TO_DATE,
        ':presenttoss:preDebugBuild': UP_TO_DATE,
        ':presenttoss:processApplicationManifestDebugForBundle': SUCCESS,
        ':presenttoss:processDebugJavaRes': NO_SOURCE,
        ':presenttoss:processDebugMainManifest': FROM_CACHE,
        ':presenttoss:processDebugManifest': FROM_CACHE,
        ':presenttoss:processDebugManifestForPackage': FROM_CACHE,
        ':presenttoss:processDebugResources': FROM_CACHE,
        ':presenttoss:processManifestDebugForFeature': FROM_CACHE,
        ':presenttoss:stripDebugDebugSymbols': NO_SOURCE,
        ':rocketsleigh:assembleDebug': SUCCESS,
        ':rocketsleigh:checkDebugAarMetadata': SUCCESS,
        ':rocketsleigh:checkDebugDuplicateClasses': SUCCESS,
        ':rocketsleigh:checkKotlinGradlePluginConfigurationErrors': SUCCESS,
        ':rocketsleigh:compileDebugJavaWithJavac': FROM_CACHE,
        ':rocketsleigh:compileDebugKotlin': FROM_CACHE,
        ':rocketsleigh:compileDebugShaders': NO_SOURCE,
        ':rocketsleigh:compressDebugAssets': FROM_CACHE,
        ':rocketsleigh:createDebugApkListingFileRedirect': SUCCESS,
        ':rocketsleigh:createDebugCompatibleScreenManifests': SUCCESS,
        ':rocketsleigh:desugarDebugFileDependencies': FROM_CACHE,
        ':rocketsleigh:dexBuilderDebug': FROM_CACHE,
        ':rocketsleigh:extractDeepLinksDebug': FROM_CACHE,
        ':rocketsleigh:featureDebugWriter': SUCCESS,
        ':rocketsleigh:featureGlobalSyntheticsDebugMerge': SUCCESS,
        ':rocketsleigh:generateDebugAssets': UP_TO_DATE,
        ':rocketsleigh:generateDebugBuildConfig': FROM_CACHE,
        ':rocketsleigh:generateDebugFeatureTransitiveDeps': SUCCESS,
        ':rocketsleigh:generateDebugResValues': FROM_CACHE,
        ':rocketsleigh:generateDebugResources': UP_TO_DATE,
        ':rocketsleigh:javaPreCompileDebug': FROM_CACHE,
        ':rocketsleigh:mapDebugSourceSetPaths': SUCCESS,
        ':rocketsleigh:mergeDebugAssets': FROM_CACHE,
        ':rocketsleigh:mergeDebugGlobalSynthetics': FROM_CACHE,
        ':rocketsleigh:mergeDebugJavaResource': SUCCESS,
        ':rocketsleigh:mergeDebugJniLibFolders': FROM_CACHE,
        ':rocketsleigh:mergeDebugNativeLibs': NO_SOURCE,
        ':rocketsleigh:mergeDebugResources': FROM_CACHE,
        ':rocketsleigh:mergeDebugShaders': FROM_CACHE,
        ':rocketsleigh:mergeExtDexDebug': FROM_CACHE,
        ':rocketsleigh:mergeLibDexDebug': FROM_CACHE,
        ':rocketsleigh:mergeProjectDexDebug': FROM_CACHE,
        ':rocketsleigh:packageDebug': SUCCESS,
        ':rocketsleigh:preBuild': UP_TO_DATE,
        ':rocketsleigh:preDebugBuild': UP_TO_DATE,
        ':rocketsleigh:processApplicationManifestDebugForBundle': SUCCESS,
        ':rocketsleigh:processDebugJavaRes': NO_SOURCE,
        ':rocketsleigh:processDebugMainManifest': FROM_CACHE,
        ':rocketsleigh:processDebugManifest': FROM_CACHE,
        ':rocketsleigh:processDebugManifestForPackage': FROM_CACHE,
        ':rocketsleigh:processDebugResources': FROM_CACHE,
        ':rocketsleigh:processManifestDebugForFeature': FROM_CACHE,
        ':rocketsleigh:stripDebugDebugSymbols': NO_SOURCE,
        ':santa-tracker:assembleDebug': SUCCESS,
        ':santa-tracker:bundleDebugClassesToCompileJar': SUCCESS,
        ':santa-tracker:bundleDebugClassesToRuntimeJar': SUCCESS,
        ':santa-tracker:checkDebugAarMetadata': SUCCESS,
        ':santa-tracker:checkDebugDuplicateClasses': SUCCESS,
        ':santa-tracker:checkDebugLibraries': SUCCESS,
        ':santa-tracker:checkKotlinGradlePluginConfigurationErrors': SUCCESS,
        ':santa-tracker:compileDebugJavaWithJavac': FROM_CACHE,
        ':santa-tracker:compileDebugKotlin': FROM_CACHE,
        ':santa-tracker:compileDebugShaders': NO_SOURCE,
        ':santa-tracker:compressDebugAssets': FROM_CACHE,
        ':santa-tracker:createDebugApkListingFileRedirect': SUCCESS,
        ':santa-tracker:createDebugCompatibleScreenManifests': SUCCESS,
        ':santa-tracker:dataBindingGenBaseClassesDebug': FROM_CACHE,
        ':santa-tracker:dataBindingMergeDependencyArtifactsDebug': SUCCESS,
        ':santa-tracker:desugarDebugFileDependencies': FROM_CACHE,
        ':santa-tracker:dexBuilderDebug': FROM_CACHE,
        ':santa-tracker:extractDeepLinksDebug': FROM_CACHE,
        ':santa-tracker:generateDebugAssets': UP_TO_DATE,
        ':santa-tracker:generateDebugBuildConfig': FROM_CACHE,
        ':santa-tracker:generateDebugFeatureMetadata': FROM_CACHE,
        ':santa-tracker:generateDebugFeatureTransitiveDeps': SUCCESS,
        ':santa-tracker:generateDebugResValues': FROM_CACHE,
        ':santa-tracker:generateDebugResources': UP_TO_DATE,
        ':santa-tracker:handleDebugMicroApk': FROM_CACHE,
        ':santa-tracker:javaPreCompileDebug': FROM_CACHE,
        ':santa-tracker:kaptDebugKotlin': FROM_CACHE,
        ':santa-tracker:kaptGenerateStubsDebugKotlin': FROM_CACHE,
        ':santa-tracker:mapDebugSourceSetPaths': SUCCESS,
        ':santa-tracker:mergeDebugAssets': FROM_CACHE,
        ':santa-tracker:mergeDebugGlobalSynthetics': FROM_CACHE,
        ':santa-tracker:mergeDebugJavaResource': SUCCESS,
        ':santa-tracker:mergeDebugJniLibFolders': FROM_CACHE,
        ':santa-tracker:mergeDebugNativeDebugMetadata': NO_SOURCE,
        ':santa-tracker:mergeDebugNativeLibs': NO_SOURCE,
        ':santa-tracker:mergeDebugResources': FROM_CACHE,
        ':santa-tracker:mergeDebugShaders': FROM_CACHE,
        ':santa-tracker:mergeExtDexDebug': FROM_CACHE,
        ':santa-tracker:mergeLibDexDebug': FROM_CACHE,
        ':santa-tracker:mergeProjectDexDebug': FROM_CACHE,
        ':santa-tracker:packageDebug': SUCCESS,
        ':santa-tracker:preBuild': UP_TO_DATE,
        ':santa-tracker:preDebugBuild': SUCCESS,
        ':santa-tracker:processDebugJavaRes': NO_SOURCE,
        ':santa-tracker:processDebugMainManifest': FROM_CACHE,
        ':santa-tracker:processDebugManifest': FROM_CACHE,
        ':santa-tracker:processDebugManifestForPackage': FROM_CACHE,
        ':santa-tracker:processDebugResources': FROM_CACHE,
        ':santa-tracker:signingConfigWriterDebug': SUCCESS,
        ':santa-tracker:stripDebugDebugSymbols': NO_SOURCE,
        ':santa-tracker:validateSigningDebug': SUCCESS,
        ':santa-tracker:writeDebugAppMetadata': SUCCESS,
        ':santa-tracker:writeDebugModuleMetadata': SUCCESS,
        ':santa-tracker:writeDebugSigningConfigVersions': SUCCESS,
        ':snowballrun:assembleDebug': SUCCESS,
        ':snowballrun:checkDebugAarMetadata': SUCCESS,
        ':snowballrun:checkDebugDuplicateClasses': SUCCESS,
        ':snowballrun:compileDebugJavaWithJavac': FROM_CACHE,
        ':snowballrun:compileDebugShaders': NO_SOURCE,
        ':snowballrun:compressDebugAssets': FROM_CACHE,
        ':snowballrun:createDebugApkListingFileRedirect': SUCCESS,
        ':snowballrun:createDebugCompatibleScreenManifests': SUCCESS,
        ':snowballrun:desugarDebugFileDependencies': FROM_CACHE,
        ':snowballrun:dexBuilderDebug': FROM_CACHE,
        ':snowballrun:extractDeepLinksDebug': FROM_CACHE,
        ':snowballrun:featureDebugWriter': SUCCESS,
        ':snowballrun:featureGlobalSyntheticsDebugMerge': SUCCESS,
        ':snowballrun:generateDebugAssets': UP_TO_DATE,
        ':snowballrun:generateDebugFeatureTransitiveDeps': SUCCESS,
        ':snowballrun:generateDebugResValues': FROM_CACHE,
        ':snowballrun:generateDebugResources': UP_TO_DATE,
        ':snowballrun:javaPreCompileDebug': FROM_CACHE,
        ':snowballrun:mapDebugSourceSetPaths': SUCCESS,
        ':snowballrun:mergeDebugAssets': FROM_CACHE,
        ':snowballrun:mergeDebugGlobalSynthetics': FROM_CACHE,
        ':snowballrun:mergeDebugJavaResource': SUCCESS,
        ':snowballrun:mergeDebugJniLibFolders': FROM_CACHE,
        ':snowballrun:mergeDebugNativeLibs': NO_SOURCE,
        ':snowballrun:mergeDebugResources': FROM_CACHE,
        ':snowballrun:mergeDebugShaders': FROM_CACHE,
        ':snowballrun:mergeExtDexDebug': FROM_CACHE,
        ':snowballrun:mergeLibDexDebug': FROM_CACHE,
        ':snowballrun:mergeProjectDexDebug': FROM_CACHE,
        ':snowballrun:packageDebug': SUCCESS,
        ':snowballrun:preBuild': UP_TO_DATE,
        ':snowballrun:preDebugBuild': UP_TO_DATE,
        ':snowballrun:processApplicationManifestDebugForBundle': SUCCESS,
        ':snowballrun:processDebugJavaRes': NO_SOURCE,
        ':snowballrun:processDebugMainManifest': FROM_CACHE,
        ':snowballrun:processDebugManifest': FROM_CACHE,
        ':snowballrun:processDebugManifestForPackage': FROM_CACHE,
        ':snowballrun:processDebugResources': FROM_CACHE,
        ':snowballrun:processManifestDebugForFeature': FROM_CACHE,
        ':snowballrun:stripDebugDebugSymbols': NO_SOURCE,
        ':tracker:assembleDebug': SUCCESS,
        ':tracker:bundleDebugAar': SUCCESS,
        ':tracker:bundleLibCompileToJarDebug': SUCCESS,
        ':tracker:bundleLibResDebug': SUCCESS,
        ':tracker:bundleLibRuntimeToJarDebug': SUCCESS,
        ':tracker:checkKotlinGradlePluginConfigurationErrors': SUCCESS,
        ':tracker:compileDebugJavaWithJavac': SUCCESS,
        ':tracker:compileDebugKotlin': FROM_CACHE,
        ':tracker:compileDebugLibraryResources': FROM_CACHE,
        ':tracker:compileDebugShaders': NO_SOURCE,
        ':tracker:copyDebugJniLibsProjectAndLocalJars': SUCCESS,
        ':tracker:copyDebugJniLibsProjectOnly': SUCCESS,
        ':tracker:createFullJarDebug': FROM_CACHE,
        ':tracker:extractDebugAnnotations': FROM_CACHE,
        ':tracker:extractDeepLinksDebug': FROM_CACHE,
        ':tracker:extractDeepLinksForAarDebug': FROM_CACHE,
        ':tracker:generateDebugAssets': UP_TO_DATE,
        ':tracker:generateDebugRFile': FROM_CACHE,
        ':tracker:generateDebugResValues': FROM_CACHE,
        ':tracker:generateDebugResources': UP_TO_DATE,
        ':tracker:javaPreCompileDebug': FROM_CACHE,
        ':tracker:kaptDebugKotlin': SUCCESS,
        ':tracker:kaptGenerateStubsDebugKotlin': FROM_CACHE,
        ':tracker:mergeDebugConsumerProguardFiles': SUCCESS,
        ':tracker:mergeDebugGeneratedProguardFiles': SUCCESS,
        ':tracker:mergeDebugJavaResource': SUCCESS,
        ':tracker:mergeDebugJniLibFolders': FROM_CACHE,
        ':tracker:mergeDebugNativeLibs': NO_SOURCE,
        ':tracker:mergeDebugShaders': FROM_CACHE,
        ':tracker:packageDebugAssets': FROM_CACHE,
        ':tracker:packageDebugResources': FROM_CACHE,
        ':tracker:parseDebugLocalResources': FROM_CACHE,
        ':tracker:preBuild': UP_TO_DATE,
        ':tracker:preDebugBuild': UP_TO_DATE,
        ':tracker:prepareDebugArtProfile': SUCCESS,
        ':tracker:prepareLintJarForPublish': SUCCESS,
        ':tracker:processDebugJavaRes': NO_SOURCE,
        ':tracker:processDebugManifest': FROM_CACHE,
        ':tracker:stripDebugDebugSymbols': NO_SOURCE,
        ':tracker:syncDebugLibJars': FROM_CACHE,
        ':tracker:writeDebugAarMetadata': SUCCESS,
        ':wearable:assembleDebug': SUCCESS,
        ':wearable:checkDebugAarMetadata': SUCCESS,
        ':wearable:checkDebugDuplicateClasses': SUCCESS,
        ':wearable:checkKotlinGradlePluginConfigurationErrors': SUCCESS,
        ':wearable:compileDebugJavaWithJavac': FROM_CACHE,
        ':wearable:compileDebugKotlin': FROM_CACHE,
        ':wearable:compileDebugShaders': NO_SOURCE,
        ':wearable:compressDebugAssets': FROM_CACHE,
        ':wearable:createDebugApkListingFileRedirect': SUCCESS,
        ':wearable:createDebugCompatibleScreenManifests': SUCCESS,
        ':wearable:desugarDebugFileDependencies': FROM_CACHE,
        ':wearable:dexBuilderDebug': FROM_CACHE,
        ':wearable:extractDeepLinksDebug': FROM_CACHE,
        ':wearable:generateDebugAssets': UP_TO_DATE,
        ':wearable:generateDebugResValues': FROM_CACHE,
        ':wearable:generateDebugResources': UP_TO_DATE,
        ':wearable:javaPreCompileDebug': FROM_CACHE,
        ':wearable:kaptDebugKotlin': SKIPPED,
        ':wearable:kaptGenerateStubsDebugKotlin': SKIPPED,
        ':wearable:mapDebugSourceSetPaths': SUCCESS,
        ':wearable:mergeDebugAssets': FROM_CACHE,
        ':wearable:mergeDebugGlobalSynthetics': FROM_CACHE,
        ':wearable:mergeDebugJavaResource': SUCCESS,
        ':wearable:mergeDebugJniLibFolders': FROM_CACHE,
        ':wearable:mergeDebugNativeDebugMetadata': NO_SOURCE,
        ':wearable:mergeDebugNativeLibs': NO_SOURCE,
        ':wearable:mergeDebugResources': FROM_CACHE,
        ':wearable:mergeDebugShaders': FROM_CACHE,
        ':wearable:mergeExtDexDebug': FROM_CACHE,
        ':wearable:mergeLibDexDebug': FROM_CACHE,
        ':wearable:mergeProjectDexDebug': FROM_CACHE,
        ':wearable:packageDebug': SUCCESS,
        ':wearable:preBuild': UP_TO_DATE,
        ':wearable:preDebugBuild': UP_TO_DATE,
        ':wearable:processDebugJavaRes': NO_SOURCE,
        ':wearable:processDebugMainManifest': FROM_CACHE,
        ':wearable:processDebugManifest': FROM_CACHE,
        ':wearable:processDebugManifestForPackage': FROM_CACHE,
        ':wearable:processDebugResources': FROM_CACHE,
        ':wearable:stripDebugDebugSymbols': NO_SOURCE,
        ':wearable:validateSigningDebug': SUCCESS,
        ':wearable:writeDebugAppMetadata': SUCCESS,
        ':wearable:writeDebugSigningConfigVersions': SUCCESS,
    ]
}

class AndroidPluginExpectations7 {

    static final EXPECTED_RESULTS_7_4 = [
        ':cityquiz:assembleDebug': SUCCESS,
        ':cityquiz:checkDebugAarMetadata': SUCCESS,
        ':cityquiz:checkDebugDuplicateClasses': SUCCESS,
        ':cityquiz:checkKotlinGradlePluginConfigurationErrors': SUCCESS,
        ':cityquiz:compileDebugAidl': NO_SOURCE,
        ':cityquiz:compileDebugJavaWithJavac': FROM_CACHE,
        ':cityquiz:compileDebugKotlin': FROM_CACHE,
        ':cityquiz:compileDebugRenderscript': NO_SOURCE,
        ':cityquiz:compileDebugShaders': NO_SOURCE,
        ':cityquiz:compressDebugAssets': FROM_CACHE,
        ':cityquiz:createDebugApkListingFileRedirect': SUCCESS,
        ':cityquiz:createDebugCompatibleScreenManifests': SUCCESS,
        ':cityquiz:desugarDebugFileDependencies': FROM_CACHE,
        ':cityquiz:dexBuilderDebug': FROM_CACHE,
        ':cityquiz:extractDeepLinksDebug': FROM_CACHE,
        ':cityquiz:featureDebugWriter': SUCCESS,
        ':cityquiz:generateDebugAssets': UP_TO_DATE,
        ':cityquiz:generateDebugBuildConfig': FROM_CACHE,
        ':cityquiz:generateDebugFeatureTransitiveDeps': SUCCESS,
        ':cityquiz:generateDebugResValues': FROM_CACHE,
        ':cityquiz:generateDebugResources': UP_TO_DATE,
        ':cityquiz:javaPreCompileDebug': FROM_CACHE,
        ':cityquiz:mapDebugSourceSetPaths': SUCCESS,
        ':cityquiz:mergeDebugAssets': FROM_CACHE,
        ':cityquiz:mergeDebugJavaResource': SUCCESS,
        ':cityquiz:mergeDebugJniLibFolders': FROM_CACHE,
        ':cityquiz:mergeDebugNativeLibs': NO_SOURCE,
        ':cityquiz:mergeDebugResources': FROM_CACHE,
        ':cityquiz:mergeDebugShaders': FROM_CACHE,
        ':cityquiz:mergeExtDexDebug': FROM_CACHE,
        ':cityquiz:mergeLibDexDebug': FROM_CACHE,
        ':cityquiz:mergeProjectDexDebug': FROM_CACHE,
        ':cityquiz:packageDebug': SUCCESS,
        ':cityquiz:preBuild': UP_TO_DATE,
        ':cityquiz:preDebugBuild': UP_TO_DATE,
        ':cityquiz:processApplicationManifestDebugForBundle': SUCCESS,
        ':cityquiz:processDebugJavaRes': NO_SOURCE,
        ':cityquiz:processDebugMainManifest': FROM_CACHE,
        ':cityquiz:processDebugManifest': FROM_CACHE,
        ':cityquiz:processDebugManifestForPackage': FROM_CACHE,
        ':cityquiz:processDebugResources': FROM_CACHE,
        ':cityquiz:processManifestDebugForFeature': FROM_CACHE,
        ':cityquiz:stripDebugDebugSymbols': NO_SOURCE,
        ':common:assembleDebug': SUCCESS,
        ':common:bundleDebugAar': SUCCESS,
        ':common:bundleLibCompileToJarDebug': SUCCESS,
        ':common:bundleLibResDebug': SUCCESS,
        ':common:bundleLibRuntimeToJarDebug': SUCCESS,
        ':common:checkKotlinGradlePluginConfigurationErrors': SUCCESS,
        ':common:compileDebugAidl': NO_SOURCE,
        ':common:compileDebugJavaWithJavac': FROM_CACHE,
        ':common:compileDebugKotlin': FROM_CACHE,
        ':common:compileDebugLibraryResources': FROM_CACHE,
        ':common:compileDebugRenderscript': NO_SOURCE,
        ':common:compileDebugShaders': NO_SOURCE,
        ':common:copyDebugJniLibsProjectAndLocalJars': SUCCESS,
        ':common:copyDebugJniLibsProjectOnly': SUCCESS,
        ':common:createFullJarDebug': FROM_CACHE,
        ':common:dataBindingGenBaseClassesDebug': FROM_CACHE,
        ':common:dataBindingMergeDependencyArtifactsDebug': SUCCESS,
        ':common:extractDebugAnnotations': FROM_CACHE,
        ':common:extractDeepLinksDebug': FROM_CACHE,
        ':common:extractDeepLinksForAarDebug': FROM_CACHE,
        ':common:generateDebugAssets': UP_TO_DATE,
        ':common:generateDebugBuildConfig': FROM_CACHE,
        ':common:generateDebugRFile': FROM_CACHE,
        ':common:generateDebugResValues': FROM_CACHE,
        ':common:generateDebugResources': UP_TO_DATE,
        ':common:javaPreCompileDebug': FROM_CACHE,
        ':common:mergeDebugConsumerProguardFiles': SUCCESS,
        ':common:mergeDebugGeneratedProguardFiles': SUCCESS,
        ':common:mergeDebugJavaResource': SUCCESS,
        ':common:mergeDebugJniLibFolders': FROM_CACHE,
        ':common:mergeDebugNativeLibs': NO_SOURCE,
        ':common:mergeDebugShaders': FROM_CACHE,
        ':common:packageDebugAssets': FROM_CACHE,
        ':common:packageDebugRenderscript': NO_SOURCE,
        ':common:packageDebugResources': FROM_CACHE,
        ':common:parseDebugLocalResources': FROM_CACHE,
        ':common:preBuild': UP_TO_DATE,
        ':common:preDebugBuild': UP_TO_DATE,
        ':common:prepareDebugArtProfile': SUCCESS,
        ':common:prepareLintJarForPublish': SUCCESS,
        ':common:processDebugJavaRes': NO_SOURCE,
        ':common:processDebugManifest': FROM_CACHE,
        ':common:stripDebugDebugSymbols': NO_SOURCE,
        ':common:syncDebugLibJars': FROM_CACHE,
        ':common:writeDebugAarMetadata': SUCCESS,
        ':dasherdancer:assembleDebug': SUCCESS,
        ':dasherdancer:checkDebugAarMetadata': SUCCESS,
        ':dasherdancer:checkDebugDuplicateClasses': SUCCESS,
        ':dasherdancer:checkKotlinGradlePluginConfigurationErrors': SUCCESS,
        ':dasherdancer:compileDebugAidl': NO_SOURCE,
        ':dasherdancer:compileDebugJavaWithJavac': FROM_CACHE,
        ':dasherdancer:compileDebugKotlin': FROM_CACHE,
        ':dasherdancer:compileDebugRenderscript': NO_SOURCE,
        ':dasherdancer:compileDebugShaders': NO_SOURCE,
        ':dasherdancer:compressDebugAssets': FROM_CACHE,
        ':dasherdancer:createDebugApkListingFileRedirect': SUCCESS,
        ':dasherdancer:createDebugCompatibleScreenManifests': SUCCESS,
        ':dasherdancer:desugarDebugFileDependencies': FROM_CACHE,
        ':dasherdancer:dexBuilderDebug': FROM_CACHE,
        ':dasherdancer:extractDeepLinksDebug': FROM_CACHE,
        ':dasherdancer:featureDebugWriter': SUCCESS,
        ':dasherdancer:generateDebugAssets': UP_TO_DATE,
        ':dasherdancer:generateDebugBuildConfig': FROM_CACHE,
        ':dasherdancer:generateDebugFeatureTransitiveDeps': SUCCESS,
        ':dasherdancer:generateDebugResValues': FROM_CACHE,
        ':dasherdancer:generateDebugResources': UP_TO_DATE,
        ':dasherdancer:javaPreCompileDebug': FROM_CACHE,
        ':dasherdancer:mapDebugSourceSetPaths': SUCCESS,
        ':dasherdancer:mergeDebugAssets': FROM_CACHE,
        ':dasherdancer:mergeDebugJavaResource': SUCCESS,
        ':dasherdancer:mergeDebugJniLibFolders': FROM_CACHE,
        ':dasherdancer:mergeDebugNativeLibs': NO_SOURCE,
        ':dasherdancer:mergeDebugResources': FROM_CACHE,
        ':dasherdancer:mergeDebugShaders': FROM_CACHE,
        ':dasherdancer:mergeExtDexDebug': FROM_CACHE,
        ':dasherdancer:mergeLibDexDebug': FROM_CACHE,
        ':dasherdancer:mergeProjectDexDebug': FROM_CACHE,
        ':dasherdancer:packageDebug': SUCCESS,
        ':dasherdancer:preBuild': UP_TO_DATE,
        ':dasherdancer:preDebugBuild': UP_TO_DATE,
        ':dasherdancer:processApplicationManifestDebugForBundle': SUCCESS,
        ':dasherdancer:processDebugJavaRes': NO_SOURCE,
        ':dasherdancer:processDebugMainManifest': FROM_CACHE,
        ':dasherdancer:processDebugManifest': FROM_CACHE,
        ':dasherdancer:processDebugManifestForPackage': FROM_CACHE,
        ':dasherdancer:processDebugResources': FROM_CACHE,
        ':dasherdancer:processManifestDebugForFeature': FROM_CACHE,
        ':dasherdancer:stripDebugDebugSymbols': NO_SOURCE,
        ':doodles-lib:assembleDebug': SUCCESS,
        ':doodles-lib:bundleDebugAar': SUCCESS,
        ':doodles-lib:bundleLibCompileToJarDebug': SUCCESS,
        ':doodles-lib:bundleLibResDebug': NO_SOURCE,
        ':doodles-lib:bundleLibRuntimeToJarDebug': SUCCESS,
        ':doodles-lib:compileDebugAidl': NO_SOURCE,
        ':doodles-lib:compileDebugJavaWithJavac': FROM_CACHE,
        ':doodles-lib:compileDebugLibraryResources': FROM_CACHE,
        ':doodles-lib:compileDebugRenderscript': NO_SOURCE,
        ':doodles-lib:compileDebugShaders': NO_SOURCE,
        ':doodles-lib:copyDebugJniLibsProjectAndLocalJars': SUCCESS,
        ':doodles-lib:copyDebugJniLibsProjectOnly': SUCCESS,
        ':doodles-lib:createFullJarDebug': FROM_CACHE,
        ':doodles-lib:extractDebugAnnotations': FROM_CACHE,
        ':doodles-lib:extractDeepLinksDebug': FROM_CACHE,
        ':doodles-lib:extractDeepLinksForAarDebug': FROM_CACHE,
        ':doodles-lib:generateDebugAssets': UP_TO_DATE,
        ':doodles-lib:generateDebugBuildConfig': FROM_CACHE,
        ':doodles-lib:generateDebugRFile': FROM_CACHE,
        ':doodles-lib:generateDebugResValues': FROM_CACHE,
        ':doodles-lib:generateDebugResources': UP_TO_DATE,
        ':doodles-lib:javaPreCompileDebug': FROM_CACHE,
        ':doodles-lib:mergeDebugConsumerProguardFiles': SUCCESS,
        ':doodles-lib:mergeDebugGeneratedProguardFiles': SUCCESS,
        ':doodles-lib:mergeDebugJavaResource': SUCCESS,
        ':doodles-lib:mergeDebugJniLibFolders': FROM_CACHE,
        ':doodles-lib:mergeDebugNativeLibs': NO_SOURCE,
        ':doodles-lib:mergeDebugShaders': FROM_CACHE,
        ':doodles-lib:packageDebugAssets': FROM_CACHE,
        ':doodles-lib:packageDebugRenderscript': NO_SOURCE,
        ':doodles-lib:packageDebugResources': FROM_CACHE,
        ':doodles-lib:parseDebugLocalResources': FROM_CACHE,
        ':doodles-lib:preBuild': UP_TO_DATE,
        ':doodles-lib:preDebugBuild': UP_TO_DATE,
        ':doodles-lib:prepareDebugArtProfile': SUCCESS,
        ':doodles-lib:prepareLintJarForPublish': SUCCESS,
        ':doodles-lib:processDebugJavaRes': NO_SOURCE,
        ':doodles-lib:processDebugManifest': FROM_CACHE,
        ':doodles-lib:stripDebugDebugSymbols': NO_SOURCE,
        ':doodles-lib:syncDebugLibJars': FROM_CACHE,
        ':doodles-lib:writeDebugAarMetadata': SUCCESS,
        ':gumball:assembleDebug': SUCCESS,
        ':gumball:checkDebugAarMetadata': SUCCESS,
        ':gumball:checkDebugDuplicateClasses': SUCCESS,
        ':gumball:compileDebugAidl': NO_SOURCE,
        ':gumball:compileDebugJavaWithJavac': FROM_CACHE,
        ':gumball:compileDebugRenderscript': NO_SOURCE,
        ':gumball:compileDebugShaders': NO_SOURCE,
        ':gumball:compressDebugAssets': FROM_CACHE,
        ':gumball:createDebugApkListingFileRedirect': SUCCESS,
        ':gumball:createDebugCompatibleScreenManifests': SUCCESS,
        ':gumball:desugarDebugFileDependencies': FROM_CACHE,
        ':gumball:dexBuilderDebug': FROM_CACHE,
        ':gumball:extractDeepLinksDebug': FROM_CACHE,
        ':gumball:featureDebugWriter': SUCCESS,
        ':gumball:generateDebugAssets': UP_TO_DATE,
        ':gumball:generateDebugBuildConfig': FROM_CACHE,
        ':gumball:generateDebugFeatureTransitiveDeps': SUCCESS,
        ':gumball:generateDebugResValues': FROM_CACHE,
        ':gumball:generateDebugResources': UP_TO_DATE,
        ':gumball:javaPreCompileDebug': FROM_CACHE,
        ':gumball:mapDebugSourceSetPaths': SUCCESS,
        ':gumball:mergeDebugAssets': FROM_CACHE,
        ':gumball:mergeDebugJavaResource': SUCCESS,
        ':gumball:mergeDebugJniLibFolders': FROM_CACHE,
        ':gumball:mergeDebugNativeLibs': NO_SOURCE,
        ':gumball:mergeDebugResources': FROM_CACHE,
        ':gumball:mergeDebugShaders': FROM_CACHE,
        ':gumball:mergeExtDexDebug': FROM_CACHE,
        ':gumball:mergeLibDexDebug': FROM_CACHE,
        ':gumball:mergeProjectDexDebug': FROM_CACHE,
        ':gumball:packageDebug': SUCCESS,
        ':gumball:preBuild': UP_TO_DATE,
        ':gumball:preDebugBuild': UP_TO_DATE,
        ':gumball:processApplicationManifestDebugForBundle': SUCCESS,
        ':gumball:processDebugJavaRes': NO_SOURCE,
        ':gumball:processDebugMainManifest': FROM_CACHE,
        ':gumball:processDebugManifest': FROM_CACHE,
        ':gumball:processDebugManifestForPackage': FROM_CACHE,
        ':gumball:processDebugResources': FROM_CACHE,
        ':gumball:processManifestDebugForFeature': FROM_CACHE,
        ':gumball:stripDebugDebugSymbols': NO_SOURCE,
        ':jetpack:assembleDebug': SUCCESS,
        ':jetpack:checkDebugAarMetadata': SUCCESS,
        ':jetpack:checkDebugDuplicateClasses': SUCCESS,
        ':jetpack:checkKotlinGradlePluginConfigurationErrors': SUCCESS,
        ':jetpack:compileDebugAidl': NO_SOURCE,
        ':jetpack:compileDebugJavaWithJavac': FROM_CACHE,
        ':jetpack:compileDebugKotlin': FROM_CACHE,
        ':jetpack:compileDebugRenderscript': NO_SOURCE,
        ':jetpack:compileDebugShaders': NO_SOURCE,
        ':jetpack:compressDebugAssets': FROM_CACHE,
        ':jetpack:createDebugApkListingFileRedirect': SUCCESS,
        ':jetpack:createDebugCompatibleScreenManifests': SUCCESS,
        ':jetpack:desugarDebugFileDependencies': FROM_CACHE,
        ':jetpack:dexBuilderDebug': FROM_CACHE,
        ':jetpack:extractDeepLinksDebug': FROM_CACHE,
        ':jetpack:featureDebugWriter': SUCCESS,
        ':jetpack:generateDebugAssets': UP_TO_DATE,
        ':jetpack:generateDebugBuildConfig': FROM_CACHE,
        ':jetpack:generateDebugFeatureTransitiveDeps': SUCCESS,
        ':jetpack:generateDebugResValues': FROM_CACHE,
        ':jetpack:generateDebugResources': UP_TO_DATE,
        ':jetpack:javaPreCompileDebug': FROM_CACHE,
        ':jetpack:mapDebugSourceSetPaths': SUCCESS,
        ':jetpack:mergeDebugAssets': FROM_CACHE,
        ':jetpack:mergeDebugJavaResource': SUCCESS,
        ':jetpack:mergeDebugJniLibFolders': FROM_CACHE,
        ':jetpack:mergeDebugNativeLibs': NO_SOURCE,
        ':jetpack:mergeDebugResources': FROM_CACHE,
        ':jetpack:mergeDebugShaders': FROM_CACHE,
        ':jetpack:mergeExtDexDebug': FROM_CACHE,
        ':jetpack:mergeLibDexDebug': FROM_CACHE,
        ':jetpack:mergeProjectDexDebug': FROM_CACHE,
        ':jetpack:packageDebug': SUCCESS,
        ':jetpack:preBuild': UP_TO_DATE,
        ':jetpack:preDebugBuild': UP_TO_DATE,
        ':jetpack:processApplicationManifestDebugForBundle': SUCCESS,
        ':jetpack:processDebugJavaRes': NO_SOURCE,
        ':jetpack:processDebugMainManifest': FROM_CACHE,
        ':jetpack:processDebugManifest': FROM_CACHE,
        ':jetpack:processDebugManifestForPackage': FROM_CACHE,
        ':jetpack:processDebugResources': FROM_CACHE,
        ':jetpack:processManifestDebugForFeature': FROM_CACHE,
        ':jetpack:stripDebugDebugSymbols': NO_SOURCE,
        ':memory:assembleDebug': SUCCESS,
        ':memory:checkDebugAarMetadata': SUCCESS,
        ':memory:checkDebugDuplicateClasses': SUCCESS,
        ':memory:compileDebugAidl': NO_SOURCE,
        ':memory:compileDebugJavaWithJavac': FROM_CACHE,
        ':memory:compileDebugRenderscript': NO_SOURCE,
        ':memory:compileDebugShaders': NO_SOURCE,
        ':memory:compressDebugAssets': FROM_CACHE,
        ':memory:createDebugApkListingFileRedirect': SUCCESS,
        ':memory:createDebugCompatibleScreenManifests': SUCCESS,
        ':memory:desugarDebugFileDependencies': FROM_CACHE,
        ':memory:dexBuilderDebug': FROM_CACHE,
        ':memory:extractDeepLinksDebug': FROM_CACHE,
        ':memory:featureDebugWriter': SUCCESS,
        ':memory:generateDebugAssets': UP_TO_DATE,
        ':memory:generateDebugBuildConfig': FROM_CACHE,
        ':memory:generateDebugFeatureTransitiveDeps': SUCCESS,
        ':memory:generateDebugResValues': FROM_CACHE,
        ':memory:generateDebugResources': UP_TO_DATE,
        ':memory:javaPreCompileDebug': FROM_CACHE,
        ':memory:mapDebugSourceSetPaths': SUCCESS,
        ':memory:mergeDebugAssets': FROM_CACHE,
        ':memory:mergeDebugJavaResource': SUCCESS,
        ':memory:mergeDebugJniLibFolders': FROM_CACHE,
        ':memory:mergeDebugNativeLibs': NO_SOURCE,
        ':memory:mergeDebugResources': FROM_CACHE,
        ':memory:mergeDebugShaders': FROM_CACHE,
        ':memory:mergeExtDexDebug': FROM_CACHE,
        ':memory:mergeLibDexDebug': FROM_CACHE,
        ':memory:mergeProjectDexDebug': FROM_CACHE,
        ':memory:packageDebug': SUCCESS,
        ':memory:preBuild': UP_TO_DATE,
        ':memory:preDebugBuild': UP_TO_DATE,
        ':memory:processApplicationManifestDebugForBundle': SUCCESS,
        ':memory:processDebugJavaRes': NO_SOURCE,
        ':memory:processDebugMainManifest': FROM_CACHE,
        ':memory:processDebugManifest': FROM_CACHE,
        ':memory:processDebugManifestForPackage': FROM_CACHE,
        ':memory:processDebugResources': FROM_CACHE,
        ':memory:processManifestDebugForFeature': FROM_CACHE,
        ':memory:stripDebugDebugSymbols': NO_SOURCE,
        ':penguinswim:assembleDebug': SUCCESS,
        ':penguinswim:checkDebugAarMetadata': SUCCESS,
        ':penguinswim:checkDebugDuplicateClasses': SUCCESS,
        ':penguinswim:compileDebugAidl': NO_SOURCE,
        ':penguinswim:compileDebugJavaWithJavac': FROM_CACHE,
        ':penguinswim:compileDebugRenderscript': NO_SOURCE,
        ':penguinswim:compileDebugShaders': NO_SOURCE,
        ':penguinswim:compressDebugAssets': FROM_CACHE,
        ':penguinswim:createDebugApkListingFileRedirect': SUCCESS,
        ':penguinswim:createDebugCompatibleScreenManifests': SUCCESS,
        ':penguinswim:desugarDebugFileDependencies': FROM_CACHE,
        ':penguinswim:dexBuilderDebug': FROM_CACHE,
        ':penguinswim:extractDeepLinksDebug': FROM_CACHE,
        ':penguinswim:featureDebugWriter': SUCCESS,
        ':penguinswim:generateDebugAssets': UP_TO_DATE,
        ':penguinswim:generateDebugBuildConfig': FROM_CACHE,
        ':penguinswim:generateDebugFeatureTransitiveDeps': SUCCESS,
        ':penguinswim:generateDebugResValues': FROM_CACHE,
        ':penguinswim:generateDebugResources': UP_TO_DATE,
        ':penguinswim:javaPreCompileDebug': FROM_CACHE,
        ':penguinswim:mapDebugSourceSetPaths': SUCCESS,
        ':penguinswim:mergeDebugAssets': FROM_CACHE,
        ':penguinswim:mergeDebugJavaResource': SUCCESS,
        ':penguinswim:mergeDebugJniLibFolders': FROM_CACHE,
        ':penguinswim:mergeDebugNativeLibs': NO_SOURCE,
        ':penguinswim:mergeDebugResources': FROM_CACHE,
        ':penguinswim:mergeDebugShaders': FROM_CACHE,
        ':penguinswim:mergeExtDexDebug': FROM_CACHE,
        ':penguinswim:mergeLibDexDebug': FROM_CACHE,
        ':penguinswim:mergeProjectDexDebug': FROM_CACHE,
        ':penguinswim:packageDebug': SUCCESS,
        ':penguinswim:preBuild': UP_TO_DATE,
        ':penguinswim:preDebugBuild': UP_TO_DATE,
        ':penguinswim:processApplicationManifestDebugForBundle': SUCCESS,
        ':penguinswim:processDebugJavaRes': NO_SOURCE,
        ':penguinswim:processDebugMainManifest': FROM_CACHE,
        ':penguinswim:processDebugManifest': FROM_CACHE,
        ':penguinswim:processDebugManifestForPackage': FROM_CACHE,
        ':penguinswim:processDebugResources': FROM_CACHE,
        ':penguinswim:processManifestDebugForFeature': FROM_CACHE,
        ':penguinswim:stripDebugDebugSymbols': NO_SOURCE,
        ':playgames:assembleDebug': SUCCESS,
        ':playgames:bundleDebugAar': SUCCESS,
        ':playgames:bundleLibCompileToJarDebug': SUCCESS,
        ':playgames:bundleLibResDebug': NO_SOURCE,
        ':playgames:bundleLibRuntimeToJarDebug': SUCCESS,
        ':playgames:compileDebugAidl': NO_SOURCE,
        ':playgames:compileDebugJavaWithJavac': FROM_CACHE,
        ':playgames:compileDebugLibraryResources': FROM_CACHE,
        ':playgames:compileDebugRenderscript': NO_SOURCE,
        ':playgames:compileDebugShaders': NO_SOURCE,
        ':playgames:copyDebugJniLibsProjectAndLocalJars': SUCCESS,
        ':playgames:copyDebugJniLibsProjectOnly': SUCCESS,
        ':playgames:createFullJarDebug': FROM_CACHE,
        ':playgames:extractDebugAnnotations': FROM_CACHE,
        ':playgames:extractDeepLinksDebug': FROM_CACHE,
        ':playgames:extractDeepLinksForAarDebug': FROM_CACHE,
        ':playgames:generateDebugAssets': UP_TO_DATE,
        ':playgames:generateDebugBuildConfig': FROM_CACHE,
        ':playgames:generateDebugRFile': FROM_CACHE,
        ':playgames:generateDebugResValues': FROM_CACHE,
        ':playgames:generateDebugResources': UP_TO_DATE,
        ':playgames:javaPreCompileDebug': FROM_CACHE,
        ':playgames:mergeDebugConsumerProguardFiles': SUCCESS,
        ':playgames:mergeDebugGeneratedProguardFiles': SUCCESS,
        ':playgames:mergeDebugJavaResource': SUCCESS,
        ':playgames:mergeDebugJniLibFolders': FROM_CACHE,
        ':playgames:mergeDebugNativeLibs': NO_SOURCE,
        ':playgames:mergeDebugShaders': FROM_CACHE,
        ':playgames:packageDebugAssets': FROM_CACHE,
        ':playgames:packageDebugRenderscript': NO_SOURCE,
        ':playgames:packageDebugResources': FROM_CACHE,
        ':playgames:parseDebugLocalResources': FROM_CACHE,
        ':playgames:preBuild': UP_TO_DATE,
        ':playgames:preDebugBuild': UP_TO_DATE,
        ':playgames:prepareDebugArtProfile': SUCCESS,
        ':playgames:prepareLintJarForPublish': SUCCESS,
        ':playgames:processDebugJavaRes': NO_SOURCE,
        ':playgames:processDebugManifest': FROM_CACHE,
        ':playgames:stripDebugDebugSymbols': NO_SOURCE,
        ':playgames:syncDebugLibJars': FROM_CACHE,
        ':playgames:writeDebugAarMetadata': SUCCESS,
        ':presenttoss:assembleDebug': SUCCESS,
        ':presenttoss:checkDebugAarMetadata': SUCCESS,
        ':presenttoss:checkDebugDuplicateClasses': SUCCESS,
        ':presenttoss:compileDebugAidl': NO_SOURCE,
        ':presenttoss:compileDebugJavaWithJavac': FROM_CACHE,
        ':presenttoss:compileDebugRenderscript': NO_SOURCE,
        ':presenttoss:compileDebugShaders': NO_SOURCE,
        ':presenttoss:compressDebugAssets': FROM_CACHE,
        ':presenttoss:createDebugApkListingFileRedirect': SUCCESS,
        ':presenttoss:createDebugCompatibleScreenManifests': SUCCESS,
        ':presenttoss:desugarDebugFileDependencies': FROM_CACHE,
        ':presenttoss:dexBuilderDebug': FROM_CACHE,
        ':presenttoss:extractDeepLinksDebug': FROM_CACHE,
        ':presenttoss:featureDebugWriter': SUCCESS,
        ':presenttoss:generateDebugAssets': UP_TO_DATE,
        ':presenttoss:generateDebugBuildConfig': FROM_CACHE,
        ':presenttoss:generateDebugFeatureTransitiveDeps': SUCCESS,
        ':presenttoss:generateDebugResValues': FROM_CACHE,
        ':presenttoss:generateDebugResources': UP_TO_DATE,
        ':presenttoss:javaPreCompileDebug': FROM_CACHE,
        ':presenttoss:mapDebugSourceSetPaths': SUCCESS,
        ':presenttoss:mergeDebugAssets': FROM_CACHE,
        ':presenttoss:mergeDebugJavaResource': SUCCESS,
        ':presenttoss:mergeDebugJniLibFolders': FROM_CACHE,
        ':presenttoss:mergeDebugNativeLibs': NO_SOURCE,
        ':presenttoss:mergeDebugResources': FROM_CACHE,
        ':presenttoss:mergeDebugShaders': FROM_CACHE,
        ':presenttoss:mergeExtDexDebug': FROM_CACHE,
        ':presenttoss:mergeLibDexDebug': FROM_CACHE,
        ':presenttoss:mergeProjectDexDebug': FROM_CACHE,
        ':presenttoss:packageDebug': SUCCESS,
        ':presenttoss:preBuild': UP_TO_DATE,
        ':presenttoss:preDebugBuild': UP_TO_DATE,
        ':presenttoss:processApplicationManifestDebugForBundle': SUCCESS,
        ':presenttoss:processDebugJavaRes': NO_SOURCE,
        ':presenttoss:processDebugMainManifest': FROM_CACHE,
        ':presenttoss:processDebugManifest': FROM_CACHE,
        ':presenttoss:processDebugManifestForPackage': FROM_CACHE,
        ':presenttoss:processDebugResources': FROM_CACHE,
        ':presenttoss:processManifestDebugForFeature': FROM_CACHE,
        ':presenttoss:stripDebugDebugSymbols': NO_SOURCE,
        ':rocketsleigh:assembleDebug': SUCCESS,
        ':rocketsleigh:checkDebugAarMetadata': SUCCESS,
        ':rocketsleigh:checkDebugDuplicateClasses': SUCCESS,
        ':rocketsleigh:checkKotlinGradlePluginConfigurationErrors': SUCCESS,
        ':rocketsleigh:compileDebugAidl': NO_SOURCE,
        ':rocketsleigh:compileDebugJavaWithJavac': FROM_CACHE,
        ':rocketsleigh:compileDebugKotlin': FROM_CACHE,
        ':rocketsleigh:compileDebugRenderscript': NO_SOURCE,
        ':rocketsleigh:compileDebugShaders': NO_SOURCE,
        ':rocketsleigh:compressDebugAssets': FROM_CACHE,
        ':rocketsleigh:createDebugApkListingFileRedirect': SUCCESS,
        ':rocketsleigh:createDebugCompatibleScreenManifests': SUCCESS,
        ':rocketsleigh:desugarDebugFileDependencies': FROM_CACHE,
        ':rocketsleigh:dexBuilderDebug': FROM_CACHE,
        ':rocketsleigh:extractDeepLinksDebug': FROM_CACHE,
        ':rocketsleigh:featureDebugWriter': SUCCESS,
        ':rocketsleigh:generateDebugAssets': UP_TO_DATE,
        ':rocketsleigh:generateDebugBuildConfig': FROM_CACHE,
        ':rocketsleigh:generateDebugFeatureTransitiveDeps': SUCCESS,
        ':rocketsleigh:generateDebugResValues': FROM_CACHE,
        ':rocketsleigh:generateDebugResources': UP_TO_DATE,
        ':rocketsleigh:javaPreCompileDebug': FROM_CACHE,
        ':rocketsleigh:mapDebugSourceSetPaths': SUCCESS,
        ':rocketsleigh:mergeDebugAssets': FROM_CACHE,
        ':rocketsleigh:mergeDebugJavaResource': SUCCESS,
        ':rocketsleigh:mergeDebugJniLibFolders': FROM_CACHE,
        ':rocketsleigh:mergeDebugNativeLibs': NO_SOURCE,
        ':rocketsleigh:mergeDebugResources': FROM_CACHE,
        ':rocketsleigh:mergeDebugShaders': FROM_CACHE,
        ':rocketsleigh:mergeExtDexDebug': FROM_CACHE,
        ':rocketsleigh:mergeLibDexDebug': FROM_CACHE,
        ':rocketsleigh:mergeProjectDexDebug': FROM_CACHE,
        ':rocketsleigh:packageDebug': SUCCESS,
        ':rocketsleigh:preBuild': UP_TO_DATE,
        ':rocketsleigh:preDebugBuild': UP_TO_DATE,
        ':rocketsleigh:processApplicationManifestDebugForBundle': SUCCESS,
        ':rocketsleigh:processDebugJavaRes': NO_SOURCE,
        ':rocketsleigh:processDebugMainManifest': FROM_CACHE,
        ':rocketsleigh:processDebugManifest': FROM_CACHE,
        ':rocketsleigh:processDebugManifestForPackage': FROM_CACHE,
        ':rocketsleigh:processDebugResources': FROM_CACHE,
        ':rocketsleigh:processManifestDebugForFeature': FROM_CACHE,
        ':rocketsleigh:stripDebugDebugSymbols': NO_SOURCE,
        ':santa-tracker:assembleDebug': SUCCESS,
        ':santa-tracker:bundleDebugClassesToCompileJar': SUCCESS,
        ':santa-tracker:bundleDebugClassesToRuntimeJar': SUCCESS,
        ':santa-tracker:checkDebugAarMetadata': SUCCESS,
        ':santa-tracker:checkDebugDuplicateClasses': SUCCESS,
        ':santa-tracker:checkDebugLibraries': SUCCESS,
        ':santa-tracker:checkKotlinGradlePluginConfigurationErrors': SUCCESS,
        ':santa-tracker:compileDebugAidl': NO_SOURCE,
        ':santa-tracker:compileDebugJavaWithJavac': FROM_CACHE,
        ':santa-tracker:compileDebugKotlin': FROM_CACHE,
        ':santa-tracker:compileDebugRenderscript': NO_SOURCE,
        ':santa-tracker:compileDebugShaders': NO_SOURCE,
        ':santa-tracker:compressDebugAssets': FROM_CACHE,
        ':santa-tracker:createDebugApkListingFileRedirect': SUCCESS,
        ':santa-tracker:createDebugCompatibleScreenManifests': SUCCESS,
        ':santa-tracker:dataBindingGenBaseClassesDebug': FROM_CACHE,
        ':santa-tracker:dataBindingMergeDependencyArtifactsDebug': SUCCESS,
        ':santa-tracker:desugarDebugFileDependencies': FROM_CACHE,
        ':santa-tracker:dexBuilderDebug': FROM_CACHE,
        ':santa-tracker:extractDeepLinksDebug': FROM_CACHE,
        ':santa-tracker:generateDebugAssets': UP_TO_DATE,
        ':santa-tracker:generateDebugBuildConfig': FROM_CACHE,
        ':santa-tracker:generateDebugFeatureMetadata': FROM_CACHE,
        ':santa-tracker:generateDebugFeatureTransitiveDeps': SUCCESS,
        ':santa-tracker:generateDebugResValues': FROM_CACHE,
        ':santa-tracker:generateDebugResources': UP_TO_DATE,
        ':santa-tracker:handleDebugMicroApk': FROM_CACHE,
        ':santa-tracker:javaPreCompileDebug': FROM_CACHE,
        ':santa-tracker:kaptDebugKotlin': FROM_CACHE,
        ':santa-tracker:kaptGenerateStubsDebugKotlin': FROM_CACHE,
        ':santa-tracker:mapDebugSourceSetPaths': SUCCESS,
        ':santa-tracker:mergeDebugAssets': FROM_CACHE,
        ':santa-tracker:mergeDebugJavaResource': SUCCESS,
        ':santa-tracker:mergeDebugJniLibFolders': FROM_CACHE,
        ':santa-tracker:mergeDebugNativeDebugMetadata': NO_SOURCE,
        ':santa-tracker:mergeDebugNativeLibs': NO_SOURCE,
        ':santa-tracker:mergeDebugResources': FROM_CACHE,
        ':santa-tracker:mergeDebugShaders': FROM_CACHE,
        ':santa-tracker:mergeExtDexDebug': FROM_CACHE,
        ':santa-tracker:mergeLibDexDebug': FROM_CACHE,
        ':santa-tracker:mergeProjectDexDebug': FROM_CACHE,
        ':santa-tracker:packageDebug': SUCCESS,
        ':santa-tracker:preBuild': UP_TO_DATE,
        ':santa-tracker:preDebugBuild': SUCCESS,
        ':santa-tracker:processDebugJavaRes': NO_SOURCE,
        ':santa-tracker:processDebugMainManifest': FROM_CACHE,
        ':santa-tracker:processDebugManifest': FROM_CACHE,
        ':santa-tracker:processDebugManifestForPackage': FROM_CACHE,
        ':santa-tracker:processDebugResources': FROM_CACHE,
        ':santa-tracker:signingConfigWriterDebug': SUCCESS,
        ':santa-tracker:stripDebugDebugSymbols': NO_SOURCE,
        ':santa-tracker:validateSigningDebug': SUCCESS,
        ':santa-tracker:writeDebugAppMetadata': SUCCESS,
        ':santa-tracker:writeDebugModuleMetadata': SUCCESS,
        ':santa-tracker:writeDebugSigningConfigVersions': SUCCESS,
        ':snowballrun:assembleDebug': SUCCESS,
        ':snowballrun:checkDebugAarMetadata': SUCCESS,
        ':snowballrun:checkDebugDuplicateClasses': SUCCESS,
        ':snowballrun:compileDebugAidl': NO_SOURCE,
        ':snowballrun:compileDebugJavaWithJavac': FROM_CACHE,
        ':snowballrun:compileDebugRenderscript': NO_SOURCE,
        ':snowballrun:compileDebugShaders': NO_SOURCE,
        ':snowballrun:compressDebugAssets': FROM_CACHE,
        ':snowballrun:createDebugApkListingFileRedirect': SUCCESS,
        ':snowballrun:createDebugCompatibleScreenManifests': SUCCESS,
        ':snowballrun:desugarDebugFileDependencies': FROM_CACHE,
        ':snowballrun:dexBuilderDebug': FROM_CACHE,
        ':snowballrun:extractDeepLinksDebug': FROM_CACHE,
        ':snowballrun:featureDebugWriter': SUCCESS,
        ':snowballrun:generateDebugAssets': UP_TO_DATE,
        ':snowballrun:generateDebugBuildConfig': FROM_CACHE,
        ':snowballrun:generateDebugFeatureTransitiveDeps': SUCCESS,
        ':snowballrun:generateDebugResValues': FROM_CACHE,
        ':snowballrun:generateDebugResources': UP_TO_DATE,
        ':snowballrun:javaPreCompileDebug': FROM_CACHE,
        ':snowballrun:mapDebugSourceSetPaths': SUCCESS,
        ':snowballrun:mergeDebugAssets': FROM_CACHE,
        ':snowballrun:mergeDebugJavaResource': SUCCESS,
        ':snowballrun:mergeDebugJniLibFolders': FROM_CACHE,
        ':snowballrun:mergeDebugNativeLibs': NO_SOURCE,
        ':snowballrun:mergeDebugResources': FROM_CACHE,
        ':snowballrun:mergeDebugShaders': FROM_CACHE,
        ':snowballrun:mergeExtDexDebug': FROM_CACHE,
        ':snowballrun:mergeLibDexDebug': FROM_CACHE,
        ':snowballrun:mergeProjectDexDebug': FROM_CACHE,
        ':snowballrun:packageDebug': SUCCESS,
        ':snowballrun:preBuild': UP_TO_DATE,
        ':snowballrun:preDebugBuild': UP_TO_DATE,
        ':snowballrun:processApplicationManifestDebugForBundle': SUCCESS,
        ':snowballrun:processDebugJavaRes': NO_SOURCE,
        ':snowballrun:processDebugMainManifest': FROM_CACHE,
        ':snowballrun:processDebugManifest': FROM_CACHE,
        ':snowballrun:processDebugManifestForPackage': FROM_CACHE,
        ':snowballrun:processDebugResources': FROM_CACHE,
        ':snowballrun:processManifestDebugForFeature': FROM_CACHE,
        ':snowballrun:stripDebugDebugSymbols': NO_SOURCE,
        ':tracker:assembleDebug': SUCCESS,
        ':tracker:bundleDebugAar': SUCCESS,
        ':tracker:bundleLibCompileToJarDebug': SUCCESS,
        ':tracker:bundleLibResDebug': SUCCESS,
        ':tracker:bundleLibRuntimeToJarDebug': SUCCESS,
        ':tracker:checkKotlinGradlePluginConfigurationErrors': SUCCESS,
        ':tracker:compileDebugAidl': NO_SOURCE,
        ':tracker:compileDebugJavaWithJavac': SUCCESS,
        ':tracker:compileDebugKotlin': FROM_CACHE,
        ':tracker:compileDebugLibraryResources': FROM_CACHE,
        ':tracker:compileDebugRenderscript': NO_SOURCE,
        ':tracker:compileDebugShaders': NO_SOURCE,
        ':tracker:copyDebugJniLibsProjectAndLocalJars': SUCCESS,
        ':tracker:copyDebugJniLibsProjectOnly': SUCCESS,
        ':tracker:createFullJarDebug': FROM_CACHE,
        ':tracker:extractDebugAnnotations': FROM_CACHE,
        ':tracker:extractDeepLinksDebug': FROM_CACHE,
        ':tracker:extractDeepLinksForAarDebug': FROM_CACHE,
        ':tracker:generateDebugAssets': UP_TO_DATE,
        ':tracker:generateDebugBuildConfig': FROM_CACHE,
        ':tracker:generateDebugRFile': FROM_CACHE,
        ':tracker:generateDebugResValues': FROM_CACHE,
        ':tracker:generateDebugResources': UP_TO_DATE,
        ':tracker:javaPreCompileDebug': FROM_CACHE,
        ':tracker:kaptDebugKotlin': SUCCESS,
        ':tracker:kaptGenerateStubsDebugKotlin': FROM_CACHE,
        ':tracker:mergeDebugConsumerProguardFiles': SUCCESS,
        ':tracker:mergeDebugGeneratedProguardFiles': SUCCESS,
        ':tracker:mergeDebugJavaResource': SUCCESS,
        ':tracker:mergeDebugJniLibFolders': FROM_CACHE,
        ':tracker:mergeDebugNativeLibs': NO_SOURCE,
        ':tracker:mergeDebugShaders': FROM_CACHE,
        ':tracker:packageDebugAssets': FROM_CACHE,
        ':tracker:packageDebugRenderscript': NO_SOURCE,
        ':tracker:packageDebugResources': FROM_CACHE,
        ':tracker:parseDebugLocalResources': FROM_CACHE,
        ':tracker:preBuild': UP_TO_DATE,
        ':tracker:preDebugBuild': UP_TO_DATE,
        ':tracker:prepareDebugArtProfile': SUCCESS,
        ':tracker:prepareLintJarForPublish': SUCCESS,
        ':tracker:processDebugJavaRes': NO_SOURCE,
        ':tracker:processDebugManifest': FROM_CACHE,
        ':tracker:stripDebugDebugSymbols': NO_SOURCE,
        ':tracker:syncDebugLibJars': FROM_CACHE,
        ':tracker:writeDebugAarMetadata': SUCCESS,
        ':wearable:assembleDebug': SUCCESS,
        ':wearable:checkDebugAarMetadata': SUCCESS,
        ':wearable:checkDebugDuplicateClasses': SUCCESS,
        ':wearable:checkKotlinGradlePluginConfigurationErrors': SUCCESS,
        ':wearable:compileDebugAidl': NO_SOURCE,
        ':wearable:compileDebugJavaWithJavac': FROM_CACHE,
        ':wearable:compileDebugKotlin': FROM_CACHE,
        ':wearable:compileDebugRenderscript': NO_SOURCE,
        ':wearable:compileDebugShaders': NO_SOURCE,
        ':wearable:compressDebugAssets': FROM_CACHE,
        ':wearable:createDebugApkListingFileRedirect': SUCCESS,
        ':wearable:createDebugCompatibleScreenManifests': SUCCESS,
        ':wearable:desugarDebugFileDependencies': FROM_CACHE,
        ':wearable:dexBuilderDebug': FROM_CACHE,
        ':wearable:extractDeepLinksDebug': FROM_CACHE,
        ':wearable:generateDebugAssets': UP_TO_DATE,
        ':wearable:generateDebugBuildConfig': FROM_CACHE,
        ':wearable:generateDebugResValues': FROM_CACHE,
        ':wearable:generateDebugResources': UP_TO_DATE,
        ':wearable:javaPreCompileDebug': FROM_CACHE,
        ':wearable:kaptDebugKotlin': SKIPPED,
        ':wearable:kaptGenerateStubsDebugKotlin': SKIPPED,
        ':wearable:mapDebugSourceSetPaths': SUCCESS,
        ':wearable:mergeDebugAssets': FROM_CACHE,
        ':wearable:mergeDebugJavaResource': SUCCESS,
        ':wearable:mergeDebugJniLibFolders': FROM_CACHE,
        ':wearable:mergeDebugNativeDebugMetadata': NO_SOURCE,
        ':wearable:mergeDebugNativeLibs': NO_SOURCE,
        ':wearable:mergeDebugResources': FROM_CACHE,
        ':wearable:mergeDebugShaders': FROM_CACHE,
        ':wearable:mergeExtDexDebug': FROM_CACHE,
        ':wearable:mergeLibDexDebug': FROM_CACHE,
        ':wearable:mergeProjectDexDebug': FROM_CACHE,
        ':wearable:packageDebug': SUCCESS,
        ':wearable:preBuild': UP_TO_DATE,
        ':wearable:preDebugBuild': UP_TO_DATE,
        ':wearable:processDebugJavaRes': NO_SOURCE,
        ':wearable:processDebugMainManifest': FROM_CACHE,
        ':wearable:processDebugManifest': FROM_CACHE,
        ':wearable:processDebugManifestForPackage': FROM_CACHE,
        ':wearable:processDebugResources': FROM_CACHE,
        ':wearable:stripDebugDebugSymbols': NO_SOURCE,
        ':wearable:validateSigningDebug': SUCCESS,
        ':wearable:writeDebugAppMetadata': SUCCESS,
        ':wearable:writeDebugSigningConfigVersions': SUCCESS,
    ]

    static final EXPECTED_RESULTS_7_3 = [
        ':cityquiz:assembleDebug': SUCCESS,
        ':cityquiz:checkDebugAarMetadata': SUCCESS,
        ':cityquiz:checkDebugDuplicateClasses': SUCCESS,
        ':cityquiz:checkKotlinGradlePluginConfigurationErrors': SUCCESS,
        ':cityquiz:compileDebugAidl': NO_SOURCE,
        ':cityquiz:compileDebugJavaWithJavac': FROM_CACHE,
        ':cityquiz:compileDebugKotlin': FROM_CACHE,
        ':cityquiz:compileDebugRenderscript': NO_SOURCE,
        ':cityquiz:compileDebugShaders': NO_SOURCE,
        ':cityquiz:compressDebugAssets': FROM_CACHE,
        ':cityquiz:createDebugApkListingFileRedirect': SUCCESS,
        ':cityquiz:createDebugCompatibleScreenManifests': SUCCESS,
        ':cityquiz:desugarDebugFileDependencies': FROM_CACHE,
        ':cityquiz:dexBuilderDebug': FROM_CACHE,
        ':cityquiz:extractDeepLinksDebug': FROM_CACHE,
        ':cityquiz:featureDebugWriter': SUCCESS,
        ':cityquiz:generateDebugAssets': UP_TO_DATE,
        ':cityquiz:generateDebugBuildConfig': FROM_CACHE,
        ':cityquiz:generateDebugFeatureTransitiveDeps': SUCCESS,
        ':cityquiz:generateDebugResValues': FROM_CACHE,
        ':cityquiz:generateDebugResources': UP_TO_DATE,
        ':cityquiz:javaPreCompileDebug': FROM_CACHE,
        ':cityquiz:mapDebugSourceSetPaths': SUCCESS,
        ':cityquiz:mergeDebugAssets': FROM_CACHE,
        ':cityquiz:mergeDebugJavaResource': SUCCESS,
        ':cityquiz:mergeDebugJniLibFolders': FROM_CACHE,
        ':cityquiz:mergeDebugNativeLibs': NO_SOURCE,
        ':cityquiz:mergeDebugResources': FROM_CACHE,
        ':cityquiz:mergeDebugShaders': FROM_CACHE,
        ':cityquiz:mergeExtDexDebug': FROM_CACHE,
        ':cityquiz:mergeLibDexDebug': FROM_CACHE,
        ':cityquiz:mergeProjectDexDebug': FROM_CACHE,
        ':cityquiz:packageDebug': SUCCESS,
        ':cityquiz:preBuild': UP_TO_DATE,
        ':cityquiz:preDebugBuild': UP_TO_DATE,
        ':cityquiz:processApplicationManifestDebugForBundle': SUCCESS,
        ':cityquiz:processDebugJavaRes': NO_SOURCE,
        ':cityquiz:processDebugMainManifest': FROM_CACHE,
        ':cityquiz:processDebugManifest': FROM_CACHE,
        ':cityquiz:processDebugManifestForPackage': FROM_CACHE,
        ':cityquiz:processDebugResources': FROM_CACHE,
        ':cityquiz:processManifestDebugForFeature': FROM_CACHE,
        ':cityquiz:stripDebugDebugSymbols': NO_SOURCE,
        ':common:assembleDebug': SUCCESS,
        ':common:bundleDebugAar': SUCCESS,
        ':common:bundleLibCompileToJarDebug': SUCCESS,
        ':common:bundleLibResDebug': SUCCESS,
        ':common:bundleLibRuntimeToJarDebug': SUCCESS,
        ':common:checkKotlinGradlePluginConfigurationErrors': SUCCESS,
        ':common:compileDebugAidl': NO_SOURCE,
        ':common:compileDebugJavaWithJavac': FROM_CACHE,
        ':common:compileDebugKotlin': FROM_CACHE,
        ':common:compileDebugLibraryResources': FROM_CACHE,
        ':common:compileDebugRenderscript': NO_SOURCE,
        ':common:compileDebugShaders': NO_SOURCE,
        ':common:copyDebugJniLibsProjectAndLocalJars': SUCCESS,
        ':common:copyDebugJniLibsProjectOnly': SUCCESS,
        ':common:createFullJarDebug': FROM_CACHE,
        ':common:dataBindingGenBaseClassesDebug': FROM_CACHE,
        ':common:dataBindingMergeDependencyArtifactsDebug': SUCCESS,
        ':common:extractDebugAnnotations': FROM_CACHE,
        ':common:extractDeepLinksDebug': FROM_CACHE,
        ':common:extractDeepLinksForAarDebug': FROM_CACHE,
        ':common:generateDebugAssets': UP_TO_DATE,
        ':common:generateDebugBuildConfig': FROM_CACHE,
        ':common:generateDebugRFile': FROM_CACHE,
        ':common:generateDebugResValues': FROM_CACHE,
        ':common:generateDebugResources': UP_TO_DATE,
        ':common:javaPreCompileDebug': FROM_CACHE,
        ':common:mergeDebugConsumerProguardFiles': SUCCESS,
        ':common:mergeDebugGeneratedProguardFiles': SUCCESS,
        ':common:mergeDebugJavaResource': SUCCESS,
        ':common:mergeDebugJniLibFolders': FROM_CACHE,
        ':common:mergeDebugNativeLibs': NO_SOURCE,
        ':common:mergeDebugShaders': FROM_CACHE,
        ':common:packageDebugAssets': FROM_CACHE,
        ':common:packageDebugRenderscript': NO_SOURCE,
        ':common:packageDebugResources': FROM_CACHE,
        ':common:parseDebugLocalResources': FROM_CACHE,
        ':common:preBuild': UP_TO_DATE,
        ':common:preDebugBuild': UP_TO_DATE,
        ':common:prepareDebugArtProfile': SUCCESS,
        ':common:prepareLintJarForPublish': SUCCESS,
        ':common:processDebugJavaRes': NO_SOURCE,
        ':common:processDebugManifest': FROM_CACHE,
        ':common:stripDebugDebugSymbols': NO_SOURCE,
        ':common:syncDebugLibJars': FROM_CACHE,
        ':common:writeDebugAarMetadata': SUCCESS,
        ':dasherdancer:assembleDebug': SUCCESS,
        ':dasherdancer:checkDebugAarMetadata': SUCCESS,
        ':dasherdancer:checkDebugDuplicateClasses': SUCCESS,
        ':dasherdancer:checkKotlinGradlePluginConfigurationErrors': SUCCESS,
        ':dasherdancer:compileDebugAidl': NO_SOURCE,
        ':dasherdancer:compileDebugJavaWithJavac': FROM_CACHE,
        ':dasherdancer:compileDebugKotlin': FROM_CACHE,
        ':dasherdancer:compileDebugRenderscript': NO_SOURCE,
        ':dasherdancer:compileDebugShaders': NO_SOURCE,
        ':dasherdancer:compressDebugAssets': FROM_CACHE,
        ':dasherdancer:createDebugApkListingFileRedirect': SUCCESS,
        ':dasherdancer:createDebugCompatibleScreenManifests': SUCCESS,
        ':dasherdancer:desugarDebugFileDependencies': FROM_CACHE,
        ':dasherdancer:dexBuilderDebug': FROM_CACHE,
        ':dasherdancer:extractDeepLinksDebug': FROM_CACHE,
        ':dasherdancer:featureDebugWriter': SUCCESS,
        ':dasherdancer:generateDebugAssets': UP_TO_DATE,
        ':dasherdancer:generateDebugBuildConfig': FROM_CACHE,
        ':dasherdancer:generateDebugFeatureTransitiveDeps': SUCCESS,
        ':dasherdancer:generateDebugResValues': FROM_CACHE,
        ':dasherdancer:generateDebugResources': UP_TO_DATE,
        ':dasherdancer:javaPreCompileDebug': FROM_CACHE,
        ':dasherdancer:mapDebugSourceSetPaths': SUCCESS,
        ':dasherdancer:mergeDebugAssets': FROM_CACHE,
        ':dasherdancer:mergeDebugJavaResource': SUCCESS,
        ':dasherdancer:mergeDebugJniLibFolders': FROM_CACHE,
        ':dasherdancer:mergeDebugNativeLibs': NO_SOURCE,
        ':dasherdancer:mergeDebugResources': FROM_CACHE,
        ':dasherdancer:mergeDebugShaders': FROM_CACHE,
        ':dasherdancer:mergeExtDexDebug': FROM_CACHE,
        ':dasherdancer:mergeLibDexDebug': FROM_CACHE,
        ':dasherdancer:mergeProjectDexDebug': FROM_CACHE,
        ':dasherdancer:packageDebug': SUCCESS,
        ':dasherdancer:preBuild': UP_TO_DATE,
        ':dasherdancer:preDebugBuild': UP_TO_DATE,
        ':dasherdancer:processApplicationManifestDebugForBundle': SUCCESS,
        ':dasherdancer:processDebugJavaRes': NO_SOURCE,
        ':dasherdancer:processDebugMainManifest': FROM_CACHE,
        ':dasherdancer:processDebugManifest': FROM_CACHE,
        ':dasherdancer:processDebugManifestForPackage': FROM_CACHE,
        ':dasherdancer:processDebugResources': FROM_CACHE,
        ':dasherdancer:processManifestDebugForFeature': FROM_CACHE,
        ':dasherdancer:stripDebugDebugSymbols': NO_SOURCE,
        ':doodles-lib:assembleDebug': SUCCESS,
        ':doodles-lib:bundleDebugAar': SUCCESS,
        ':doodles-lib:bundleLibCompileToJarDebug': SUCCESS,
        ':doodles-lib:bundleLibResDebug': NO_SOURCE,
        ':doodles-lib:bundleLibRuntimeToJarDebug': SUCCESS,
        ':doodles-lib:compileDebugAidl': NO_SOURCE,
        ':doodles-lib:compileDebugJavaWithJavac': FROM_CACHE,
        ':doodles-lib:compileDebugLibraryResources': FROM_CACHE,
        ':doodles-lib:compileDebugRenderscript': NO_SOURCE,
        ':doodles-lib:compileDebugShaders': NO_SOURCE,
        ':doodles-lib:copyDebugJniLibsProjectAndLocalJars': SUCCESS,
        ':doodles-lib:copyDebugJniLibsProjectOnly': SUCCESS,
        ':doodles-lib:createFullJarDebug': FROM_CACHE,
        ':doodles-lib:extractDebugAnnotations': FROM_CACHE,
        ':doodles-lib:extractDeepLinksDebug': FROM_CACHE,
        ':doodles-lib:extractDeepLinksForAarDebug': FROM_CACHE,
        ':doodles-lib:generateDebugAssets': UP_TO_DATE,
        ':doodles-lib:generateDebugBuildConfig': FROM_CACHE,
        ':doodles-lib:generateDebugRFile': FROM_CACHE,
        ':doodles-lib:generateDebugResValues': FROM_CACHE,
        ':doodles-lib:generateDebugResources': UP_TO_DATE,
        ':doodles-lib:javaPreCompileDebug': FROM_CACHE,
        ':doodles-lib:mergeDebugConsumerProguardFiles': SUCCESS,
        ':doodles-lib:mergeDebugGeneratedProguardFiles': SUCCESS,
        ':doodles-lib:mergeDebugJavaResource': SUCCESS,
        ':doodles-lib:mergeDebugJniLibFolders': FROM_CACHE,
        ':doodles-lib:mergeDebugNativeLibs': NO_SOURCE,
        ':doodles-lib:mergeDebugShaders': FROM_CACHE,
        ':doodles-lib:packageDebugAssets': FROM_CACHE,
        ':doodles-lib:packageDebugRenderscript': NO_SOURCE,
        ':doodles-lib:packageDebugResources': FROM_CACHE,
        ':doodles-lib:parseDebugLocalResources': FROM_CACHE,
        ':doodles-lib:preBuild': UP_TO_DATE,
        ':doodles-lib:preDebugBuild': UP_TO_DATE,
        ':doodles-lib:prepareDebugArtProfile': SUCCESS,
        ':doodles-lib:prepareLintJarForPublish': SUCCESS,
        ':doodles-lib:processDebugJavaRes': NO_SOURCE,
        ':doodles-lib:processDebugManifest': FROM_CACHE,
        ':doodles-lib:stripDebugDebugSymbols': NO_SOURCE,
        ':doodles-lib:syncDebugLibJars': FROM_CACHE,
        ':doodles-lib:writeDebugAarMetadata': SUCCESS,
        ':gumball:assembleDebug': SUCCESS,
        ':gumball:checkDebugAarMetadata': SUCCESS,
        ':gumball:checkDebugDuplicateClasses': SUCCESS,
        ':gumball:compileDebugAidl': NO_SOURCE,
        ':gumball:compileDebugJavaWithJavac': FROM_CACHE,
        ':gumball:compileDebugRenderscript': NO_SOURCE,
        ':gumball:compileDebugShaders': NO_SOURCE,
        ':gumball:compressDebugAssets': FROM_CACHE,
        ':gumball:createDebugApkListingFileRedirect': SUCCESS,
        ':gumball:createDebugCompatibleScreenManifests': SUCCESS,
        ':gumball:desugarDebugFileDependencies': FROM_CACHE,
        ':gumball:dexBuilderDebug': FROM_CACHE,
        ':gumball:extractDeepLinksDebug': FROM_CACHE,
        ':gumball:featureDebugWriter': SUCCESS,
        ':gumball:generateDebugAssets': UP_TO_DATE,
        ':gumball:generateDebugBuildConfig': FROM_CACHE,
        ':gumball:generateDebugFeatureTransitiveDeps': SUCCESS,
        ':gumball:generateDebugResValues': FROM_CACHE,
        ':gumball:generateDebugResources': UP_TO_DATE,
        ':gumball:javaPreCompileDebug': FROM_CACHE,
        ':gumball:mapDebugSourceSetPaths': SUCCESS,
        ':gumball:mergeDebugAssets': FROM_CACHE,
        ':gumball:mergeDebugJavaResource': SUCCESS,
        ':gumball:mergeDebugJniLibFolders': FROM_CACHE,
        ':gumball:mergeDebugNativeLibs': NO_SOURCE,
        ':gumball:mergeDebugResources': FROM_CACHE,
        ':gumball:mergeDebugShaders': FROM_CACHE,
        ':gumball:mergeExtDexDebug': FROM_CACHE,
        ':gumball:mergeLibDexDebug': FROM_CACHE,
        ':gumball:mergeProjectDexDebug': FROM_CACHE,
        ':gumball:packageDebug': SUCCESS,
        ':gumball:preBuild': UP_TO_DATE,
        ':gumball:preDebugBuild': UP_TO_DATE,
        ':gumball:processApplicationManifestDebugForBundle': SUCCESS,
        ':gumball:processDebugJavaRes': NO_SOURCE,
        ':gumball:processDebugMainManifest': FROM_CACHE,
        ':gumball:processDebugManifest': FROM_CACHE,
        ':gumball:processDebugManifestForPackage': FROM_CACHE,
        ':gumball:processDebugResources': FROM_CACHE,
        ':gumball:processManifestDebugForFeature': FROM_CACHE,
        ':gumball:stripDebugDebugSymbols': NO_SOURCE,
        ':jetpack:assembleDebug': SUCCESS,
        ':jetpack:checkDebugAarMetadata': SUCCESS,
        ':jetpack:checkDebugDuplicateClasses': SUCCESS,
        ':jetpack:checkKotlinGradlePluginConfigurationErrors': SUCCESS,
        ':jetpack:compileDebugAidl': NO_SOURCE,
        ':jetpack:compileDebugJavaWithJavac': FROM_CACHE,
        ':jetpack:compileDebugKotlin': FROM_CACHE,
        ':jetpack:compileDebugRenderscript': NO_SOURCE,
        ':jetpack:compileDebugShaders': NO_SOURCE,
        ':jetpack:compressDebugAssets': FROM_CACHE,
        ':jetpack:createDebugApkListingFileRedirect': SUCCESS,
        ':jetpack:createDebugCompatibleScreenManifests': SUCCESS,
        ':jetpack:desugarDebugFileDependencies': FROM_CACHE,
        ':jetpack:dexBuilderDebug': FROM_CACHE,
        ':jetpack:extractDeepLinksDebug': FROM_CACHE,
        ':jetpack:featureDebugWriter': SUCCESS,
        ':jetpack:generateDebugAssets': UP_TO_DATE,
        ':jetpack:generateDebugBuildConfig': FROM_CACHE,
        ':jetpack:generateDebugFeatureTransitiveDeps': SUCCESS,
        ':jetpack:generateDebugResValues': FROM_CACHE,
        ':jetpack:generateDebugResources': UP_TO_DATE,
        ':jetpack:javaPreCompileDebug': FROM_CACHE,
        ':jetpack:mapDebugSourceSetPaths': SUCCESS,
        ':jetpack:mergeDebugAssets': FROM_CACHE,
        ':jetpack:mergeDebugJavaResource': SUCCESS,
        ':jetpack:mergeDebugJniLibFolders': FROM_CACHE,
        ':jetpack:mergeDebugNativeLibs': NO_SOURCE,
        ':jetpack:mergeDebugResources': FROM_CACHE,
        ':jetpack:mergeDebugShaders': FROM_CACHE,
        ':jetpack:mergeExtDexDebug': FROM_CACHE,
        ':jetpack:mergeLibDexDebug': FROM_CACHE,
        ':jetpack:mergeProjectDexDebug': FROM_CACHE,
        ':jetpack:packageDebug': SUCCESS,
        ':jetpack:preBuild': UP_TO_DATE,
        ':jetpack:preDebugBuild': UP_TO_DATE,
        ':jetpack:processApplicationManifestDebugForBundle': SUCCESS,
        ':jetpack:processDebugJavaRes': NO_SOURCE,
        ':jetpack:processDebugMainManifest': FROM_CACHE,
        ':jetpack:processDebugManifest': FROM_CACHE,
        ':jetpack:processDebugManifestForPackage': FROM_CACHE,
        ':jetpack:processDebugResources': FROM_CACHE,
        ':jetpack:processManifestDebugForFeature': FROM_CACHE,
        ':jetpack:stripDebugDebugSymbols': NO_SOURCE,
        ':memory:assembleDebug': SUCCESS,
        ':memory:checkDebugAarMetadata': SUCCESS,
        ':memory:checkDebugDuplicateClasses': SUCCESS,
        ':memory:compileDebugAidl': NO_SOURCE,
        ':memory:compileDebugJavaWithJavac': FROM_CACHE,
        ':memory:compileDebugRenderscript': NO_SOURCE,
        ':memory:compileDebugShaders': NO_SOURCE,
        ':memory:compressDebugAssets': FROM_CACHE,
        ':memory:createDebugApkListingFileRedirect': SUCCESS,
        ':memory:createDebugCompatibleScreenManifests': SUCCESS,
        ':memory:desugarDebugFileDependencies': FROM_CACHE,
        ':memory:dexBuilderDebug': FROM_CACHE,
        ':memory:extractDeepLinksDebug': FROM_CACHE,
        ':memory:featureDebugWriter': SUCCESS,
        ':memory:generateDebugAssets': UP_TO_DATE,
        ':memory:generateDebugBuildConfig': FROM_CACHE,
        ':memory:generateDebugFeatureTransitiveDeps': SUCCESS,
        ':memory:generateDebugResValues': FROM_CACHE,
        ':memory:generateDebugResources': UP_TO_DATE,
        ':memory:javaPreCompileDebug': FROM_CACHE,
        ':memory:mapDebugSourceSetPaths': SUCCESS,
        ':memory:mergeDebugAssets': FROM_CACHE,
        ':memory:mergeDebugJavaResource': SUCCESS,
        ':memory:mergeDebugJniLibFolders': FROM_CACHE,
        ':memory:mergeDebugNativeLibs': NO_SOURCE,
        ':memory:mergeDebugResources': FROM_CACHE,
        ':memory:mergeDebugShaders': FROM_CACHE,
        ':memory:mergeExtDexDebug': FROM_CACHE,
        ':memory:mergeLibDexDebug': FROM_CACHE,
        ':memory:mergeProjectDexDebug': FROM_CACHE,
        ':memory:packageDebug': SUCCESS,
        ':memory:preBuild': UP_TO_DATE,
        ':memory:preDebugBuild': UP_TO_DATE,
        ':memory:processApplicationManifestDebugForBundle': SUCCESS,
        ':memory:processDebugJavaRes': NO_SOURCE,
        ':memory:processDebugMainManifest': FROM_CACHE,
        ':memory:processDebugManifest': FROM_CACHE,
        ':memory:processDebugManifestForPackage': FROM_CACHE,
        ':memory:processDebugResources': FROM_CACHE,
        ':memory:processManifestDebugForFeature': FROM_CACHE,
        ':memory:stripDebugDebugSymbols': NO_SOURCE,
        ':penguinswim:assembleDebug': SUCCESS,
        ':penguinswim:checkDebugAarMetadata': SUCCESS,
        ':penguinswim:checkDebugDuplicateClasses': SUCCESS,
        ':penguinswim:compileDebugAidl': NO_SOURCE,
        ':penguinswim:compileDebugJavaWithJavac': FROM_CACHE,
        ':penguinswim:compileDebugRenderscript': NO_SOURCE,
        ':penguinswim:compileDebugShaders': NO_SOURCE,
        ':penguinswim:compressDebugAssets': FROM_CACHE,
        ':penguinswim:createDebugApkListingFileRedirect': SUCCESS,
        ':penguinswim:createDebugCompatibleScreenManifests': SUCCESS,
        ':penguinswim:desugarDebugFileDependencies': FROM_CACHE,
        ':penguinswim:dexBuilderDebug': FROM_CACHE,
        ':penguinswim:extractDeepLinksDebug': FROM_CACHE,
        ':penguinswim:featureDebugWriter': SUCCESS,
        ':penguinswim:generateDebugAssets': UP_TO_DATE,
        ':penguinswim:generateDebugBuildConfig': FROM_CACHE,
        ':penguinswim:generateDebugFeatureTransitiveDeps': SUCCESS,
        ':penguinswim:generateDebugResValues': FROM_CACHE,
        ':penguinswim:generateDebugResources': UP_TO_DATE,
        ':penguinswim:javaPreCompileDebug': FROM_CACHE,
        ':penguinswim:mapDebugSourceSetPaths': SUCCESS,
        ':penguinswim:mergeDebugAssets': FROM_CACHE,
        ':penguinswim:mergeDebugJavaResource': SUCCESS,
        ':penguinswim:mergeDebugJniLibFolders': FROM_CACHE,
        ':penguinswim:mergeDebugNativeLibs': NO_SOURCE,
        ':penguinswim:mergeDebugResources': FROM_CACHE,
        ':penguinswim:mergeDebugShaders': FROM_CACHE,
        ':penguinswim:mergeExtDexDebug': FROM_CACHE,
        ':penguinswim:mergeLibDexDebug': FROM_CACHE,
        ':penguinswim:mergeProjectDexDebug': FROM_CACHE,
        ':penguinswim:packageDebug': SUCCESS,
        ':penguinswim:preBuild': UP_TO_DATE,
        ':penguinswim:preDebugBuild': UP_TO_DATE,
        ':penguinswim:processApplicationManifestDebugForBundle': SUCCESS,
        ':penguinswim:processDebugJavaRes': NO_SOURCE,
        ':penguinswim:processDebugMainManifest': FROM_CACHE,
        ':penguinswim:processDebugManifest': FROM_CACHE,
        ':penguinswim:processDebugManifestForPackage': FROM_CACHE,
        ':penguinswim:processDebugResources': FROM_CACHE,
        ':penguinswim:processManifestDebugForFeature': FROM_CACHE,
        ':penguinswim:stripDebugDebugSymbols': NO_SOURCE,
        ':playgames:assembleDebug': SUCCESS,
        ':playgames:bundleDebugAar': SUCCESS,
        ':playgames:bundleLibCompileToJarDebug': SUCCESS,
        ':playgames:bundleLibResDebug': NO_SOURCE,
        ':playgames:bundleLibRuntimeToJarDebug': SUCCESS,
        ':playgames:compileDebugAidl': NO_SOURCE,
        ':playgames:compileDebugJavaWithJavac': FROM_CACHE,
        ':playgames:compileDebugLibraryResources': FROM_CACHE,
        ':playgames:compileDebugRenderscript': NO_SOURCE,
        ':playgames:compileDebugShaders': NO_SOURCE,
        ':playgames:copyDebugJniLibsProjectAndLocalJars': SUCCESS,
        ':playgames:copyDebugJniLibsProjectOnly': SUCCESS,
        ':playgames:createFullJarDebug': FROM_CACHE,
        ':playgames:extractDebugAnnotations': FROM_CACHE,
        ':playgames:extractDeepLinksDebug': FROM_CACHE,
        ':playgames:extractDeepLinksForAarDebug': FROM_CACHE,
        ':playgames:generateDebugAssets': UP_TO_DATE,
        ':playgames:generateDebugBuildConfig': FROM_CACHE,
        ':playgames:generateDebugRFile': FROM_CACHE,
        ':playgames:generateDebugResValues': FROM_CACHE,
        ':playgames:generateDebugResources': UP_TO_DATE,
        ':playgames:javaPreCompileDebug': FROM_CACHE,
        ':playgames:mergeDebugConsumerProguardFiles': SUCCESS,
        ':playgames:mergeDebugGeneratedProguardFiles': SUCCESS,
        ':playgames:mergeDebugJavaResource': SUCCESS,
        ':playgames:mergeDebugJniLibFolders': FROM_CACHE,
        ':playgames:mergeDebugNativeLibs': NO_SOURCE,
        ':playgames:mergeDebugShaders': FROM_CACHE,
        ':playgames:packageDebugAssets': FROM_CACHE,
        ':playgames:packageDebugRenderscript': NO_SOURCE,
        ':playgames:packageDebugResources': FROM_CACHE,
        ':playgames:parseDebugLocalResources': FROM_CACHE,
        ':playgames:preBuild': UP_TO_DATE,
        ':playgames:preDebugBuild': UP_TO_DATE,
        ':playgames:prepareDebugArtProfile': SUCCESS,
        ':playgames:prepareLintJarForPublish': SUCCESS,
        ':playgames:processDebugJavaRes': NO_SOURCE,
        ':playgames:processDebugManifest': FROM_CACHE,
        ':playgames:stripDebugDebugSymbols': NO_SOURCE,
        ':playgames:syncDebugLibJars': FROM_CACHE,
        ':playgames:writeDebugAarMetadata': SUCCESS,
        ':presenttoss:assembleDebug': SUCCESS,
        ':presenttoss:checkDebugAarMetadata': SUCCESS,
        ':presenttoss:checkDebugDuplicateClasses': SUCCESS,
        ':presenttoss:compileDebugAidl': NO_SOURCE,
        ':presenttoss:compileDebugJavaWithJavac': FROM_CACHE,
        ':presenttoss:compileDebugRenderscript': NO_SOURCE,
        ':presenttoss:compileDebugShaders': NO_SOURCE,
        ':presenttoss:compressDebugAssets': FROM_CACHE,
        ':presenttoss:createDebugApkListingFileRedirect': SUCCESS,
        ':presenttoss:createDebugCompatibleScreenManifests': SUCCESS,
        ':presenttoss:desugarDebugFileDependencies': FROM_CACHE,
        ':presenttoss:dexBuilderDebug': FROM_CACHE,
        ':presenttoss:extractDeepLinksDebug': FROM_CACHE,
        ':presenttoss:featureDebugWriter': SUCCESS,
        ':presenttoss:generateDebugAssets': UP_TO_DATE,
        ':presenttoss:generateDebugBuildConfig': FROM_CACHE,
        ':presenttoss:generateDebugFeatureTransitiveDeps': SUCCESS,
        ':presenttoss:generateDebugResValues': FROM_CACHE,
        ':presenttoss:generateDebugResources': UP_TO_DATE,
        ':presenttoss:javaPreCompileDebug': FROM_CACHE,
        ':presenttoss:mapDebugSourceSetPaths': SUCCESS,
        ':presenttoss:mergeDebugAssets': FROM_CACHE,
        ':presenttoss:mergeDebugJavaResource': SUCCESS,
        ':presenttoss:mergeDebugJniLibFolders': FROM_CACHE,
        ':presenttoss:mergeDebugNativeLibs': NO_SOURCE,
        ':presenttoss:mergeDebugResources': FROM_CACHE,
        ':presenttoss:mergeDebugShaders': FROM_CACHE,
        ':presenttoss:mergeExtDexDebug': FROM_CACHE,
        ':presenttoss:mergeLibDexDebug': FROM_CACHE,
        ':presenttoss:mergeProjectDexDebug': FROM_CACHE,
        ':presenttoss:packageDebug': SUCCESS,
        ':presenttoss:preBuild': UP_TO_DATE,
        ':presenttoss:preDebugBuild': UP_TO_DATE,
        ':presenttoss:processApplicationManifestDebugForBundle': SUCCESS,
        ':presenttoss:processDebugJavaRes': NO_SOURCE,
        ':presenttoss:processDebugMainManifest': FROM_CACHE,
        ':presenttoss:processDebugManifest': FROM_CACHE,
        ':presenttoss:processDebugManifestForPackage': FROM_CACHE,
        ':presenttoss:processDebugResources': FROM_CACHE,
        ':presenttoss:processManifestDebugForFeature': FROM_CACHE,
        ':presenttoss:stripDebugDebugSymbols': NO_SOURCE,
        ':rocketsleigh:assembleDebug': SUCCESS,
        ':rocketsleigh:checkDebugAarMetadata': SUCCESS,
        ':rocketsleigh:checkDebugDuplicateClasses': SUCCESS,
        ':rocketsleigh:checkKotlinGradlePluginConfigurationErrors': SUCCESS,
        ':rocketsleigh:compileDebugAidl': NO_SOURCE,
        ':rocketsleigh:compileDebugJavaWithJavac': FROM_CACHE,
        ':rocketsleigh:compileDebugKotlin': FROM_CACHE,
        ':rocketsleigh:compileDebugRenderscript': NO_SOURCE,
        ':rocketsleigh:compileDebugShaders': NO_SOURCE,
        ':rocketsleigh:compressDebugAssets': FROM_CACHE,
        ':rocketsleigh:createDebugApkListingFileRedirect': SUCCESS,
        ':rocketsleigh:createDebugCompatibleScreenManifests': SUCCESS,
        ':rocketsleigh:desugarDebugFileDependencies': FROM_CACHE,
        ':rocketsleigh:dexBuilderDebug': FROM_CACHE,
        ':rocketsleigh:extractDeepLinksDebug': FROM_CACHE,
        ':rocketsleigh:featureDebugWriter': SUCCESS,
        ':rocketsleigh:generateDebugAssets': UP_TO_DATE,
        ':rocketsleigh:generateDebugBuildConfig': FROM_CACHE,
        ':rocketsleigh:generateDebugFeatureTransitiveDeps': SUCCESS,
        ':rocketsleigh:generateDebugResValues': FROM_CACHE,
        ':rocketsleigh:generateDebugResources': UP_TO_DATE,
        ':rocketsleigh:javaPreCompileDebug': FROM_CACHE,
        ':rocketsleigh:mapDebugSourceSetPaths': SUCCESS,
        ':rocketsleigh:mergeDebugAssets': FROM_CACHE,
        ':rocketsleigh:mergeDebugJavaResource': SUCCESS,
        ':rocketsleigh:mergeDebugJniLibFolders': FROM_CACHE,
        ':rocketsleigh:mergeDebugNativeLibs': NO_SOURCE,
        ':rocketsleigh:mergeDebugResources': FROM_CACHE,
        ':rocketsleigh:mergeDebugShaders': FROM_CACHE,
        ':rocketsleigh:mergeExtDexDebug': FROM_CACHE,
        ':rocketsleigh:mergeLibDexDebug': FROM_CACHE,
        ':rocketsleigh:mergeProjectDexDebug': FROM_CACHE,
        ':rocketsleigh:packageDebug': SUCCESS,
        ':rocketsleigh:preBuild': UP_TO_DATE,
        ':rocketsleigh:preDebugBuild': UP_TO_DATE,
        ':rocketsleigh:processApplicationManifestDebugForBundle': SUCCESS,
        ':rocketsleigh:processDebugJavaRes': NO_SOURCE,
        ':rocketsleigh:processDebugMainManifest': FROM_CACHE,
        ':rocketsleigh:processDebugManifest': FROM_CACHE,
        ':rocketsleigh:processDebugManifestForPackage': FROM_CACHE,
        ':rocketsleigh:processDebugResources': FROM_CACHE,
        ':rocketsleigh:processManifestDebugForFeature': FROM_CACHE,
        ':rocketsleigh:stripDebugDebugSymbols': NO_SOURCE,
        ':santa-tracker:assembleDebug': SUCCESS,
        ':santa-tracker:bundleDebugClassesToCompileJar': SUCCESS,
        ':santa-tracker:bundleDebugClassesToRuntimeJar': SUCCESS,
        ':santa-tracker:checkDebugAarMetadata': SUCCESS,
        ':santa-tracker:checkDebugDuplicateClasses': SUCCESS,
        ':santa-tracker:checkDebugLibraries': SUCCESS,
        ':santa-tracker:checkKotlinGradlePluginConfigurationErrors': SUCCESS,
        ':santa-tracker:compileDebugAidl': NO_SOURCE,
        ':santa-tracker:compileDebugJavaWithJavac': FROM_CACHE,
        ':santa-tracker:compileDebugKotlin': FROM_CACHE,
        ':santa-tracker:compileDebugRenderscript': NO_SOURCE,
        ':santa-tracker:compileDebugShaders': NO_SOURCE,
        ':santa-tracker:compressDebugAssets': FROM_CACHE,
        ':santa-tracker:createDebugApkListingFileRedirect': SUCCESS,
        ':santa-tracker:createDebugCompatibleScreenManifests': SUCCESS,
        ':santa-tracker:dataBindingGenBaseClassesDebug': FROM_CACHE,
        ':santa-tracker:dataBindingMergeDependencyArtifactsDebug': SUCCESS,
        ':santa-tracker:desugarDebugFileDependencies': FROM_CACHE,
        ':santa-tracker:dexBuilderDebug': FROM_CACHE,
        ':santa-tracker:extractDeepLinksDebug': FROM_CACHE,
        ':santa-tracker:generateDebugAssets': UP_TO_DATE,
        ':santa-tracker:generateDebugBuildConfig': FROM_CACHE,
        ':santa-tracker:generateDebugFeatureMetadata': FROM_CACHE,
        ':santa-tracker:generateDebugFeatureTransitiveDeps': SUCCESS,
        ':santa-tracker:generateDebugResValues': FROM_CACHE,
        ':santa-tracker:generateDebugResources': UP_TO_DATE,
        ':santa-tracker:handleDebugMicroApk': FROM_CACHE,
        ':santa-tracker:javaPreCompileDebug': FROM_CACHE,
        ':santa-tracker:kaptDebugKotlin': FROM_CACHE,
        ':santa-tracker:kaptGenerateStubsDebugKotlin': FROM_CACHE,
        ':santa-tracker:mapDebugSourceSetPaths': SUCCESS,
        ':santa-tracker:mergeDebugAssets': FROM_CACHE,
        ':santa-tracker:mergeDebugJavaResource': SUCCESS,
        ':santa-tracker:mergeDebugJniLibFolders': FROM_CACHE,
        ':santa-tracker:mergeDebugNativeDebugMetadata': NO_SOURCE,
        ':santa-tracker:mergeDebugNativeLibs': NO_SOURCE,
        ':santa-tracker:mergeDebugResources': FROM_CACHE,
        ':santa-tracker:mergeDebugShaders': FROM_CACHE,
        ':santa-tracker:mergeExtDexDebug': FROM_CACHE,
        ':santa-tracker:mergeLibDexDebug': FROM_CACHE,
        ':santa-tracker:mergeProjectDexDebug': FROM_CACHE,
        ':santa-tracker:packageDebug': SUCCESS,
        ':santa-tracker:preBuild': UP_TO_DATE,
        ':santa-tracker:preDebugBuild': SUCCESS,
        ':santa-tracker:processDebugJavaRes': NO_SOURCE,
        ':santa-tracker:processDebugMainManifest': FROM_CACHE,
        ':santa-tracker:processDebugManifest': FROM_CACHE,
        ':santa-tracker:processDebugManifestForPackage': FROM_CACHE,
        ':santa-tracker:processDebugResources': FROM_CACHE,
        ':santa-tracker:signingConfigWriterDebug': SUCCESS,
        ':santa-tracker:stripDebugDebugSymbols': NO_SOURCE,
        ':santa-tracker:validateSigningDebug': SUCCESS,
        ':santa-tracker:writeDebugAppMetadata': SUCCESS,
        ':santa-tracker:writeDebugModuleMetadata': SUCCESS,
        ':santa-tracker:writeDebugSigningConfigVersions': SUCCESS,
        ':snowballrun:assembleDebug': SUCCESS,
        ':snowballrun:checkDebugAarMetadata': SUCCESS,
        ':snowballrun:checkDebugDuplicateClasses': SUCCESS,
        ':snowballrun:compileDebugAidl': NO_SOURCE,
        ':snowballrun:compileDebugJavaWithJavac': FROM_CACHE,
        ':snowballrun:compileDebugRenderscript': NO_SOURCE,
        ':snowballrun:compileDebugShaders': NO_SOURCE,
        ':snowballrun:compressDebugAssets': FROM_CACHE,
        ':snowballrun:createDebugApkListingFileRedirect': SUCCESS,
        ':snowballrun:createDebugCompatibleScreenManifests': SUCCESS,
        ':snowballrun:desugarDebugFileDependencies': FROM_CACHE,
        ':snowballrun:dexBuilderDebug': FROM_CACHE,
        ':snowballrun:extractDeepLinksDebug': FROM_CACHE,
        ':snowballrun:featureDebugWriter': SUCCESS,
        ':snowballrun:generateDebugAssets': UP_TO_DATE,
        ':snowballrun:generateDebugBuildConfig': FROM_CACHE,
        ':snowballrun:generateDebugFeatureTransitiveDeps': SUCCESS,
        ':snowballrun:generateDebugResValues': FROM_CACHE,
        ':snowballrun:generateDebugResources': UP_TO_DATE,
        ':snowballrun:javaPreCompileDebug': FROM_CACHE,
        ':snowballrun:mapDebugSourceSetPaths': SUCCESS,
        ':snowballrun:mergeDebugAssets': FROM_CACHE,
        ':snowballrun:mergeDebugJavaResource': SUCCESS,
        ':snowballrun:mergeDebugJniLibFolders': FROM_CACHE,
        ':snowballrun:mergeDebugNativeLibs': NO_SOURCE,
        ':snowballrun:mergeDebugResources': FROM_CACHE,
        ':snowballrun:mergeDebugShaders': FROM_CACHE,
        ':snowballrun:mergeExtDexDebug': FROM_CACHE,
        ':snowballrun:mergeLibDexDebug': FROM_CACHE,
        ':snowballrun:mergeProjectDexDebug': FROM_CACHE,
        ':snowballrun:packageDebug': SUCCESS,
        ':snowballrun:preBuild': UP_TO_DATE,
        ':snowballrun:preDebugBuild': UP_TO_DATE,
        ':snowballrun:processApplicationManifestDebugForBundle': SUCCESS,
        ':snowballrun:processDebugJavaRes': NO_SOURCE,
        ':snowballrun:processDebugMainManifest': FROM_CACHE,
        ':snowballrun:processDebugManifest': FROM_CACHE,
        ':snowballrun:processDebugManifestForPackage': FROM_CACHE,
        ':snowballrun:processDebugResources': FROM_CACHE,
        ':snowballrun:processManifestDebugForFeature': FROM_CACHE,
        ':snowballrun:stripDebugDebugSymbols': NO_SOURCE,
        ':tracker:assembleDebug': SUCCESS,
        ':tracker:bundleDebugAar': SUCCESS,
        ':tracker:bundleLibCompileToJarDebug': SUCCESS,
        ':tracker:bundleLibResDebug': SUCCESS,
        ':tracker:bundleLibRuntimeToJarDebug': SUCCESS,
        ':tracker:checkKotlinGradlePluginConfigurationErrors': SUCCESS,
        ':tracker:compileDebugAidl': NO_SOURCE,
        ':tracker:compileDebugJavaWithJavac': SUCCESS,
        ':tracker:compileDebugKotlin': FROM_CACHE,
        ':tracker:compileDebugLibraryResources': FROM_CACHE,
        ':tracker:compileDebugRenderscript': NO_SOURCE,
        ':tracker:compileDebugShaders': NO_SOURCE,
        ':tracker:copyDebugJniLibsProjectAndLocalJars': SUCCESS,
        ':tracker:copyDebugJniLibsProjectOnly': SUCCESS,
        ':tracker:createFullJarDebug': FROM_CACHE,
        ':tracker:extractDebugAnnotations': FROM_CACHE,
        ':tracker:extractDeepLinksDebug': FROM_CACHE,
        ':tracker:extractDeepLinksForAarDebug': FROM_CACHE,
        ':tracker:generateDebugAssets': UP_TO_DATE,
        ':tracker:generateDebugBuildConfig': FROM_CACHE,
        ':tracker:generateDebugRFile': FROM_CACHE,
        ':tracker:generateDebugResValues': FROM_CACHE,
        ':tracker:generateDebugResources': UP_TO_DATE,
        ':tracker:javaPreCompileDebug': FROM_CACHE,
        ':tracker:kaptDebugKotlin': SUCCESS,
        ':tracker:kaptGenerateStubsDebugKotlin': FROM_CACHE,
        ':tracker:mergeDebugConsumerProguardFiles': SUCCESS,
        ':tracker:mergeDebugGeneratedProguardFiles': SUCCESS,
        ':tracker:mergeDebugJavaResource': SUCCESS,
        ':tracker:mergeDebugJniLibFolders': FROM_CACHE,
        ':tracker:mergeDebugNativeLibs': NO_SOURCE,
        ':tracker:mergeDebugShaders': FROM_CACHE,
        ':tracker:packageDebugAssets': FROM_CACHE,
        ':tracker:packageDebugRenderscript': NO_SOURCE,
        ':tracker:packageDebugResources': FROM_CACHE,
        ':tracker:parseDebugLocalResources': FROM_CACHE,
        ':tracker:preBuild': UP_TO_DATE,
        ':tracker:preDebugBuild': UP_TO_DATE,
        ':tracker:prepareDebugArtProfile': SUCCESS,
        ':tracker:prepareLintJarForPublish': SUCCESS,
        ':tracker:processDebugJavaRes': NO_SOURCE,
        ':tracker:processDebugManifest': FROM_CACHE,
        ':tracker:stripDebugDebugSymbols': NO_SOURCE,
        ':tracker:syncDebugLibJars': FROM_CACHE,
        ':tracker:writeDebugAarMetadata': SUCCESS,
        ':wearable:assembleDebug': SUCCESS,
        ':wearable:checkDebugAarMetadata': SUCCESS,
        ':wearable:checkDebugDuplicateClasses': SUCCESS,
        ':wearable:checkKotlinGradlePluginConfigurationErrors': SUCCESS,
        ':wearable:compileDebugAidl': NO_SOURCE,
        ':wearable:compileDebugJavaWithJavac': FROM_CACHE,
        ':wearable:compileDebugKotlin': FROM_CACHE,
        ':wearable:compileDebugRenderscript': NO_SOURCE,
        ':wearable:compileDebugShaders': NO_SOURCE,
        ':wearable:compressDebugAssets': FROM_CACHE,
        ':wearable:createDebugApkListingFileRedirect': SUCCESS,
        ':wearable:createDebugCompatibleScreenManifests': SUCCESS,
        ':wearable:desugarDebugFileDependencies': FROM_CACHE,
        ':wearable:dexBuilderDebug': FROM_CACHE,
        ':wearable:extractDeepLinksDebug': FROM_CACHE,
        ':wearable:generateDebugAssets': UP_TO_DATE,
        ':wearable:generateDebugBuildConfig': FROM_CACHE,
        ':wearable:generateDebugResValues': FROM_CACHE,
        ':wearable:generateDebugResources': UP_TO_DATE,
        ':wearable:javaPreCompileDebug': FROM_CACHE,
        ':wearable:kaptDebugKotlin': SKIPPED,
        ':wearable:kaptGenerateStubsDebugKotlin': SKIPPED,
        ':wearable:mapDebugSourceSetPaths': SUCCESS,
        ':wearable:mergeDebugAssets': FROM_CACHE,
        ':wearable:mergeDebugJavaResource': SUCCESS,
        ':wearable:mergeDebugJniLibFolders': FROM_CACHE,
        ':wearable:mergeDebugNativeDebugMetadata': NO_SOURCE,
        ':wearable:mergeDebugNativeLibs': NO_SOURCE,
        ':wearable:mergeDebugResources': FROM_CACHE,
        ':wearable:mergeDebugShaders': FROM_CACHE,
        ':wearable:mergeExtDexDebug': FROM_CACHE,
        ':wearable:mergeLibDexDebug': FROM_CACHE,
        ':wearable:mergeProjectDexDebug': FROM_CACHE,
        ':wearable:packageDebug': SUCCESS,
        ':wearable:preBuild': UP_TO_DATE,
        ':wearable:preDebugBuild': UP_TO_DATE,
        ':wearable:processDebugJavaRes': NO_SOURCE,
        ':wearable:processDebugMainManifest': FROM_CACHE,
        ':wearable:processDebugManifest': FROM_CACHE,
        ':wearable:processDebugManifestForPackage': FROM_CACHE,
        ':wearable:processDebugResources': FROM_CACHE,
        ':wearable:stripDebugDebugSymbols': NO_SOURCE,
        ':wearable:validateSigningDebug': SUCCESS,
        ':wearable:writeDebugAppMetadata': SUCCESS,
        ':wearable:writeDebugSigningConfigVersions': SUCCESS,
    ]
}
