/*
 * Copyright 2020 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.gradle.api.plugins.catalog.internal

import com.google.common.collect.Interners
import groovy.transform.Canonical
import org.gradle.api.internal.catalog.DefaultVersionCatalog
import org.gradle.api.internal.catalog.DefaultVersionCatalogBuilder
import org.gradle.api.internal.catalog.parser.TomlCatalogFileParser
import org.gradle.util.TestUtil
import spock.lang.Specification
import spock.lang.Subject

import java.util.function.Supplier

class TomlWriterTest extends Specification {

    private StringWriter output = new StringWriter()
    private Model sourceModel
    private Model outputModel

    @Subject
    private TomlWriter writer = new TomlWriter(output)

    def "generates an equivalent file from an input (#file)"() {
        parse(file)

        when:
        generateFromModel()

        then:
        outputModel == sourceModel

        where:
        file << [
            'dependencies',
            'dependencies-notations',
            'plugin-notations',
        ]
    }

    def "generated file contains model version"() {
        when:
        writer.generate(Stub(DefaultVersionCatalog))

        then:
        output.toString().contains """#
# This file has been generated by Gradle and is intended to be consumed by Gradle
#
[metadata]
format.version = "1.1"
"""
    }

    private void generateFromModel() {
        writer.generate(sourceModel.deps)
        outputModel = parse(new ByteArrayInputStream(output.toString().getBytes("utf-8")))
    }

    private Model parse(String fileName) {
        sourceModel = parse(this.class.getResourceAsStream("${fileName}.toml"))
    }

    private Model parse(InputStream ins) {
        def builder = new DefaultVersionCatalogBuilder("libs",
            Interners.newStrongInterner(),
            Interners.newStrongInterner(),
            TestUtil.objectFactory(),
            Stub(Supplier))
        ins.withCloseable {
            TomlCatalogFileParser.parse(it, builder)
        }
        return new Model(builder.build())
    }

    @Canonical
    private static class Model {
        DefaultVersionCatalog deps
    }
}
