/*
 * Copyright 2019 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.gradle.internal.execution.steps

import com.google.common.collect.ImmutableSortedMap
import org.gradle.internal.execution.AfterPreviousExecutionContext
import org.gradle.internal.execution.BeforeExecutionContext
import org.gradle.internal.execution.OutputSnapshotter
import org.gradle.internal.execution.UnitOfWork
import org.gradle.internal.execution.history.AfterPreviousExecutionState
import org.gradle.internal.execution.history.ExecutionHistoryStore
import org.gradle.internal.execution.history.OverlappingOutputDetector
import org.gradle.internal.fingerprint.CurrentFileCollectionFingerprint
import org.gradle.internal.hash.ClassLoaderHierarchyHasher
import org.gradle.internal.hash.HashCode
import org.gradle.internal.snapshot.FileSystemSnapshot
import org.gradle.internal.snapshot.ValueSnapshot
import org.gradle.internal.snapshot.ValueSnapshotter
import org.gradle.internal.snapshot.impl.ImplementationSnapshot

import static org.gradle.internal.execution.UnitOfWork.IdentityKind.NON_IDENTITY
import static org.gradle.internal.execution.UnitOfWork.InputPropertyType.NON_INCREMENTAL
import static org.gradle.internal.execution.UnitOfWork.OverlappingOutputHandling.DETECT_OVERLAPS
import static org.gradle.internal.execution.UnitOfWork.OverlappingOutputHandling.IGNORE_OVERLAPS
import static org.gradle.internal.execution.steps.CaptureStateBeforeExecutionStep.Operation.Result

class CaptureStateBeforeExecutionStepTest extends StepSpec<AfterPreviousExecutionContext> {

    def classloaderHierarchyHasher = Mock(ClassLoaderHierarchyHasher)
    def outputSnapshotter = Mock(OutputSnapshotter)
    def valueSnapshotter = Mock(ValueSnapshotter)
    def implementationSnapshot = ImplementationSnapshot.of("MyWorkClass", HashCode.fromInt(1234))
    def overlappingOutputDetector = Mock(OverlappingOutputDetector)
    def executionHistoryStore = Mock(ExecutionHistoryStore)

    def step = new CaptureStateBeforeExecutionStep(buildOperationExecutor, classloaderHierarchyHasher, outputSnapshotter, overlappingOutputDetector, valueSnapshotter, delegate)

    @Override
    protected AfterPreviousExecutionContext createContext() {
        Stub(AfterPreviousExecutionContext) {
            getInputProperties() >> ImmutableSortedMap.of()
            getInputFileProperties() >> ImmutableSortedMap.of()
        }
    }

    def setup() {
        _ * work.history >> Optional.of(executionHistoryStore)
    }

    def "no state is captured when task history is not maintained"() {
        when:
        step.execute(work, context)
        then:
        assertNoOperation()
        _ * work.history >> Optional.empty()
        1 * delegate.execute(work, _ as BeforeExecutionContext) >> { UnitOfWork work, BeforeExecutionContext beforeExecution ->
            assert !beforeExecution.beforeExecutionState.present
        }
        0 * _
    }

    def "implementations are snapshotted"() {
        def additionalImplementations = [
            ImplementationSnapshot.of("FirstAction", HashCode.fromInt(2345)),
            ImplementationSnapshot.of("SecondAction", HashCode.fromInt(3456))
        ]

        when:
        step.execute(work, context)

        then:
        _ * work.visitImplementations(_) >> { UnitOfWork.ImplementationVisitor visitor ->
            visitor.visitImplementation(implementationSnapshot)
            additionalImplementations.each {
                visitor.visitImplementation(it)
            }
        }
        interaction { snapshotState() }
        1 * delegate.execute(work, _ as BeforeExecutionContext) >> { UnitOfWork work, BeforeExecutionContext beforeExecution ->
            def state = beforeExecution.beforeExecutionState.get()
            assert !state.detectedOverlappingOutputs.present
            assert state.implementation == implementationSnapshot
            assert state.additionalImplementations == additionalImplementations
        }
        0 * _

        assertOperationForInputsBeforeExecution()
    }

    def "input properties are snapshotted"() {
        def inputPropertyValue = 'myValue'
        def valueSnapshot = Mock(ValueSnapshot)

        when:
        step.execute(work, context)
        then:
        _ * work.visitInputs(_) >> { UnitOfWork.InputVisitor visitor ->
            visitor.visitInputProperty("inputString", NON_IDENTITY) { inputPropertyValue }
        }
        1 * valueSnapshotter.snapshot(inputPropertyValue) >> valueSnapshot
        interaction { snapshotState() }
        1 * delegate.execute(work, _ as BeforeExecutionContext) >> { UnitOfWork work, BeforeExecutionContext beforeExecution ->
            def state = beforeExecution.beforeExecutionState.get()
            assert !state.detectedOverlappingOutputs.present
            assert state.inputProperties == ImmutableSortedMap.<String, ValueSnapshot>of('inputString', valueSnapshot)
        }
        0 * _

        assertOperationForInputsBeforeExecution()
    }

    def "uses previous input property snapshots"() {
        def inputPropertyValue = 'myValue'
        def valueSnapshot = Mock(ValueSnapshot)
        def afterPreviousExecutionState = Mock(AfterPreviousExecutionState)

        when:
        step.execute(work, context)
        then:
        _ * context.afterPreviousExecutionState >> Optional.of(afterPreviousExecutionState)
        1 * afterPreviousExecutionState.inputProperties >> ImmutableSortedMap.<String, ValueSnapshot>of("inputString", valueSnapshot)
        1 * afterPreviousExecutionState.outputFilesProducedByWork >> ImmutableSortedMap.of()
        _ * work.visitInputs(_) >> { UnitOfWork.InputVisitor visitor ->
            visitor.visitInputProperty("inputString", NON_IDENTITY) { inputPropertyValue }
        }
        1 * valueSnapshotter.snapshot(inputPropertyValue, valueSnapshot) >> valueSnapshot
        interaction { snapshotState() }
        1 * delegate.execute(work, _ as BeforeExecutionContext) >> { UnitOfWork work, BeforeExecutionContext beforeExecution ->
            def state = beforeExecution.beforeExecutionState.get()
            assert !state.detectedOverlappingOutputs.present
            assert state.inputProperties == ImmutableSortedMap.<String, ValueSnapshot>of('inputString', valueSnapshot)
        }
        0 * _

        assertOperationForInputsBeforeExecution()
    }

    def "input file properties are fingerprinted"() {
        def fingerprint = Mock(CurrentFileCollectionFingerprint)

        when:
        step.execute(work, context)

        then:
        _ * work.visitInputs(_) >> { UnitOfWork.InputVisitor visitor ->
            visitor.visitInputFileProperty("inputFile", NON_INCREMENTAL, NON_IDENTITY, "ignored", { -> fingerprint })
        }
        interaction { snapshotState() }
        1 * delegate.execute(work, _ as BeforeExecutionContext) >> { UnitOfWork work, BeforeExecutionContext beforeExecution ->
            def state = beforeExecution.beforeExecutionState.get()
            assert !state.detectedOverlappingOutputs.present
            assert state.inputFileProperties == ImmutableSortedMap.<String, CurrentFileCollectionFingerprint>of('inputFile', fingerprint)
        }
        0 * _

        assertOperationForInputsBeforeExecution()
    }

    def "output file properties are snapshotted"() {
        def outputDirSnapshot = Mock(FileSystemSnapshot)

        when:
        step.execute(work, context)

        then:
        _ * outputSnapshotter.snapshotOutputs(work, _) >> ImmutableSortedMap.<String, FileSystemSnapshot>of("outputDir", outputDirSnapshot)
        interaction { snapshotState() }
        1 * delegate.execute(work, _ as BeforeExecutionContext) >> { UnitOfWork work, BeforeExecutionContext beforeExecution ->
            def state = beforeExecution.beforeExecutionState.get()
            assert !state.detectedOverlappingOutputs.present
            assert state.outputFileLocationSnapshots == ImmutableSortedMap.of('outputDir', outputDirSnapshot)
        }
        0 * _

        assertOperationForInputsBeforeExecution()
    }

    def "detects overlapping outputs when instructed"() {
        def afterPreviousExecutionState = Mock(AfterPreviousExecutionState)
        def afterPreviousOutputSnapshot = Mock(FileSystemSnapshot)
        def afterPreviousOutputSnapshots = ImmutableSortedMap.of("outputDir", afterPreviousOutputSnapshot)
        def beforeExecutionOutputSnapshot = Mock(FileSystemSnapshot)
        def beforeExecutionOutputSnapshots = ImmutableSortedMap.of("outputDir", beforeExecutionOutputSnapshot)

        when:
        step.execute(work, context)
        then:
        _ * context.afterPreviousExecutionState >> Optional.of(afterPreviousExecutionState)
        1 * afterPreviousExecutionState.inputProperties >> ImmutableSortedMap.of()
        1 * afterPreviousExecutionState.outputFilesProducedByWork >> afterPreviousOutputSnapshots
        _ * outputSnapshotter.snapshotOutputs(work, _) >> beforeExecutionOutputSnapshots

        _ * work.overlappingOutputHandling >> DETECT_OVERLAPS
        1 * overlappingOutputDetector.detect(afterPreviousOutputSnapshots, beforeExecutionOutputSnapshots) >> null

        interaction { snapshotState() }
        1 * delegate.execute(work, _ as BeforeExecutionContext) >> { UnitOfWork work, BeforeExecutionContext beforeExecution ->
            def state = beforeExecution.beforeExecutionState.get()
            assert !state.detectedOverlappingOutputs.present
            assert state.outputFileLocationSnapshots == beforeExecutionOutputSnapshots
        }
        0 * _

        assertOperationForInputsBeforeExecution()
    }

    void snapshotState() {
        _ * context.afterPreviousExecutionState >> Optional.empty()
        _ * work.visitImplementations(_ as UnitOfWork.ImplementationVisitor) >> { UnitOfWork.ImplementationVisitor visitor ->
            visitor.visitImplementation(implementationSnapshot)
        }
        _ * work.visitInputs(_ as UnitOfWork.InputVisitor)
        _ * work.overlappingOutputHandling >> IGNORE_OVERLAPS
        _ * outputSnapshotter.snapshotOutputs(work, _) >> ImmutableSortedMap.of()
        _ * context.history >> Optional.of(executionHistoryStore)
    }

    private void assertOperationForInputsBeforeExecution() {
        withOnlyOperation(CaptureStateBeforeExecutionStep.Operation) {
            assert it.descriptor.displayName == "Snapshot inputs and outputs before executing job ':test'"
            assert it.result == Result.INSTANCE
        }
    }
}
