/*
 * Copyright 2020 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.gradle.smoketests

import org.gradle.integtests.fixtures.UnsupportedWithConfigurationCache
import org.gradle.testkit.runner.BuildResult
import org.gradle.util.Requires
import org.gradle.util.TestPrecondition
import spock.lang.Unroll

import static org.gradle.testkit.runner.TaskOutcome.FROM_CACHE
import static org.gradle.testkit.runner.TaskOutcome.NO_SOURCE
import static org.gradle.testkit.runner.TaskOutcome.SUCCESS
import static org.gradle.testkit.runner.TaskOutcome.UP_TO_DATE
import static org.hamcrest.CoreMatchers.equalTo
import static org.hamcrest.CoreMatchers.not
import static org.junit.Assume.assumeThat


@Requires(TestPrecondition.JDK11_OR_EARLIER)
class AndroidSantaTrackerJavaCachingSmokeTest extends AbstractAndroidSantaTrackerSmokeTest {

    // TODO:configuration-cache remove once fixed upstream
    @Override
    protected int maxConfigurationCacheProblems() {
        return 100
    }

    @Unroll
    @UnsupportedWithConfigurationCache(iterationMatchers = [AGP_3_ITERATION_MATCHER, AGP_4_0_ITERATION_MATCHER])
    def "can cache Santa Tracker Java Android application (agp=#agpVersion)"() {

        // TODO remove once next 4.2 is available
        assumeThat(agpVersion, not(equalTo("4.2.0-alpha07")))

        given:
        def originalDir = temporaryFolder.createDir("original")
        def relocatedDir = temporaryFolder.createDir("relocated")

        and:
        setupCopyOfSantaTracker(originalDir, 'Java', agpVersion)
        setupCopyOfSantaTracker(relocatedDir, 'Java', agpVersion)

        when: 'clean build'
        buildLocation(originalDir, agpVersion)

        then:
        assertConfigurationCacheStateStored()

        when: 'up-to-date build, reusing configuration cache when enabled'
        buildLocation(originalDir, agpVersion)

        then:
        assertConfigurationCacheStateLoaded()

        when: 'clean cached build'
        BuildResult relocatedResult = buildLocation(relocatedDir, agpVersion)

        then:
        assertConfigurationCacheStateStored()

        and:
        def expectedResults = agpVersion.startsWith('3.6')
            ? EXPECTED_RESULTS_3_6
            : agpVersion.startsWith('4.0')
            ? EXPECTED_RESULTS_4_0
            : agpVersion.startsWith('4.1')
            ? EXPECTED_RESULTS_4_1
            : EXPECTED_RESULTS_4_2
        verify(relocatedResult, expectedResults)

        when: 'clean cached build, reusing configuration cache when enabled'
        cleanLocation(relocatedDir, agpVersion)
        buildLocation(relocatedDir, agpVersion)

        then:
        assertConfigurationCacheStateLoaded()

        where:
        agpVersion << TESTED_AGP_VERSIONS
    }

    private static final EXPECTED_RESULTS_4_2 = [
        ':common:assembleDebug': SUCCESS,
        ':common:bundleDebugAar': SUCCESS,
        ':common:bundleLibCompileToJarDebug': FROM_CACHE,
        ':common:bundleLibResDebug': NO_SOURCE,
        ':common:bundleLibRuntimeToDirDebug': FROM_CACHE,
        ':common:compileDebugAidl': NO_SOURCE,
        ':common:compileDebugJavaWithJavac': FROM_CACHE,
        ':common:compileDebugLibraryResources': SUCCESS,
        ':common:compileDebugRenderscript': NO_SOURCE,
        ':common:compileDebugShaders': NO_SOURCE,
        ':common:compileDebugSources': UP_TO_DATE,
        ':common:copyDebugJniLibsProjectAndLocalJars': FROM_CACHE,
        ':common:copyDebugJniLibsProjectOnly': FROM_CACHE,
        ':common:extractDebugAnnotations': FROM_CACHE,
        ':common:extractDeepLinksDebug': FROM_CACHE,
        ':common:generateDebugAssets': UP_TO_DATE,
        ':common:generateDebugBuildConfig': FROM_CACHE,
        ':common:generateDebugRFile': FROM_CACHE,
        ':common:generateDebugResValues': FROM_CACHE,
        ':common:generateDebugResources': UP_TO_DATE,
        ':common:javaPreCompileDebug': FROM_CACHE,
        ':common:mergeDebugConsumerProguardFiles': FROM_CACHE,
        ':common:mergeDebugGeneratedProguardFiles': FROM_CACHE,
        ':common:mergeDebugJavaResource': FROM_CACHE,
        ':common:mergeDebugJniLibFolders': FROM_CACHE,
        ':common:mergeDebugNativeLibs': NO_SOURCE,
        ':common:mergeDebugShaders': FROM_CACHE,
        ':common:packageDebugAssets': FROM_CACHE,
        ':common:packageDebugRenderscript': NO_SOURCE,
        ':common:packageDebugResources': FROM_CACHE,
        ':common:parseDebugLocalResources': FROM_CACHE,
        ':common:preBuild': UP_TO_DATE,
        ':common:preDebugBuild': UP_TO_DATE,
        ':common:prepareLintJarForPublish': SUCCESS,
        ':common:processDebugJavaRes': NO_SOURCE,
        ':common:processDebugManifest': FROM_CACHE,
        ':common:stripDebugDebugSymbols': NO_SOURCE,
        ':common:syncDebugLibJars': FROM_CACHE,
        ':common:writeDebugAarMetadata': FROM_CACHE,
        ':dasherdancer:assembleDebug': SUCCESS,
        ':dasherdancer:bundleDebugAar': SUCCESS,
        ':dasherdancer:bundleLibCompileToJarDebug': FROM_CACHE,
        ':dasherdancer:bundleLibResDebug': NO_SOURCE,
        ':dasherdancer:bundleLibRuntimeToDirDebug': FROM_CACHE,
        ':dasherdancer:compileDebugAidl': NO_SOURCE,
        ':dasherdancer:compileDebugJavaWithJavac': FROM_CACHE,
        ':dasherdancer:compileDebugLibraryResources': SUCCESS,
        ':dasherdancer:compileDebugRenderscript': NO_SOURCE,
        ':dasherdancer:compileDebugShaders': NO_SOURCE,
        ':dasherdancer:compileDebugSources': UP_TO_DATE,
        ':dasherdancer:copyDebugJniLibsProjectAndLocalJars': FROM_CACHE,
        ':dasherdancer:copyDebugJniLibsProjectOnly': FROM_CACHE,
        ':dasherdancer:extractDebugAnnotations': FROM_CACHE,
        ':dasherdancer:extractDeepLinksDebug': FROM_CACHE,
        ':dasherdancer:generateDebugAssets': UP_TO_DATE,
        ':dasherdancer:generateDebugBuildConfig': FROM_CACHE,
        ':dasherdancer:generateDebugRFile': FROM_CACHE,
        ':dasherdancer:generateDebugResValues': FROM_CACHE,
        ':dasherdancer:generateDebugResources': UP_TO_DATE,
        ':dasherdancer:javaPreCompileDebug': FROM_CACHE,
        ':dasherdancer:mergeDebugConsumerProguardFiles': FROM_CACHE,
        ':dasherdancer:mergeDebugGeneratedProguardFiles': FROM_CACHE,
        ':dasherdancer:mergeDebugJavaResource': FROM_CACHE,
        ':dasherdancer:mergeDebugJniLibFolders': FROM_CACHE,
        ':dasherdancer:mergeDebugNativeLibs': NO_SOURCE,
        ':dasherdancer:mergeDebugShaders': FROM_CACHE,
        ':dasherdancer:packageDebugAssets': FROM_CACHE,
        ':dasherdancer:packageDebugRenderscript': NO_SOURCE,
        ':dasherdancer:packageDebugResources': FROM_CACHE,
        ':dasherdancer:parseDebugLocalResources': FROM_CACHE,
        ':dasherdancer:preBuild': UP_TO_DATE,
        ':dasherdancer:preDebugBuild': UP_TO_DATE,
        ':dasherdancer:prepareLintJarForPublish': SUCCESS,
        ':dasherdancer:processDebugJavaRes': NO_SOURCE,
        ':dasherdancer:processDebugManifest': FROM_CACHE,
        ':dasherdancer:stripDebugDebugSymbols': NO_SOURCE,
        ':dasherdancer:syncDebugLibJars': FROM_CACHE,
        ':dasherdancer:writeDebugAarMetadata': FROM_CACHE,
        ':doodles:assembleDebug': SUCCESS,
        ':doodles:bundleDebugAar': SUCCESS,
        ':doodles:bundleLibCompileToJarDebug': FROM_CACHE,
        ':doodles:bundleLibResDebug': NO_SOURCE,
        ':doodles:bundleLibRuntimeToDirDebug': FROM_CACHE,
        ':doodles:compileDebugAidl': NO_SOURCE,
        ':doodles:compileDebugJavaWithJavac': FROM_CACHE,
        ':doodles:compileDebugLibraryResources': SUCCESS,
        ':doodles:compileDebugRenderscript': NO_SOURCE,
        ':doodles:compileDebugShaders': NO_SOURCE,
        ':doodles:compileDebugSources': UP_TO_DATE,
        ':doodles:copyDebugJniLibsProjectAndLocalJars': FROM_CACHE,
        ':doodles:copyDebugJniLibsProjectOnly': FROM_CACHE,
        ':doodles:extractDebugAnnotations': FROM_CACHE,
        ':doodles:extractDeepLinksDebug': FROM_CACHE,
        ':doodles:generateDebugAssets': UP_TO_DATE,
        ':doodles:generateDebugBuildConfig': FROM_CACHE,
        ':doodles:generateDebugRFile': FROM_CACHE,
        ':doodles:generateDebugResValues': FROM_CACHE,
        ':doodles:generateDebugResources': UP_TO_DATE,
        ':doodles:javaPreCompileDebug': FROM_CACHE,
        ':doodles:mergeDebugConsumerProguardFiles': FROM_CACHE,
        ':doodles:mergeDebugGeneratedProguardFiles': FROM_CACHE,
        ':doodles:mergeDebugJavaResource': FROM_CACHE,
        ':doodles:mergeDebugJniLibFolders': FROM_CACHE,
        ':doodles:mergeDebugNativeLibs': NO_SOURCE,
        ':doodles:mergeDebugShaders': FROM_CACHE,
        ':doodles:packageDebugAssets': FROM_CACHE,
        ':doodles:packageDebugRenderscript': NO_SOURCE,
        ':doodles:packageDebugResources': FROM_CACHE,
        ':doodles:parseDebugLocalResources': FROM_CACHE,
        ':doodles:preBuild': UP_TO_DATE,
        ':doodles:preDebugBuild': UP_TO_DATE,
        ':doodles:prepareLintJarForPublish': SUCCESS,
        ':doodles:processDebugJavaRes': NO_SOURCE,
        ':doodles:processDebugManifest': FROM_CACHE,
        ':doodles:stripDebugDebugSymbols': NO_SOURCE,
        ':doodles:syncDebugLibJars': FROM_CACHE,
        ':doodles:writeDebugAarMetadata': FROM_CACHE,
        ':presentquest:assembleDebug': SUCCESS,
        ':presentquest:bundleDebugAar': SUCCESS,
        ':presentquest:bundleLibCompileToJarDebug': FROM_CACHE,
        ':presentquest:bundleLibResDebug': NO_SOURCE,
        ':presentquest:bundleLibRuntimeToDirDebug': FROM_CACHE,
        ':presentquest:compileDebugAidl': NO_SOURCE,
        ':presentquest:compileDebugJavaWithJavac': FROM_CACHE,
        ':presentquest:compileDebugLibraryResources': SUCCESS,
        ':presentquest:compileDebugRenderscript': NO_SOURCE,
        ':presentquest:compileDebugShaders': NO_SOURCE,
        ':presentquest:compileDebugSources': UP_TO_DATE,
        ':presentquest:copyDebugJniLibsProjectAndLocalJars': FROM_CACHE,
        ':presentquest:copyDebugJniLibsProjectOnly': FROM_CACHE,
        ':presentquest:extractDebugAnnotations': FROM_CACHE,
        ':presentquest:extractDeepLinksDebug': FROM_CACHE,
        ':presentquest:generateDebugAssets': UP_TO_DATE,
        ':presentquest:generateDebugBuildConfig': FROM_CACHE,
        ':presentquest:generateDebugRFile': FROM_CACHE,
        ':presentquest:generateDebugResValues': FROM_CACHE,
        ':presentquest:generateDebugResources': UP_TO_DATE,
        ':presentquest:javaPreCompileDebug': FROM_CACHE,
        ':presentquest:mergeDebugConsumerProguardFiles': FROM_CACHE,
        ':presentquest:mergeDebugGeneratedProguardFiles': FROM_CACHE,
        ':presentquest:mergeDebugJavaResource': FROM_CACHE,
        ':presentquest:mergeDebugJniLibFolders': FROM_CACHE,
        ':presentquest:mergeDebugNativeLibs': NO_SOURCE,
        ':presentquest:mergeDebugShaders': FROM_CACHE,
        ':presentquest:packageDebugAssets': FROM_CACHE,
        ':presentquest:packageDebugRenderscript': NO_SOURCE,
        ':presentquest:packageDebugResources': FROM_CACHE,
        ':presentquest:parseDebugLocalResources': FROM_CACHE,
        ':presentquest:preBuild': UP_TO_DATE,
        ':presentquest:preDebugBuild': UP_TO_DATE,
        ':presentquest:prepareLintJarForPublish': SUCCESS,
        ':presentquest:processDebugJavaRes': NO_SOURCE,
        ':presentquest:processDebugManifest': FROM_CACHE,
        ':presentquest:stripDebugDebugSymbols': NO_SOURCE,
        ':presentquest:syncDebugLibJars': FROM_CACHE,
        ':presentquest:writeDebugAarMetadata': FROM_CACHE,
        ':rocketsleigh:assembleDebug': SUCCESS,
        ':rocketsleigh:bundleDebugAar': SUCCESS,
        ':rocketsleigh:bundleLibCompileToJarDebug': FROM_CACHE,
        ':rocketsleigh:bundleLibResDebug': NO_SOURCE,
        ':rocketsleigh:bundleLibRuntimeToDirDebug': FROM_CACHE,
        ':rocketsleigh:compileDebugAidl': NO_SOURCE,
        ':rocketsleigh:compileDebugJavaWithJavac': FROM_CACHE,
        ':rocketsleigh:compileDebugLibraryResources': SUCCESS,
        ':rocketsleigh:compileDebugRenderscript': NO_SOURCE,
        ':rocketsleigh:compileDebugShaders': NO_SOURCE,
        ':rocketsleigh:compileDebugSources': UP_TO_DATE,
        ':rocketsleigh:copyDebugJniLibsProjectAndLocalJars': FROM_CACHE,
        ':rocketsleigh:copyDebugJniLibsProjectOnly': FROM_CACHE,
        ':rocketsleigh:extractDebugAnnotations': FROM_CACHE,
        ':rocketsleigh:extractDeepLinksDebug': FROM_CACHE,
        ':rocketsleigh:generateDebugAssets': UP_TO_DATE,
        ':rocketsleigh:generateDebugBuildConfig': FROM_CACHE,
        ':rocketsleigh:generateDebugRFile': FROM_CACHE,
        ':rocketsleigh:generateDebugResValues': FROM_CACHE,
        ':rocketsleigh:generateDebugResources': UP_TO_DATE,
        ':rocketsleigh:javaPreCompileDebug': FROM_CACHE,
        ':rocketsleigh:mergeDebugConsumerProguardFiles': FROM_CACHE,
        ':rocketsleigh:mergeDebugGeneratedProguardFiles': FROM_CACHE,
        ':rocketsleigh:mergeDebugJavaResource': FROM_CACHE,
        ':rocketsleigh:mergeDebugJniLibFolders': FROM_CACHE,
        ':rocketsleigh:mergeDebugNativeLibs': NO_SOURCE,
        ':rocketsleigh:mergeDebugShaders': FROM_CACHE,
        ':rocketsleigh:packageDebugAssets': FROM_CACHE,
        ':rocketsleigh:packageDebugRenderscript': NO_SOURCE,
        ':rocketsleigh:packageDebugResources': FROM_CACHE,
        ':rocketsleigh:parseDebugLocalResources': FROM_CACHE,
        ':rocketsleigh:preBuild': UP_TO_DATE,
        ':rocketsleigh:preDebugBuild': UP_TO_DATE,
        ':rocketsleigh:prepareLintJarForPublish': SUCCESS,
        ':rocketsleigh:processDebugJavaRes': NO_SOURCE,
        ':rocketsleigh:processDebugManifest': FROM_CACHE,
        ':rocketsleigh:stripDebugDebugSymbols': NO_SOURCE,
        ':rocketsleigh:syncDebugLibJars': FROM_CACHE,
        ':rocketsleigh:writeDebugAarMetadata': FROM_CACHE,
        ':santa-tracker:assembleDebug': SUCCESS,
        ':santa-tracker:assembleDevelopmentDebug': SUCCESS,
        ':santa-tracker:checkDevelopmentDebugAarMetadata': FROM_CACHE,
        ':santa-tracker:checkDevelopmentDebugDuplicateClasses': FROM_CACHE,
        ':santa-tracker:compileDevelopmentDebugAidl': NO_SOURCE,
        ':santa-tracker:compileDevelopmentDebugJavaWithJavac': FROM_CACHE,
        ':santa-tracker:compileDevelopmentDebugRenderscript': NO_SOURCE,
        ':santa-tracker:compileDevelopmentDebugShaders': NO_SOURCE,
        ':santa-tracker:compileDevelopmentDebugSources': UP_TO_DATE,
        ':santa-tracker:compressDevelopmentDebugAssets': FROM_CACHE,
        ':santa-tracker:createDevelopmentDebugCompatibleScreenManifests': FROM_CACHE,
        ':santa-tracker:desugarDevelopmentDebugFileDependencies': SUCCESS,
        ':santa-tracker:dexBuilderDevelopmentDebug': FROM_CACHE,
        ':santa-tracker:extractDeepLinksDevelopmentDebug': FROM_CACHE,
        ':santa-tracker:generateDevelopmentDebugAssets': UP_TO_DATE,
        ':santa-tracker:generateDevelopmentDebugBuildConfig': FROM_CACHE,
        ':santa-tracker:generateDevelopmentDebugResValues': FROM_CACHE,
        ':santa-tracker:generateDevelopmentDebugResources': UP_TO_DATE,
        ':santa-tracker:javaPreCompileDevelopmentDebug': FROM_CACHE,
        ':santa-tracker:mergeDevelopmentDebugAssets': FROM_CACHE,
        ':santa-tracker:mergeDevelopmentDebugJavaResource': FROM_CACHE,
        ':santa-tracker:mergeDevelopmentDebugJniLibFolders': FROM_CACHE,
        ':santa-tracker:mergeDevelopmentDebugNativeDebugMetadata': NO_SOURCE,
        ':santa-tracker:mergeDevelopmentDebugNativeLibs': FROM_CACHE,
        ':santa-tracker:mergeDevelopmentDebugResources': SUCCESS,
        ':santa-tracker:mergeDevelopmentDebugShaders': FROM_CACHE,
        ':santa-tracker:mergeExtDexDevelopmentDebug': FROM_CACHE,
        ':santa-tracker:mergeLibDexDevelopmentDebug': FROM_CACHE,
        ':santa-tracker:mergeProjectDexDevelopmentDebug': FROM_CACHE,
        ':santa-tracker:packageDevelopmentDebug': SUCCESS,
        ':santa-tracker:preBuild': UP_TO_DATE,
        ':santa-tracker:preDevelopmentDebugBuild': UP_TO_DATE,
        ':santa-tracker:processDevelopmentDebugJavaRes': NO_SOURCE,
        ':santa-tracker:processDevelopmentDebugMainManifest': FROM_CACHE,
        ':santa-tracker:processDevelopmentDebugManifest': FROM_CACHE,
        ':santa-tracker:processDevelopmentDebugManifestForPackage': FROM_CACHE,
        ':santa-tracker:processDevelopmentDebugResources': SUCCESS,
        ':santa-tracker:stripDevelopmentDebugDebugSymbols': NO_SOURCE,
        ':santa-tracker:validateSigningDevelopmentDebug': FROM_CACHE,
        ':village:assembleDebug': SUCCESS,
        ':village:bundleDebugAar': SUCCESS,
        ':village:bundleLibCompileToJarDebug': FROM_CACHE,
        ':village:bundleLibResDebug': NO_SOURCE,
        ':village:bundleLibRuntimeToDirDebug': FROM_CACHE,
        ':village:compileDebugAidl': NO_SOURCE,
        ':village:compileDebugJavaWithJavac': FROM_CACHE,
        ':village:compileDebugLibraryResources': SUCCESS,
        ':village:compileDebugRenderscript': NO_SOURCE,
        ':village:compileDebugShaders': NO_SOURCE,
        ':village:compileDebugSources': UP_TO_DATE,
        ':village:copyDebugJniLibsProjectAndLocalJars': FROM_CACHE,
        ':village:copyDebugJniLibsProjectOnly': FROM_CACHE,
        ':village:extractDebugAnnotations': FROM_CACHE,
        ':village:extractDeepLinksDebug': FROM_CACHE,
        ':village:generateDebugAssets': UP_TO_DATE,
        ':village:generateDebugBuildConfig': FROM_CACHE,
        ':village:generateDebugRFile': FROM_CACHE,
        ':village:generateDebugResValues': FROM_CACHE,
        ':village:generateDebugResources': UP_TO_DATE,
        ':village:javaPreCompileDebug': FROM_CACHE,
        ':village:mergeDebugConsumerProguardFiles': FROM_CACHE,
        ':village:mergeDebugGeneratedProguardFiles': FROM_CACHE,
        ':village:mergeDebugJavaResource': FROM_CACHE,
        ':village:mergeDebugJniLibFolders': FROM_CACHE,
        ':village:mergeDebugNativeLibs': NO_SOURCE,
        ':village:mergeDebugShaders': FROM_CACHE,
        ':village:packageDebugAssets': FROM_CACHE,
        ':village:packageDebugRenderscript': NO_SOURCE,
        ':village:packageDebugResources': FROM_CACHE,
        ':village:parseDebugLocalResources': FROM_CACHE,
        ':village:preBuild': UP_TO_DATE,
        ':village:preDebugBuild': UP_TO_DATE,
        ':village:prepareLintJarForPublish': SUCCESS,
        ':village:processDebugJavaRes': NO_SOURCE,
        ':village:processDebugManifest': FROM_CACHE,
        ':village:stripDebugDebugSymbols': NO_SOURCE,
        ':village:syncDebugLibJars': FROM_CACHE,
        ':village:writeDebugAarMetadata': FROM_CACHE,
    ]

    private static final EXPECTED_RESULTS_4_1 = [
        ':common:assembleDebug': SUCCESS,
        ':common:bundleDebugAar': SUCCESS,
        ':common:bundleLibCompileToJarDebug': FROM_CACHE,
        ':common:bundleLibResDebug': NO_SOURCE,
        ':common:bundleLibRuntimeToJarDebug': FROM_CACHE,
        ':common:compileDebugAidl': NO_SOURCE,
        ':common:compileDebugJavaWithJavac': FROM_CACHE,
        ':common:compileDebugLibraryResources': SUCCESS,
        ':common:compileDebugRenderscript': NO_SOURCE,
        ':common:compileDebugShaders': NO_SOURCE,
        ':common:compileDebugSources': UP_TO_DATE,
        ':common:copyDebugJniLibsProjectAndLocalJars': FROM_CACHE,
        ':common:copyDebugJniLibsProjectOnly': FROM_CACHE,
        ':common:extractDebugAnnotations': FROM_CACHE,
        ':common:extractDeepLinksDebug': FROM_CACHE,
        ':common:generateDebugAssets': UP_TO_DATE,
        ':common:generateDebugBuildConfig': FROM_CACHE,
        ':common:generateDebugRFile': FROM_CACHE,
        ':common:generateDebugResValues': FROM_CACHE,
        ':common:generateDebugResources': UP_TO_DATE,
        ':common:javaPreCompileDebug': FROM_CACHE,
        ':common:mergeDebugConsumerProguardFiles': FROM_CACHE,
        ':common:mergeDebugGeneratedProguardFiles': FROM_CACHE,
        ':common:mergeDebugJavaResource': FROM_CACHE,
        ':common:mergeDebugJniLibFolders': FROM_CACHE,
        ':common:mergeDebugNativeLibs': NO_SOURCE,
        ':common:mergeDebugShaders': FROM_CACHE,
        ':common:packageDebugAssets': FROM_CACHE,
        ':common:packageDebugRenderscript': NO_SOURCE,
        ':common:packageDebugResources': FROM_CACHE,
        ':common:parseDebugLocalResources': FROM_CACHE,
        ':common:preBuild': UP_TO_DATE,
        ':common:preDebugBuild': UP_TO_DATE,
        ':common:prepareLintJarForPublish': SUCCESS,
        ':common:processDebugJavaRes': NO_SOURCE,
        ':common:processDebugManifest': FROM_CACHE,
        ':common:stripDebugDebugSymbols': NO_SOURCE,
        ':common:syncDebugLibJars': FROM_CACHE,
        ':common:writeDebugAarMetadata': FROM_CACHE,
        ':dasherdancer:assembleDebug': SUCCESS,
        ':dasherdancer:bundleDebugAar': SUCCESS,
        ':dasherdancer:bundleLibCompileToJarDebug': FROM_CACHE,
        ':dasherdancer:bundleLibResDebug': NO_SOURCE,
        ':dasherdancer:bundleLibRuntimeToJarDebug': FROM_CACHE,
        ':dasherdancer:compileDebugAidl': NO_SOURCE,
        ':dasherdancer:compileDebugJavaWithJavac': FROM_CACHE,
        ':dasherdancer:compileDebugLibraryResources': SUCCESS,
        ':dasherdancer:compileDebugRenderscript': NO_SOURCE,
        ':dasherdancer:compileDebugShaders': NO_SOURCE,
        ':dasherdancer:compileDebugSources': UP_TO_DATE,
        ':dasherdancer:copyDebugJniLibsProjectAndLocalJars': FROM_CACHE,
        ':dasherdancer:copyDebugJniLibsProjectOnly': FROM_CACHE,
        ':dasherdancer:extractDebugAnnotations': FROM_CACHE,
        ':dasherdancer:extractDeepLinksDebug': FROM_CACHE,
        ':dasherdancer:generateDebugAssets': UP_TO_DATE,
        ':dasherdancer:generateDebugBuildConfig': FROM_CACHE,
        ':dasherdancer:generateDebugRFile': FROM_CACHE,
        ':dasherdancer:generateDebugResValues': FROM_CACHE,
        ':dasherdancer:generateDebugResources': UP_TO_DATE,
        ':dasherdancer:javaPreCompileDebug': FROM_CACHE,
        ':dasherdancer:mergeDebugConsumerProguardFiles': FROM_CACHE,
        ':dasherdancer:mergeDebugGeneratedProguardFiles': FROM_CACHE,
        ':dasherdancer:mergeDebugJavaResource': FROM_CACHE,
        ':dasherdancer:mergeDebugJniLibFolders': FROM_CACHE,
        ':dasherdancer:mergeDebugNativeLibs': NO_SOURCE,
        ':dasherdancer:mergeDebugShaders': FROM_CACHE,
        ':dasherdancer:packageDebugAssets': FROM_CACHE,
        ':dasherdancer:packageDebugRenderscript': NO_SOURCE,
        ':dasherdancer:packageDebugResources': FROM_CACHE,
        ':dasherdancer:parseDebugLocalResources': FROM_CACHE,
        ':dasherdancer:preBuild': UP_TO_DATE,
        ':dasherdancer:preDebugBuild': UP_TO_DATE,
        ':dasherdancer:prepareLintJarForPublish': SUCCESS,
        ':dasherdancer:processDebugJavaRes': NO_SOURCE,
        ':dasherdancer:processDebugManifest': FROM_CACHE,
        ':dasherdancer:stripDebugDebugSymbols': NO_SOURCE,
        ':dasherdancer:syncDebugLibJars': FROM_CACHE,
        ':dasherdancer:writeDebugAarMetadata': FROM_CACHE,
        ':doodles:assembleDebug': SUCCESS,
        ':doodles:bundleDebugAar': SUCCESS,
        ':doodles:bundleLibCompileToJarDebug': FROM_CACHE,
        ':doodles:bundleLibResDebug': NO_SOURCE,
        ':doodles:bundleLibRuntimeToJarDebug': FROM_CACHE,
        ':doodles:compileDebugAidl': NO_SOURCE,
        ':doodles:compileDebugJavaWithJavac': FROM_CACHE,
        ':doodles:compileDebugLibraryResources': SUCCESS,
        ':doodles:compileDebugRenderscript': NO_SOURCE,
        ':doodles:compileDebugShaders': NO_SOURCE,
        ':doodles:compileDebugSources': UP_TO_DATE,
        ':doodles:copyDebugJniLibsProjectAndLocalJars': FROM_CACHE,
        ':doodles:copyDebugJniLibsProjectOnly': FROM_CACHE,
        ':doodles:extractDebugAnnotations': FROM_CACHE,
        ':doodles:extractDeepLinksDebug': FROM_CACHE,
        ':doodles:generateDebugAssets': UP_TO_DATE,
        ':doodles:generateDebugBuildConfig': FROM_CACHE,
        ':doodles:generateDebugRFile': FROM_CACHE,
        ':doodles:generateDebugResValues': FROM_CACHE,
        ':doodles:generateDebugResources': UP_TO_DATE,
        ':doodles:javaPreCompileDebug': FROM_CACHE,
        ':doodles:mergeDebugConsumerProguardFiles': FROM_CACHE,
        ':doodles:mergeDebugGeneratedProguardFiles': FROM_CACHE,
        ':doodles:mergeDebugJavaResource': FROM_CACHE,
        ':doodles:mergeDebugJniLibFolders': FROM_CACHE,
        ':doodles:mergeDebugNativeLibs': NO_SOURCE,
        ':doodles:mergeDebugShaders': FROM_CACHE,
        ':doodles:packageDebugAssets': FROM_CACHE,
        ':doodles:packageDebugRenderscript': NO_SOURCE,
        ':doodles:packageDebugResources': FROM_CACHE,
        ':doodles:parseDebugLocalResources': FROM_CACHE,
        ':doodles:preBuild': UP_TO_DATE,
        ':doodles:preDebugBuild': UP_TO_DATE,
        ':doodles:prepareLintJarForPublish': SUCCESS,
        ':doodles:processDebugJavaRes': NO_SOURCE,
        ':doodles:processDebugManifest': FROM_CACHE,
        ':doodles:stripDebugDebugSymbols': NO_SOURCE,
        ':doodles:syncDebugLibJars': FROM_CACHE,
        ':doodles:writeDebugAarMetadata': FROM_CACHE,
        ':presentquest:assembleDebug': SUCCESS,
        ':presentquest:bundleDebugAar': SUCCESS,
        ':presentquest:bundleLibCompileToJarDebug': FROM_CACHE,
        ':presentquest:bundleLibResDebug': NO_SOURCE,
        ':presentquest:bundleLibRuntimeToJarDebug': FROM_CACHE,
        ':presentquest:compileDebugAidl': NO_SOURCE,
        ':presentquest:compileDebugJavaWithJavac': FROM_CACHE,
        ':presentquest:compileDebugLibraryResources': SUCCESS,
        ':presentquest:compileDebugRenderscript': NO_SOURCE,
        ':presentquest:compileDebugShaders': NO_SOURCE,
        ':presentquest:compileDebugSources': UP_TO_DATE,
        ':presentquest:copyDebugJniLibsProjectAndLocalJars': FROM_CACHE,
        ':presentquest:copyDebugJniLibsProjectOnly': FROM_CACHE,
        ':presentquest:extractDebugAnnotations': FROM_CACHE,
        ':presentquest:extractDeepLinksDebug': FROM_CACHE,
        ':presentquest:generateDebugAssets': UP_TO_DATE,
        ':presentquest:generateDebugBuildConfig': FROM_CACHE,
        ':presentquest:generateDebugRFile': FROM_CACHE,
        ':presentquest:generateDebugResValues': FROM_CACHE,
        ':presentquest:generateDebugResources': UP_TO_DATE,
        ':presentquest:javaPreCompileDebug': FROM_CACHE,
        ':presentquest:mergeDebugConsumerProguardFiles': FROM_CACHE,
        ':presentquest:mergeDebugGeneratedProguardFiles': FROM_CACHE,
        ':presentquest:mergeDebugJavaResource': FROM_CACHE,
        ':presentquest:mergeDebugJniLibFolders': FROM_CACHE,
        ':presentquest:mergeDebugNativeLibs': NO_SOURCE,
        ':presentquest:mergeDebugShaders': FROM_CACHE,
        ':presentquest:packageDebugAssets': FROM_CACHE,
        ':presentquest:packageDebugRenderscript': NO_SOURCE,
        ':presentquest:packageDebugResources': FROM_CACHE,
        ':presentquest:parseDebugLocalResources': FROM_CACHE,
        ':presentquest:preBuild': UP_TO_DATE,
        ':presentquest:preDebugBuild': UP_TO_DATE,
        ':presentquest:prepareLintJarForPublish': SUCCESS,
        ':presentquest:processDebugJavaRes': NO_SOURCE,
        ':presentquest:processDebugManifest': FROM_CACHE,
        ':presentquest:stripDebugDebugSymbols': NO_SOURCE,
        ':presentquest:syncDebugLibJars': FROM_CACHE,
        ':presentquest:writeDebugAarMetadata': FROM_CACHE,
        ':rocketsleigh:assembleDebug': SUCCESS,
        ':rocketsleigh:bundleDebugAar': SUCCESS,
        ':rocketsleigh:bundleLibCompileToJarDebug': FROM_CACHE,
        ':rocketsleigh:bundleLibResDebug': NO_SOURCE,
        ':rocketsleigh:bundleLibRuntimeToJarDebug': FROM_CACHE,
        ':rocketsleigh:compileDebugAidl': NO_SOURCE,
        ':rocketsleigh:compileDebugJavaWithJavac': FROM_CACHE,
        ':rocketsleigh:compileDebugLibraryResources': SUCCESS,
        ':rocketsleigh:compileDebugRenderscript': NO_SOURCE,
        ':rocketsleigh:compileDebugShaders': NO_SOURCE,
        ':rocketsleigh:compileDebugSources': UP_TO_DATE,
        ':rocketsleigh:copyDebugJniLibsProjectAndLocalJars': FROM_CACHE,
        ':rocketsleigh:copyDebugJniLibsProjectOnly': FROM_CACHE,
        ':rocketsleigh:extractDebugAnnotations': FROM_CACHE,
        ':rocketsleigh:extractDeepLinksDebug': FROM_CACHE,
        ':rocketsleigh:generateDebugAssets': UP_TO_DATE,
        ':rocketsleigh:generateDebugBuildConfig': FROM_CACHE,
        ':rocketsleigh:generateDebugRFile': FROM_CACHE,
        ':rocketsleigh:generateDebugResValues': FROM_CACHE,
        ':rocketsleigh:generateDebugResources': UP_TO_DATE,
        ':rocketsleigh:javaPreCompileDebug': FROM_CACHE,
        ':rocketsleigh:mergeDebugConsumerProguardFiles': FROM_CACHE,
        ':rocketsleigh:mergeDebugGeneratedProguardFiles': FROM_CACHE,
        ':rocketsleigh:mergeDebugJavaResource': FROM_CACHE,
        ':rocketsleigh:mergeDebugJniLibFolders': FROM_CACHE,
        ':rocketsleigh:mergeDebugNativeLibs': NO_SOURCE,
        ':rocketsleigh:mergeDebugShaders': FROM_CACHE,
        ':rocketsleigh:packageDebugAssets': FROM_CACHE,
        ':rocketsleigh:packageDebugRenderscript': NO_SOURCE,
        ':rocketsleigh:packageDebugResources': FROM_CACHE,
        ':rocketsleigh:parseDebugLocalResources': FROM_CACHE,
        ':rocketsleigh:preBuild': UP_TO_DATE,
        ':rocketsleigh:preDebugBuild': UP_TO_DATE,
        ':rocketsleigh:prepareLintJarForPublish': SUCCESS,
        ':rocketsleigh:processDebugJavaRes': NO_SOURCE,
        ':rocketsleigh:processDebugManifest': FROM_CACHE,
        ':rocketsleigh:stripDebugDebugSymbols': NO_SOURCE,
        ':rocketsleigh:syncDebugLibJars': FROM_CACHE,
        ':rocketsleigh:writeDebugAarMetadata': FROM_CACHE,
        ':santa-tracker:assembleDebug': SUCCESS,
        ':santa-tracker:assembleDevelopmentDebug': SUCCESS,
        ':santa-tracker:checkDevelopmentDebugAarMetadata': FROM_CACHE,
        ':santa-tracker:checkDevelopmentDebugDuplicateClasses': FROM_CACHE,
        ':santa-tracker:compileDevelopmentDebugAidl': NO_SOURCE,
        ':santa-tracker:compileDevelopmentDebugJavaWithJavac': FROM_CACHE,
        ':santa-tracker:compileDevelopmentDebugRenderscript': NO_SOURCE,
        ':santa-tracker:compileDevelopmentDebugShaders': NO_SOURCE,
        ':santa-tracker:compileDevelopmentDebugSources': UP_TO_DATE,
        ':santa-tracker:compressDevelopmentDebugAssets': FROM_CACHE,
        ':santa-tracker:createDevelopmentDebugCompatibleScreenManifests': FROM_CACHE,
        ':santa-tracker:dexBuilderDevelopmentDebug': FROM_CACHE,
        ':santa-tracker:extractDeepLinksDevelopmentDebug': FROM_CACHE,
        ':santa-tracker:generateDevelopmentDebugAssets': UP_TO_DATE,
        ':santa-tracker:generateDevelopmentDebugBuildConfig': FROM_CACHE,
        ':santa-tracker:generateDevelopmentDebugResValues': FROM_CACHE,
        ':santa-tracker:generateDevelopmentDebugResources': UP_TO_DATE,
        ':santa-tracker:javaPreCompileDevelopmentDebug': FROM_CACHE,
        ':santa-tracker:mergeDevelopmentDebugAssets': FROM_CACHE,
        ':santa-tracker:mergeDevelopmentDebugJavaResource': FROM_CACHE,
        ':santa-tracker:mergeDevelopmentDebugJniLibFolders': FROM_CACHE,
        ':santa-tracker:mergeDevelopmentDebugNativeDebugMetadata': NO_SOURCE,
        ':santa-tracker:mergeDevelopmentDebugNativeLibs': FROM_CACHE,
        ':santa-tracker:mergeDevelopmentDebugResources': SUCCESS,
        ':santa-tracker:mergeDevelopmentDebugShaders': FROM_CACHE,
        ':santa-tracker:mergeExtDexDevelopmentDebug': FROM_CACHE,
        ':santa-tracker:mergeLibDexDevelopmentDebug': FROM_CACHE,
        ':santa-tracker:mergeProjectDexDevelopmentDebug': FROM_CACHE,
        ':santa-tracker:packageDevelopmentDebug': SUCCESS,
        ':santa-tracker:preBuild': UP_TO_DATE,
        ':santa-tracker:preDevelopmentDebugBuild': UP_TO_DATE,
        ':santa-tracker:processDevelopmentDebugJavaRes': NO_SOURCE,
        ':santa-tracker:processDevelopmentDebugMainManifest': FROM_CACHE,
        ':santa-tracker:processDevelopmentDebugManifest': FROM_CACHE,
        ':santa-tracker:processDevelopmentDebugManifestForPackage': FROM_CACHE,
        ':santa-tracker:processDevelopmentDebugResources': SUCCESS,
        ':santa-tracker:stripDevelopmentDebugDebugSymbols': NO_SOURCE,
        ':santa-tracker:validateSigningDevelopmentDebug': FROM_CACHE,
        ':village:assembleDebug': SUCCESS,
        ':village:bundleDebugAar': SUCCESS,
        ':village:bundleLibCompileToJarDebug': FROM_CACHE,
        ':village:bundleLibResDebug': NO_SOURCE,
        ':village:bundleLibRuntimeToJarDebug': FROM_CACHE,
        ':village:compileDebugAidl': NO_SOURCE,
        ':village:compileDebugJavaWithJavac': FROM_CACHE,
        ':village:compileDebugLibraryResources': SUCCESS,
        ':village:compileDebugRenderscript': NO_SOURCE,
        ':village:compileDebugShaders': NO_SOURCE,
        ':village:compileDebugSources': UP_TO_DATE,
        ':village:copyDebugJniLibsProjectAndLocalJars': FROM_CACHE,
        ':village:copyDebugJniLibsProjectOnly': FROM_CACHE,
        ':village:extractDebugAnnotations': FROM_CACHE,
        ':village:extractDeepLinksDebug': FROM_CACHE,
        ':village:generateDebugAssets': UP_TO_DATE,
        ':village:generateDebugBuildConfig': FROM_CACHE,
        ':village:generateDebugRFile': FROM_CACHE,
        ':village:generateDebugResValues': FROM_CACHE,
        ':village:generateDebugResources': UP_TO_DATE,
        ':village:javaPreCompileDebug': FROM_CACHE,
        ':village:mergeDebugConsumerProguardFiles': FROM_CACHE,
        ':village:mergeDebugGeneratedProguardFiles': FROM_CACHE,
        ':village:mergeDebugJavaResource': FROM_CACHE,
        ':village:mergeDebugJniLibFolders': FROM_CACHE,
        ':village:mergeDebugNativeLibs': NO_SOURCE,
        ':village:mergeDebugShaders': FROM_CACHE,
        ':village:packageDebugAssets': FROM_CACHE,
        ':village:packageDebugRenderscript': NO_SOURCE,
        ':village:packageDebugResources': FROM_CACHE,
        ':village:parseDebugLocalResources': FROM_CACHE,
        ':village:preBuild': UP_TO_DATE,
        ':village:preDebugBuild': UP_TO_DATE,
        ':village:prepareLintJarForPublish': SUCCESS,
        ':village:processDebugJavaRes': NO_SOURCE,
        ':village:processDebugManifest': FROM_CACHE,
        ':village:stripDebugDebugSymbols': NO_SOURCE,
        ':village:syncDebugLibJars': FROM_CACHE,
        ':village:writeDebugAarMetadata': FROM_CACHE,
    ]

    private static final EXPECTED_RESULTS_4_0 = [
        ':common:assembleDebug': SUCCESS,
        ':common:bundleDebugAar': SUCCESS,
        ':common:bundleLibCompileToJarDebug': FROM_CACHE,
        ':common:bundleLibResDebug': NO_SOURCE,
        ':common:bundleLibRuntimeToJarDebug': FROM_CACHE,
        ':common:compileDebugAidl': NO_SOURCE,
        ':common:compileDebugJavaWithJavac': FROM_CACHE,
        ':common:compileDebugLibraryResources': SUCCESS,
        ':common:compileDebugRenderscript': NO_SOURCE,
        ':common:compileDebugShaders': NO_SOURCE,
        ':common:compileDebugSources': UP_TO_DATE,
        ':common:copyDebugJniLibsProjectAndLocalJars': FROM_CACHE,
        ':common:copyDebugJniLibsProjectOnly': FROM_CACHE,
        ':common:extractDebugAnnotations': FROM_CACHE,
        ':common:extractDeepLinksDebug': FROM_CACHE,
        ':common:generateDebugAssets': UP_TO_DATE,
        ':common:generateDebugBuildConfig': FROM_CACHE,
        ':common:generateDebugRFile': FROM_CACHE,
        ':common:generateDebugResValues': FROM_CACHE,
        ':common:generateDebugResources': UP_TO_DATE,
        ':common:javaPreCompileDebug': FROM_CACHE,
        ':common:mergeDebugConsumerProguardFiles': FROM_CACHE,
        ':common:mergeDebugGeneratedProguardFiles': FROM_CACHE,
        ':common:mergeDebugJavaResource': FROM_CACHE,
        ':common:mergeDebugJniLibFolders': FROM_CACHE,
        ':common:mergeDebugNativeLibs': FROM_CACHE,
        ':common:mergeDebugShaders': FROM_CACHE,
        ':common:packageDebugAssets': FROM_CACHE,
        ':common:packageDebugRenderscript': NO_SOURCE,
        ':common:packageDebugResources': FROM_CACHE,
        ':common:parseDebugLocalResources': FROM_CACHE,
        ':common:preBuild': UP_TO_DATE,
        ':common:preDebugBuild': UP_TO_DATE,
        ':common:prepareLintJarForPublish': SUCCESS,
        ':common:processDebugJavaRes': NO_SOURCE,
        ':common:processDebugManifest': FROM_CACHE,
        ':common:stripDebugDebugSymbols': NO_SOURCE,
        ':common:syncDebugLibJars': FROM_CACHE,
        ':dasherdancer:assembleDebug': SUCCESS,
        ':dasherdancer:bundleDebugAar': SUCCESS,
        ':dasherdancer:bundleLibCompileToJarDebug': FROM_CACHE,
        ':dasherdancer:bundleLibResDebug': NO_SOURCE,
        ':dasherdancer:bundleLibRuntimeToJarDebug': FROM_CACHE,
        ':dasherdancer:compileDebugAidl': NO_SOURCE,
        ':dasherdancer:compileDebugJavaWithJavac': FROM_CACHE,
        ':dasherdancer:compileDebugLibraryResources': SUCCESS,
        ':dasherdancer:compileDebugRenderscript': NO_SOURCE,
        ':dasherdancer:compileDebugShaders': NO_SOURCE,
        ':dasherdancer:compileDebugSources': UP_TO_DATE,
        ':dasherdancer:copyDebugJniLibsProjectAndLocalJars': FROM_CACHE,
        ':dasherdancer:copyDebugJniLibsProjectOnly': FROM_CACHE,
        ':dasherdancer:extractDebugAnnotations': FROM_CACHE,
        ':dasherdancer:extractDeepLinksDebug': FROM_CACHE,
        ':dasherdancer:generateDebugAssets': UP_TO_DATE,
        ':dasherdancer:generateDebugBuildConfig': FROM_CACHE,
        ':dasherdancer:generateDebugRFile': FROM_CACHE,
        ':dasherdancer:generateDebugResValues': FROM_CACHE,
        ':dasherdancer:generateDebugResources': UP_TO_DATE,
        ':dasherdancer:javaPreCompileDebug': FROM_CACHE,
        ':dasherdancer:mergeDebugConsumerProguardFiles': FROM_CACHE,
        ':dasherdancer:mergeDebugGeneratedProguardFiles': FROM_CACHE,
        ':dasherdancer:mergeDebugJavaResource': FROM_CACHE,
        ':dasherdancer:mergeDebugJniLibFolders': FROM_CACHE,
        ':dasherdancer:mergeDebugNativeLibs': FROM_CACHE,
        ':dasherdancer:mergeDebugShaders': FROM_CACHE,
        ':dasherdancer:packageDebugAssets': FROM_CACHE,
        ':dasherdancer:packageDebugRenderscript': NO_SOURCE,
        ':dasherdancer:packageDebugResources': FROM_CACHE,
        ':dasherdancer:parseDebugLocalResources': FROM_CACHE,
        ':dasherdancer:preBuild': UP_TO_DATE,
        ':dasherdancer:preDebugBuild': UP_TO_DATE,
        ':dasherdancer:prepareLintJarForPublish': SUCCESS,
        ':dasherdancer:processDebugJavaRes': NO_SOURCE,
        ':dasherdancer:processDebugManifest': FROM_CACHE,
        ':dasherdancer:stripDebugDebugSymbols': NO_SOURCE,
        ':dasherdancer:syncDebugLibJars': FROM_CACHE,
        ':doodles:assembleDebug': SUCCESS,
        ':doodles:bundleDebugAar': SUCCESS,
        ':doodles:bundleLibCompileToJarDebug': FROM_CACHE,
        ':doodles:bundleLibResDebug': NO_SOURCE,
        ':doodles:bundleLibRuntimeToJarDebug': FROM_CACHE,
        ':doodles:compileDebugAidl': NO_SOURCE,
        ':doodles:compileDebugJavaWithJavac': FROM_CACHE,
        ':doodles:compileDebugLibraryResources': SUCCESS,
        ':doodles:compileDebugRenderscript': NO_SOURCE,
        ':doodles:compileDebugShaders': NO_SOURCE,
        ':doodles:compileDebugSources': UP_TO_DATE,
        ':doodles:copyDebugJniLibsProjectAndLocalJars': FROM_CACHE,
        ':doodles:copyDebugJniLibsProjectOnly': FROM_CACHE,
        ':doodles:extractDebugAnnotations': FROM_CACHE,
        ':doodles:extractDeepLinksDebug': FROM_CACHE,
        ':doodles:generateDebugAssets': UP_TO_DATE,
        ':doodles:generateDebugBuildConfig': FROM_CACHE,
        ':doodles:generateDebugRFile': FROM_CACHE,
        ':doodles:generateDebugResValues': FROM_CACHE,
        ':doodles:generateDebugResources': UP_TO_DATE,
        ':doodles:javaPreCompileDebug': FROM_CACHE,
        ':doodles:mergeDebugConsumerProguardFiles': FROM_CACHE,
        ':doodles:mergeDebugGeneratedProguardFiles': FROM_CACHE,
        ':doodles:mergeDebugJavaResource': FROM_CACHE,
        ':doodles:mergeDebugJniLibFolders': FROM_CACHE,
        ':doodles:mergeDebugNativeLibs': FROM_CACHE,
        ':doodles:mergeDebugShaders': FROM_CACHE,
        ':doodles:packageDebugAssets': FROM_CACHE,
        ':doodles:packageDebugRenderscript': NO_SOURCE,
        ':doodles:packageDebugResources': FROM_CACHE,
        ':doodles:parseDebugLocalResources': FROM_CACHE,
        ':doodles:preBuild': UP_TO_DATE,
        ':doodles:preDebugBuild': UP_TO_DATE,
        ':doodles:prepareLintJarForPublish': SUCCESS,
        ':doodles:processDebugJavaRes': NO_SOURCE,
        ':doodles:processDebugManifest': FROM_CACHE,
        ':doodles:stripDebugDebugSymbols': NO_SOURCE,
        ':doodles:syncDebugLibJars': FROM_CACHE,
        ':presentquest:assembleDebug': SUCCESS,
        ':presentquest:bundleDebugAar': SUCCESS,
        ':presentquest:bundleLibCompileToJarDebug': FROM_CACHE,
        ':presentquest:bundleLibResDebug': NO_SOURCE,
        ':presentquest:bundleLibRuntimeToJarDebug': FROM_CACHE,
        ':presentquest:compileDebugAidl': NO_SOURCE,
        ':presentquest:compileDebugJavaWithJavac': FROM_CACHE,
        ':presentquest:compileDebugLibraryResources': SUCCESS,
        ':presentquest:compileDebugRenderscript': NO_SOURCE,
        ':presentquest:compileDebugShaders': NO_SOURCE,
        ':presentquest:compileDebugSources': UP_TO_DATE,
        ':presentquest:copyDebugJniLibsProjectAndLocalJars': FROM_CACHE,
        ':presentquest:copyDebugJniLibsProjectOnly': FROM_CACHE,
        ':presentquest:extractDebugAnnotations': FROM_CACHE,
        ':presentquest:extractDeepLinksDebug': FROM_CACHE,
        ':presentquest:generateDebugAssets': UP_TO_DATE,
        ':presentquest:generateDebugBuildConfig': FROM_CACHE,
        ':presentquest:generateDebugRFile': FROM_CACHE,
        ':presentquest:generateDebugResValues': FROM_CACHE,
        ':presentquest:generateDebugResources': UP_TO_DATE,
        ':presentquest:javaPreCompileDebug': FROM_CACHE,
        ':presentquest:mergeDebugConsumerProguardFiles': FROM_CACHE,
        ':presentquest:mergeDebugGeneratedProguardFiles': FROM_CACHE,
        ':presentquest:mergeDebugJavaResource': FROM_CACHE,
        ':presentquest:mergeDebugJniLibFolders': FROM_CACHE,
        ':presentquest:mergeDebugNativeLibs': FROM_CACHE,
        ':presentquest:mergeDebugShaders': FROM_CACHE,
        ':presentquest:packageDebugAssets': FROM_CACHE,
        ':presentquest:packageDebugRenderscript': NO_SOURCE,
        ':presentquest:packageDebugResources': FROM_CACHE,
        ':presentquest:parseDebugLocalResources': FROM_CACHE,
        ':presentquest:preBuild': UP_TO_DATE,
        ':presentquest:preDebugBuild': UP_TO_DATE,
        ':presentquest:prepareLintJarForPublish': SUCCESS,
        ':presentquest:processDebugJavaRes': NO_SOURCE,
        ':presentquest:processDebugManifest': FROM_CACHE,
        ':presentquest:stripDebugDebugSymbols': NO_SOURCE,
        ':presentquest:syncDebugLibJars': FROM_CACHE,
        ':rocketsleigh:assembleDebug': SUCCESS,
        ':rocketsleigh:bundleDebugAar': SUCCESS,
        ':rocketsleigh:bundleLibCompileToJarDebug': FROM_CACHE,
        ':rocketsleigh:bundleLibResDebug': NO_SOURCE,
        ':rocketsleigh:bundleLibRuntimeToJarDebug': FROM_CACHE,
        ':rocketsleigh:compileDebugAidl': NO_SOURCE,
        ':rocketsleigh:compileDebugJavaWithJavac': FROM_CACHE,
        ':rocketsleigh:compileDebugLibraryResources': SUCCESS,
        ':rocketsleigh:compileDebugRenderscript': NO_SOURCE,
        ':rocketsleigh:compileDebugShaders': NO_SOURCE,
        ':rocketsleigh:compileDebugSources': UP_TO_DATE,
        ':rocketsleigh:copyDebugJniLibsProjectAndLocalJars': FROM_CACHE,
        ':rocketsleigh:copyDebugJniLibsProjectOnly': FROM_CACHE,
        ':rocketsleigh:extractDebugAnnotations': FROM_CACHE,
        ':rocketsleigh:extractDeepLinksDebug': FROM_CACHE,
        ':rocketsleigh:generateDebugAssets': UP_TO_DATE,
        ':rocketsleigh:generateDebugBuildConfig': FROM_CACHE,
        ':rocketsleigh:generateDebugRFile': FROM_CACHE,
        ':rocketsleigh:generateDebugResValues': FROM_CACHE,
        ':rocketsleigh:generateDebugResources': UP_TO_DATE,
        ':rocketsleigh:javaPreCompileDebug': FROM_CACHE,
        ':rocketsleigh:mergeDebugConsumerProguardFiles': FROM_CACHE,
        ':rocketsleigh:mergeDebugGeneratedProguardFiles': FROM_CACHE,
        ':rocketsleigh:mergeDebugJavaResource': FROM_CACHE,
        ':rocketsleigh:mergeDebugJniLibFolders': FROM_CACHE,
        ':rocketsleigh:mergeDebugNativeLibs': FROM_CACHE,
        ':rocketsleigh:mergeDebugShaders': FROM_CACHE,
        ':rocketsleigh:packageDebugAssets': FROM_CACHE,
        ':rocketsleigh:packageDebugRenderscript': NO_SOURCE,
        ':rocketsleigh:packageDebugResources': FROM_CACHE,
        ':rocketsleigh:parseDebugLocalResources': FROM_CACHE,
        ':rocketsleigh:preBuild': UP_TO_DATE,
        ':rocketsleigh:preDebugBuild': UP_TO_DATE,
        ':rocketsleigh:prepareLintJarForPublish': SUCCESS,
        ':rocketsleigh:processDebugJavaRes': NO_SOURCE,
        ':rocketsleigh:processDebugManifest': FROM_CACHE,
        ':rocketsleigh:stripDebugDebugSymbols': NO_SOURCE,
        ':rocketsleigh:syncDebugLibJars': FROM_CACHE,
        ':santa-tracker:assembleDebug': SUCCESS,
        ':santa-tracker:assembleDevelopmentDebug': SUCCESS,
        ':santa-tracker:checkDevelopmentDebugDuplicateClasses': FROM_CACHE,
        ':santa-tracker:compileDevelopmentDebugAidl': NO_SOURCE,
        ':santa-tracker:compileDevelopmentDebugJavaWithJavac': FROM_CACHE,
        ':santa-tracker:compileDevelopmentDebugRenderscript': NO_SOURCE,
        ':santa-tracker:compileDevelopmentDebugShaders': NO_SOURCE,
        ':santa-tracker:compileDevelopmentDebugSources': UP_TO_DATE,
        ':santa-tracker:createDevelopmentDebugCompatibleScreenManifests': FROM_CACHE,
        ':santa-tracker:dexBuilderDevelopmentDebug': FROM_CACHE,
        ':santa-tracker:extractDeepLinksDevelopmentDebug': FROM_CACHE,
        ':santa-tracker:generateDevelopmentDebugAssets': UP_TO_DATE,
        ':santa-tracker:generateDevelopmentDebugBuildConfig': FROM_CACHE,
        ':santa-tracker:generateDevelopmentDebugResValues': FROM_CACHE,
        ':santa-tracker:generateDevelopmentDebugResources': UP_TO_DATE,
        ':santa-tracker:javaPreCompileDevelopmentDebug': FROM_CACHE,
        ':santa-tracker:mergeDevelopmentDebugAssets': FROM_CACHE,
        ':santa-tracker:mergeDevelopmentDebugJavaResource': FROM_CACHE,
        ':santa-tracker:mergeDevelopmentDebugJniLibFolders': FROM_CACHE,
        ':santa-tracker:mergeDevelopmentDebugNativeLibs': FROM_CACHE,
        ':santa-tracker:mergeDevelopmentDebugResources': SUCCESS,
        ':santa-tracker:mergeDevelopmentDebugShaders': FROM_CACHE,
        ':santa-tracker:mergeExtDexDevelopmentDebug': FROM_CACHE,
        ':santa-tracker:mergeLibDexDevelopmentDebug': FROM_CACHE,
        ':santa-tracker:mergeProjectDexDevelopmentDebug': FROM_CACHE,
        ':santa-tracker:packageDevelopmentDebug': SUCCESS,
        ':santa-tracker:preBuild': UP_TO_DATE,
        ':santa-tracker:preDevelopmentDebugBuild': UP_TO_DATE,
        ':santa-tracker:processDevelopmentDebugJavaRes': NO_SOURCE,
        ':santa-tracker:processDevelopmentDebugManifest': FROM_CACHE,
        ':santa-tracker:processDevelopmentDebugResources': SUCCESS,
        ':santa-tracker:stripDevelopmentDebugDebugSymbols': NO_SOURCE,
        ':santa-tracker:validateSigningDevelopmentDebug': FROM_CACHE,
        ':village:assembleDebug': SUCCESS,
        ':village:bundleDebugAar': SUCCESS,
        ':village:bundleLibCompileToJarDebug': FROM_CACHE,
        ':village:bundleLibResDebug': NO_SOURCE,
        ':village:bundleLibRuntimeToJarDebug': FROM_CACHE,
        ':village:compileDebugAidl': NO_SOURCE,
        ':village:compileDebugJavaWithJavac': FROM_CACHE,
        ':village:compileDebugLibraryResources': SUCCESS,
        ':village:compileDebugRenderscript': NO_SOURCE,
        ':village:compileDebugShaders': NO_SOURCE,
        ':village:compileDebugSources': UP_TO_DATE,
        ':village:copyDebugJniLibsProjectAndLocalJars': FROM_CACHE,
        ':village:copyDebugJniLibsProjectOnly': FROM_CACHE,
        ':village:extractDebugAnnotations': FROM_CACHE,
        ':village:extractDeepLinksDebug': FROM_CACHE,
        ':village:generateDebugAssets': UP_TO_DATE,
        ':village:generateDebugBuildConfig': FROM_CACHE,
        ':village:generateDebugRFile': FROM_CACHE,
        ':village:generateDebugResValues': FROM_CACHE,
        ':village:generateDebugResources': UP_TO_DATE,
        ':village:javaPreCompileDebug': FROM_CACHE,
        ':village:mergeDebugConsumerProguardFiles': FROM_CACHE,
        ':village:mergeDebugGeneratedProguardFiles': FROM_CACHE,
        ':village:mergeDebugJavaResource': FROM_CACHE,
        ':village:mergeDebugJniLibFolders': FROM_CACHE,
        ':village:mergeDebugNativeLibs': FROM_CACHE,
        ':village:mergeDebugShaders': FROM_CACHE,
        ':village:packageDebugAssets': FROM_CACHE,
        ':village:packageDebugRenderscript': NO_SOURCE,
        ':village:packageDebugResources': FROM_CACHE,
        ':village:parseDebugLocalResources': FROM_CACHE,
        ':village:preBuild': UP_TO_DATE,
        ':village:preDebugBuild': UP_TO_DATE,
        ':village:prepareLintJarForPublish': SUCCESS,
        ':village:processDebugJavaRes': NO_SOURCE,
        ':village:processDebugManifest': FROM_CACHE,
        ':village:stripDebugDebugSymbols': NO_SOURCE,
        ':village:syncDebugLibJars': FROM_CACHE,
    ]

    private static final EXPECTED_RESULTS_3_6 = [
        ':common:assembleDebug': SUCCESS,
        ':common:bundleDebugAar': SUCCESS,
        ':common:bundleLibCompileDebug': SUCCESS,
        ':common:bundleLibResDebug': SUCCESS,
        ':common:bundleLibRuntimeDebug': SUCCESS,
        ':common:compileDebugAidl': NO_SOURCE,
        ':common:compileDebugJavaWithJavac': FROM_CACHE,
        ':common:compileDebugLibraryResources': FROM_CACHE,
        ':common:compileDebugRenderscript': NO_SOURCE,
        ':common:compileDebugShaders': FROM_CACHE,
        ':common:compileDebugSources': UP_TO_DATE,
        ':common:copyDebugJniLibsProjectAndLocalJars': FROM_CACHE,
        ':common:copyDebugJniLibsProjectOnly': FROM_CACHE,
        ':common:extractDebugAnnotations': FROM_CACHE,
        ':common:extractDeepLinksDebug': FROM_CACHE,
        ':common:generateDebugAssets': UP_TO_DATE,
        ':common:generateDebugBuildConfig': FROM_CACHE,
        ':common:generateDebugRFile': FROM_CACHE,
        ':common:generateDebugResValues': FROM_CACHE,
        ':common:generateDebugResources': UP_TO_DATE,
        ':common:javaPreCompileDebug': FROM_CACHE,
        ':common:mergeDebugConsumerProguardFiles': SUCCESS,
        ':common:mergeDebugGeneratedProguardFiles': SUCCESS,
        ':common:mergeDebugJavaResource': FROM_CACHE,
        ':common:mergeDebugJniLibFolders': FROM_CACHE,
        ':common:mergeDebugNativeLibs': FROM_CACHE,
        ':common:mergeDebugShaders': FROM_CACHE,
        ':common:packageDebugAssets': FROM_CACHE,
        ':common:packageDebugRenderscript': NO_SOURCE,
        ':common:packageDebugResources': FROM_CACHE,
        ':common:parseDebugLocalResources': FROM_CACHE,
        ':common:preBuild': UP_TO_DATE,
        ':common:preDebugBuild': UP_TO_DATE,
        ':common:prepareLintJarForPublish': SUCCESS,
        ':common:processDebugJavaRes': NO_SOURCE,
        ':common:processDebugManifest': FROM_CACHE,
        ':common:stripDebugDebugSymbols': FROM_CACHE,
        ':common:syncDebugLibJars': FROM_CACHE,
        ':dasherdancer:assembleDebug': SUCCESS,
        ':dasherdancer:bundleDebugAar': SUCCESS,
        ':dasherdancer:bundleLibCompileDebug': SUCCESS,
        ':dasherdancer:bundleLibResDebug': SUCCESS,
        ':dasherdancer:bundleLibRuntimeDebug': SUCCESS,
        ':dasherdancer:compileDebugAidl': NO_SOURCE,
        ':dasherdancer:compileDebugJavaWithJavac': FROM_CACHE,
        ':dasherdancer:compileDebugLibraryResources': FROM_CACHE,
        ':dasherdancer:compileDebugRenderscript': NO_SOURCE,
        ':dasherdancer:compileDebugShaders': FROM_CACHE,
        ':dasherdancer:compileDebugSources': UP_TO_DATE,
        ':dasherdancer:copyDebugJniLibsProjectAndLocalJars': FROM_CACHE,
        ':dasherdancer:copyDebugJniLibsProjectOnly': FROM_CACHE,
        ':dasherdancer:extractDebugAnnotations': FROM_CACHE,
        ':dasherdancer:extractDeepLinksDebug': FROM_CACHE,
        ':dasherdancer:generateDebugAssets': UP_TO_DATE,
        ':dasherdancer:generateDebugBuildConfig': FROM_CACHE,
        ':dasherdancer:generateDebugRFile': FROM_CACHE,
        ':dasherdancer:generateDebugResValues': FROM_CACHE,
        ':dasherdancer:generateDebugResources': UP_TO_DATE,
        ':dasherdancer:javaPreCompileDebug': FROM_CACHE,
        ':dasherdancer:mergeDebugConsumerProguardFiles': SUCCESS,
        ':dasherdancer:mergeDebugGeneratedProguardFiles': SUCCESS,
        ':dasherdancer:mergeDebugJavaResource': FROM_CACHE,
        ':dasherdancer:mergeDebugJniLibFolders': FROM_CACHE,
        ':dasherdancer:mergeDebugNativeLibs': FROM_CACHE,
        ':dasherdancer:mergeDebugShaders': FROM_CACHE,
        ':dasherdancer:packageDebugAssets': FROM_CACHE,
        ':dasherdancer:packageDebugRenderscript': NO_SOURCE,
        ':dasherdancer:packageDebugResources': FROM_CACHE,
        ':dasherdancer:parseDebugLocalResources': FROM_CACHE,
        ':dasherdancer:preBuild': UP_TO_DATE,
        ':dasherdancer:preDebugBuild': UP_TO_DATE,
        ':dasherdancer:prepareLintJarForPublish': SUCCESS,
        ':dasherdancer:processDebugJavaRes': NO_SOURCE,
        ':dasherdancer:processDebugManifest': FROM_CACHE,
        ':dasherdancer:stripDebugDebugSymbols': FROM_CACHE,
        ':dasherdancer:syncDebugLibJars': FROM_CACHE,
        ':doodles:assembleDebug': SUCCESS,
        ':doodles:bundleDebugAar': SUCCESS,
        ':doodles:bundleLibCompileDebug': SUCCESS,
        ':doodles:bundleLibResDebug': SUCCESS,
        ':doodles:bundleLibRuntimeDebug': SUCCESS,
        ':doodles:compileDebugAidl': NO_SOURCE,
        ':doodles:compileDebugJavaWithJavac': FROM_CACHE,
        ':doodles:compileDebugLibraryResources': FROM_CACHE,
        ':doodles:compileDebugRenderscript': NO_SOURCE,
        ':doodles:compileDebugShaders': FROM_CACHE,
        ':doodles:compileDebugSources': UP_TO_DATE,
        ':doodles:copyDebugJniLibsProjectAndLocalJars': FROM_CACHE,
        ':doodles:copyDebugJniLibsProjectOnly': FROM_CACHE,
        ':doodles:extractDebugAnnotations': FROM_CACHE,
        ':doodles:extractDeepLinksDebug': FROM_CACHE,
        ':doodles:generateDebugAssets': UP_TO_DATE,
        ':doodles:generateDebugBuildConfig': FROM_CACHE,
        ':doodles:generateDebugRFile': FROM_CACHE,
        ':doodles:generateDebugResValues': FROM_CACHE,
        ':doodles:generateDebugResources': UP_TO_DATE,
        ':doodles:javaPreCompileDebug': FROM_CACHE,
        ':doodles:mergeDebugConsumerProguardFiles': SUCCESS,
        ':doodles:mergeDebugGeneratedProguardFiles': SUCCESS,
        ':doodles:mergeDebugJavaResource': FROM_CACHE,
        ':doodles:mergeDebugJniLibFolders': FROM_CACHE,
        ':doodles:mergeDebugNativeLibs': FROM_CACHE,
        ':doodles:mergeDebugShaders': FROM_CACHE,
        ':doodles:packageDebugAssets': FROM_CACHE,
        ':doodles:packageDebugRenderscript': NO_SOURCE,
        ':doodles:packageDebugResources': FROM_CACHE,
        ':doodles:parseDebugLocalResources': FROM_CACHE,
        ':doodles:preBuild': UP_TO_DATE,
        ':doodles:preDebugBuild': UP_TO_DATE,
        ':doodles:prepareLintJarForPublish': SUCCESS,
        ':doodles:processDebugJavaRes': NO_SOURCE,
        ':doodles:processDebugManifest': FROM_CACHE,
        ':doodles:stripDebugDebugSymbols': FROM_CACHE,
        ':doodles:syncDebugLibJars': FROM_CACHE,
        ':presentquest:assembleDebug': SUCCESS,
        ':presentquest:bundleDebugAar': SUCCESS,
        ':presentquest:bundleLibCompileDebug': SUCCESS,
        ':presentquest:bundleLibResDebug': SUCCESS,
        ':presentquest:bundleLibRuntimeDebug': SUCCESS,
        ':presentquest:compileDebugAidl': NO_SOURCE,
        ':presentquest:compileDebugJavaWithJavac': FROM_CACHE,
        ':presentquest:compileDebugLibraryResources': FROM_CACHE,
        ':presentquest:compileDebugRenderscript': NO_SOURCE,
        ':presentquest:compileDebugShaders': FROM_CACHE,
        ':presentquest:compileDebugSources': UP_TO_DATE,
        ':presentquest:copyDebugJniLibsProjectAndLocalJars': FROM_CACHE,
        ':presentquest:copyDebugJniLibsProjectOnly': FROM_CACHE,
        ':presentquest:extractDebugAnnotations': FROM_CACHE,
        ':presentquest:extractDeepLinksDebug': FROM_CACHE,
        ':presentquest:generateDebugAssets': UP_TO_DATE,
        ':presentquest:generateDebugBuildConfig': FROM_CACHE,
        ':presentquest:generateDebugRFile': FROM_CACHE,
        ':presentquest:generateDebugResValues': FROM_CACHE,
        ':presentquest:generateDebugResources': UP_TO_DATE,
        ':presentquest:javaPreCompileDebug': FROM_CACHE,
        ':presentquest:mergeDebugConsumerProguardFiles': SUCCESS,
        ':presentquest:mergeDebugGeneratedProguardFiles': SUCCESS,
        ':presentquest:mergeDebugJavaResource': FROM_CACHE,
        ':presentquest:mergeDebugJniLibFolders': FROM_CACHE,
        ':presentquest:mergeDebugNativeLibs': FROM_CACHE,
        ':presentquest:mergeDebugShaders': FROM_CACHE,
        ':presentquest:packageDebugAssets': FROM_CACHE,
        ':presentquest:packageDebugRenderscript': NO_SOURCE,
        ':presentquest:packageDebugResources': FROM_CACHE,
        ':presentquest:parseDebugLocalResources': FROM_CACHE,
        ':presentquest:preBuild': UP_TO_DATE,
        ':presentquest:preDebugBuild': UP_TO_DATE,
        ':presentquest:prepareLintJarForPublish': SUCCESS,
        ':presentquest:processDebugJavaRes': NO_SOURCE,
        ':presentquest:processDebugManifest': FROM_CACHE,
        ':presentquest:stripDebugDebugSymbols': FROM_CACHE,
        ':presentquest:syncDebugLibJars': FROM_CACHE,
        ':rocketsleigh:assembleDebug': SUCCESS,
        ':rocketsleigh:bundleDebugAar': SUCCESS,
        ':rocketsleigh:bundleLibCompileDebug': SUCCESS,
        ':rocketsleigh:bundleLibResDebug': SUCCESS,
        ':rocketsleigh:bundleLibRuntimeDebug': SUCCESS,
        ':rocketsleigh:compileDebugAidl': NO_SOURCE,
        ':rocketsleigh:compileDebugJavaWithJavac': FROM_CACHE,
        ':rocketsleigh:compileDebugLibraryResources': FROM_CACHE,
        ':rocketsleigh:compileDebugRenderscript': NO_SOURCE,
        ':rocketsleigh:compileDebugShaders': FROM_CACHE,
        ':rocketsleigh:compileDebugSources': UP_TO_DATE,
        ':rocketsleigh:copyDebugJniLibsProjectAndLocalJars': FROM_CACHE,
        ':rocketsleigh:copyDebugJniLibsProjectOnly': FROM_CACHE,
        ':rocketsleigh:extractDebugAnnotations': FROM_CACHE,
        ':rocketsleigh:extractDeepLinksDebug': FROM_CACHE,
        ':rocketsleigh:generateDebugAssets': UP_TO_DATE,
        ':rocketsleigh:generateDebugBuildConfig': FROM_CACHE,
        ':rocketsleigh:generateDebugRFile': FROM_CACHE,
        ':rocketsleigh:generateDebugResValues': FROM_CACHE,
        ':rocketsleigh:generateDebugResources': UP_TO_DATE,
        ':rocketsleigh:javaPreCompileDebug': FROM_CACHE,
        ':rocketsleigh:mergeDebugConsumerProguardFiles': SUCCESS,
        ':rocketsleigh:mergeDebugGeneratedProguardFiles': SUCCESS,
        ':rocketsleigh:mergeDebugJavaResource': FROM_CACHE,
        ':rocketsleigh:mergeDebugJniLibFolders': FROM_CACHE,
        ':rocketsleigh:mergeDebugNativeLibs': FROM_CACHE,
        ':rocketsleigh:mergeDebugShaders': FROM_CACHE,
        ':rocketsleigh:packageDebugAssets': FROM_CACHE,
        ':rocketsleigh:packageDebugRenderscript': NO_SOURCE,
        ':rocketsleigh:packageDebugResources': FROM_CACHE,
        ':rocketsleigh:parseDebugLocalResources': FROM_CACHE,
        ':rocketsleigh:preBuild': UP_TO_DATE,
        ':rocketsleigh:preDebugBuild': UP_TO_DATE,
        ':rocketsleigh:prepareLintJarForPublish': SUCCESS,
        ':rocketsleigh:processDebugJavaRes': NO_SOURCE,
        ':rocketsleigh:processDebugManifest': FROM_CACHE,
        ':rocketsleigh:stripDebugDebugSymbols': FROM_CACHE,
        ':rocketsleigh:syncDebugLibJars': FROM_CACHE,
        ':santa-tracker:assembleDebug': SUCCESS,
        ':santa-tracker:assembleDevelopmentDebug': SUCCESS,
        ':santa-tracker:checkDevelopmentDebugDuplicateClasses': FROM_CACHE,
        ':santa-tracker:compileDevelopmentDebugAidl': NO_SOURCE,
        ':santa-tracker:compileDevelopmentDebugJavaWithJavac': FROM_CACHE,
        ':santa-tracker:compileDevelopmentDebugRenderscript': NO_SOURCE,
        ':santa-tracker:compileDevelopmentDebugShaders': FROM_CACHE,
        ':santa-tracker:compileDevelopmentDebugSources': UP_TO_DATE,
        ':santa-tracker:createDevelopmentDebugCompatibleScreenManifests': FROM_CACHE,
        ':santa-tracker:dexBuilderDevelopmentDebug': FROM_CACHE,
        ':santa-tracker:extractDeepLinksDevelopmentDebug': FROM_CACHE,
        ':santa-tracker:generateDevelopmentDebugAssets': UP_TO_DATE,
        ':santa-tracker:generateDevelopmentDebugBuildConfig': FROM_CACHE,
        ':santa-tracker:generateDevelopmentDebugResValues': FROM_CACHE,
        ':santa-tracker:generateDevelopmentDebugResources': UP_TO_DATE,
        ':santa-tracker:javaPreCompileDevelopmentDebug': FROM_CACHE,
        ':santa-tracker:mainApkListPersistenceDevelopmentDebug': FROM_CACHE,
        ':santa-tracker:mergeDevelopmentDebugAssets': FROM_CACHE,
        ':santa-tracker:mergeDevelopmentDebugJavaResource': FROM_CACHE,
        ':santa-tracker:mergeDevelopmentDebugJniLibFolders': FROM_CACHE,
        ':santa-tracker:mergeDevelopmentDebugNativeLibs': FROM_CACHE,
        ':santa-tracker:mergeDevelopmentDebugResources': FROM_CACHE,
        ':santa-tracker:mergeDevelopmentDebugShaders': FROM_CACHE,
        ':santa-tracker:mergeExtDexDevelopmentDebug': FROM_CACHE,
        ':santa-tracker:mergeLibDexDevelopmentDebug': FROM_CACHE,
        ':santa-tracker:mergeProjectDexDevelopmentDebug': FROM_CACHE,
        ':santa-tracker:packageDevelopmentDebug': SUCCESS,
        ':santa-tracker:preBuild': UP_TO_DATE,
        ':santa-tracker:preDevelopmentDebugBuild': UP_TO_DATE,
        ':santa-tracker:processDevelopmentDebugJavaRes': NO_SOURCE,
        ':santa-tracker:processDevelopmentDebugManifest': FROM_CACHE,
        ':santa-tracker:processDevelopmentDebugResources': FROM_CACHE,
        ':santa-tracker:stripDevelopmentDebugDebugSymbols': FROM_CACHE,
        ':santa-tracker:validateSigningDevelopmentDebug': FROM_CACHE,
        ':village:assembleDebug': SUCCESS,
        ':village:bundleDebugAar': SUCCESS,
        ':village:bundleLibCompileDebug': SUCCESS,
        ':village:bundleLibResDebug': SUCCESS,
        ':village:bundleLibRuntimeDebug': SUCCESS,
        ':village:compileDebugAidl': NO_SOURCE,
        ':village:compileDebugJavaWithJavac': FROM_CACHE,
        ':village:compileDebugLibraryResources': FROM_CACHE,
        ':village:compileDebugRenderscript': NO_SOURCE,
        ':village:compileDebugShaders': FROM_CACHE,
        ':village:compileDebugSources': UP_TO_DATE,
        ':village:copyDebugJniLibsProjectAndLocalJars': FROM_CACHE,
        ':village:copyDebugJniLibsProjectOnly': FROM_CACHE,
        ':village:extractDebugAnnotations': FROM_CACHE,
        ':village:extractDeepLinksDebug': FROM_CACHE,
        ':village:generateDebugAssets': UP_TO_DATE,
        ':village:generateDebugBuildConfig': FROM_CACHE,
        ':village:generateDebugRFile': FROM_CACHE,
        ':village:generateDebugResValues': FROM_CACHE,
        ':village:generateDebugResources': UP_TO_DATE,
        ':village:javaPreCompileDebug': FROM_CACHE,
        ':village:mergeDebugConsumerProguardFiles': SUCCESS,
        ':village:mergeDebugGeneratedProguardFiles': SUCCESS,
        ':village:mergeDebugJavaResource': FROM_CACHE,
        ':village:mergeDebugJniLibFolders': FROM_CACHE,
        ':village:mergeDebugNativeLibs': FROM_CACHE,
        ':village:mergeDebugShaders': FROM_CACHE,
        ':village:packageDebugAssets': FROM_CACHE,
        ':village:packageDebugRenderscript': NO_SOURCE,
        ':village:packageDebugResources': FROM_CACHE,
        ':village:parseDebugLocalResources': FROM_CACHE,
        ':village:preBuild': UP_TO_DATE,
        ':village:preDebugBuild': UP_TO_DATE,
        ':village:prepareLintJarForPublish': SUCCESS,
        ':village:processDebugJavaRes': NO_SOURCE,
        ':village:processDebugManifest': FROM_CACHE,
        ':village:stripDebugDebugSymbols': FROM_CACHE,
        ':village:syncDebugLibJars': FROM_CACHE,
    ]
}
