/*
 * Copyright 2016 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.gradle.plugin.devel.impldeps

import org.gradle.integtests.fixtures.ToBeFixedForInstantExecution

class GradleImplDepsPublishingIntegrationTest extends BaseGradleImplDepsIntegrationTest {

    @ToBeFixedForInstantExecution
    def "module metadata generated by Maven plugin does not contain reference to Gradle modules for a published plugin"() {
        given:
        using m2
        buildFile << testablePluginProject()
        buildFile << """
            group = 'org.gradle.test'
            version = '1.0'
            apply plugin: 'maven'

            uploadArchives {
                repositories {
                    mavenDeployer {
                        repository(url: '$mavenRepo.uri')
                    }
                }
            }
        """
        settingsFile << "rootProject.name = 'sample'"

        file('src/main/groovy/MyPlugin.groovy') << customGroovyPlugin()

        when:
        executer.expectDeprecationWarnings(2)
        succeeds 'uploadArchives'

        then:
        def module = mavenRepo.module('org.gradle.test', 'sample', '1.0')
        def pom = module.parsedPom
        def compileScope = pom.scopes.test
        compileScope.dependencies.size() == 1
        compileScope.expectDependency('junit:junit:4.13')
    }

    @ToBeFixedForInstantExecution
    def "module metadata generated by Ivy plugin does not contain reference to Gradle modules for a published plugin"() {
        given:
        buildFile << testablePluginProject()
        buildFile << """
            group = 'org.gradle.test'
            version = '1.0'

            uploadArchives {
                repositories {
                    ivy {
                        url '${ivyRepo.uri}'
                    }
                }
            }
        """
        settingsFile << "rootProject.name = 'sample'"

        file('src/main/groovy/MyPlugin.groovy') << customGroovyPlugin()

        when:
        executer.expectDeprecationWarning()
        succeeds 'uploadArchives'

        then:
        def module = ivyRepo.module('org.gradle.test', 'sample', '1.0')
        def ivy = module.parsedIvy
        ivy.dependencies.size() == 1
        ivy.dependencies['junit:junit:4.13'].hasConf('testImplementation->default')
    }

    @ToBeFixedForInstantExecution
    def "module metadata generated by Maven publish plugin does not contain reference to Gradle modules for a published plugin"() {
        given:
        buildFile << testablePluginProject()
        buildFile << """
            apply plugin: 'maven-publish'

            publishing {
                publications {
                    mavenJava(MavenPublication) {
                        from components.java
                    }
                }
            }
        """

        file('src/main/groovy/MyPlugin.groovy') << customGroovyPlugin()

        when:
        succeeds 'generatePomFileForMavenJavaPublication'

        then:
        def xml = new XmlSlurper().parse(file('build/publications/mavenJava/pom-default.xml'))
        xml.dependencies.size() == 0
    }

    @ToBeFixedForInstantExecution
    def "module metadata generated by Ivy publish plugin does not contain reference to Gradle modules for a published plugin"() {
        given:
        buildFile << testablePluginProject()
        buildFile << """
            apply plugin: 'ivy-publish'

            publishing {
                publications {
                    ivyJava(IvyPublication) {
                        from components.java
                    }
                }
            }
        """

        file('src/main/groovy/MyPlugin.groovy') << customGroovyPlugin()

        when:
        succeeds 'generateDescriptorFileForIvyJavaPublication'

        then:
        def xml = new XmlSlurper().parse(file('build/publications/ivyJava/ivy.xml'))
        xml.dependencies.children().size() == 0
    }
}
