/*
 * Copyright 2012 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.gradle.api.internal.artifacts.ivyservice

import org.gradle.api.artifacts.component.ModuleComponentIdentifier
import org.gradle.api.internal.artifacts.DefaultModuleVersionSelector
import org.gradle.api.internal.artifacts.ivyservice.publisher.IvyXmlModuleDescriptorWriter
import org.gradle.internal.component.external.descriptor.Dependency
import org.gradle.internal.component.external.descriptor.MutableModuleDescriptorState
import org.gradle.internal.component.external.model.DefaultModuleComponentIdentifier
import org.gradle.internal.component.model.DefaultIvyArtifactName
import org.gradle.test.fixtures.file.TestNameTestDirectoryProvider
import org.junit.Rule
import spock.lang.Specification

import java.text.SimpleDateFormat

class IvyXmlModuleDescriptorWriterTest extends Specification {

    private @Rule TestNameTestDirectoryProvider temporaryFolder;
    ModuleComponentIdentifier id = DefaultModuleComponentIdentifier.newId("org.test", "projectA", "1.0")
    def ivyXmlModuleDescriptorWriter = new IvyXmlModuleDescriptorWriter()

    def "can create ivy (unmodified) descriptor"() {
        when:
        def descriptor = new MutableModuleDescriptorState(id)
        addConfiguration(descriptor, "archives")
        addConfiguration(descriptor, "compile")
        addConfiguration(descriptor, "runtime", ["compile"])
        addDependencyDescriptor(descriptor, "Dep1")
        addDependencyDescriptor(descriptor, "Dep2")
        descriptor.addArtifact(new DefaultIvyArtifactName("testartifact", "jar", "jar"), ["archives", "runtime"] as Set)
        File ivyFile = temporaryFolder.file("test/ivy/ivy.xml")
        ivyXmlModuleDescriptorWriter.write(descriptor, ivyFile);
        then:
        def ivyModule = new XmlSlurper().parse(ivyFile);
        assert ivyModule.@version == "2.0"
        assert ivyModule.info.@organisation == "org.test"
        assert ivyModule.info.@module == "projectA"
        assert ivyModule.info.@revision == "1.0"
        assert ivyModule.info.@status == "integration"
        assert ivyModule.configurations.conf.collect {it.@name } == ["archives", "compile", "runtime"]
        assert ivyModule.publications.artifact.collect {it.@name } == ["testartifact"]
        assert ivyModule.publications.artifact.collect {it.@conf } == ["archives,runtime"]
        assert ivyModule.dependencies.dependency.collect { "${it.@org}:${it.@name}:${it.@rev}" } == ["org.test:Dep1:1.0", "org.test:Dep2:1.0"]
    }

    def date(String timestamp) {
        def format = new SimpleDateFormat("yyyyMMddHHmmss")
        format.parse(timestamp)
    }

    def addDependencyDescriptor(MutableModuleDescriptorState state, String organisation = "org.test", String moduleName, String revision = "1.0") {
        Dependency dep = state.addDependency(DefaultModuleVersionSelector.newSelector(organisation, moduleName, revision))
        dep.addDependencyConfiguration("default", ["compile", "archives"])
    }

    def addConfiguration(MutableModuleDescriptorState state, String configurationName, List extended = []) {
        state.addConfiguration(configurationName, true, true, extended)
    }
}
